/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5RSmodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5FLprivate.h" 
#include "H5MMprivate.h" 
#include "H5RSprivate.h" 

#define H5RS_ALLOC_SIZE 256

struct H5RS_str_t {
    char    *s;       
    char    *end;     
    size_t   len;     
    size_t   max;     
    bool     wrapped; 
    unsigned n;       
};

static herr_t H5RS__xstrdup(H5RS_str_t *rs, const char *s);
static herr_t H5RS__prepare_for_append(H5RS_str_t *rs);
static herr_t H5RS__resize_for_append(H5RS_str_t *rs, size_t len);

bool H5_PKG_INIT_VAR = false;

H5FL_DEFINE_STATIC(H5RS_str_t);

H5FL_BLK_DEFINE_STATIC(str_buf);

static herr_t
H5RS__xstrdup(H5RS_str_t *rs, const char *s)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(rs);

    if (s) {
        size_t len = strlen(s);

        
        rs->max = H5RS_ALLOC_SIZE;
        while ((len + 1) > rs->max)
            rs->max *= 2;

        
        if (NULL == (rs->s = (char *)H5FL_BLK_MALLOC(str_buf, rs->max)))
            HGOTO_ERROR(H5E_RS, H5E_CANTALLOC, FAIL, "memory allocation failed");
        if (len)
            H5MM_memcpy(rs->s, s, len);
        rs->end  = rs->s + len;
        *rs->end = '\0';
        rs->len  = len;
    } 
    else {
        
        if (rs->s) {
            H5FL_BLK_FREE(str_buf, rs->s);
            rs->s = rs->end = NULL;
            rs->max = rs->len = 0;
        } 
        else {
            
            assert(NULL == rs->end);
            assert(0 == rs->max);
            assert(0 == rs->len);
        } 
    }     

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5RS__prepare_for_append(H5RS_str_t *rs)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(rs);

    if (NULL == rs->s) {
        rs->max = H5RS_ALLOC_SIZE;
        if (NULL == (rs->s = (char *)H5FL_BLK_MALLOC(str_buf, rs->max)))
            HGOTO_ERROR(H5E_RS, H5E_CANTALLOC, FAIL, "memory allocation failed");
        rs->end = rs->s;
        *rs->s  = '\0';
        rs->len = 0;
    } 
    else {
        
        if (rs->wrapped) {
            if (H5RS__xstrdup(rs, rs->s) < 0)
                HGOTO_ERROR(H5E_RS, H5E_CANTCOPY, FAIL, "can't copy string");
            rs->wrapped = false;
        } 
    }     

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5RS__resize_for_append(H5RS_str_t *rs, size_t len)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(rs);

    
    if (len >= (rs->max - rs->len)) {
        
        while (len >= (rs->max - rs->len))
            rs->max *= 2;
        if (NULL == (rs->s = (char *)H5FL_BLK_REALLOC(str_buf, rs->s, rs->max)))
            HGOTO_ERROR(H5E_RS, H5E_CANTALLOC, FAIL, "memory allocation failed");
        rs->end = rs->s + rs->len;
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

H5RS_str_t *
H5RS_create(const char *s)
{
    H5RS_str_t *ret_value = NULL; 

    FUNC_ENTER_NOAPI(NULL)

    
    if (NULL == (ret_value = H5FL_CALLOC(H5RS_str_t)))
        HGOTO_ERROR(H5E_RS, H5E_CANTALLOC, NULL, "memory allocation failed");

    
    if (s)
        if (H5RS__xstrdup(ret_value, s) < 0)
            HGOTO_ERROR(H5E_RS, H5E_CANTCOPY, NULL, "can't copy string");
    ret_value->n = 1;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

H5RS_str_t *
H5RS_wrap(const char *s)
{
    H5RS_str_t *ret_value = NULL; 

    FUNC_ENTER_NOAPI(NULL)

    
    if (NULL == (ret_value = H5FL_MALLOC(H5RS_str_t)))
        HGOTO_ERROR(H5E_RS, H5E_CANTALLOC, NULL, "memory allocation failed");

    
    H5_WARN_CAST_AWAY_CONST_OFF
    ret_value->s = (char *)s;
    H5_WARN_CAST_AWAY_CONST_ON

    ret_value->len = strlen(s);
    ret_value->end = ret_value->s + ret_value->len;

    ret_value->wrapped = true;
    ret_value->max     = 0; 
    ret_value->n       = 1;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5RS_asprintf_cat(H5RS_str_t *rs, const char *fmt, ...)
{
    va_list args1, args2;
    size_t  out_len;
    herr_t  ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(rs);
    assert(fmt);

    
    if (H5RS__prepare_for_append(rs) < 0)
        HGOTO_ERROR(H5E_RS, H5E_CANTINIT, FAIL, "can't initialize ref-counted string");

    
    va_start(args1, fmt);
    va_copy(args2, args1);
    while ((out_len = (size_t)vsnprintf(rs->end, (rs->max - rs->len), fmt, args1)) >= (rs->max - rs->len)) {
        
        if (H5RS__resize_for_append(rs, out_len) < 0)
            HGOTO_ERROR(H5E_RS, H5E_CANTRESIZE, FAIL, "can't resize ref-counted string buffer");

        
        va_end(args1);
        va_copy(args1, args2);
    } 

    
    rs->len += out_len;
    rs->end += out_len;

    
    va_end(args1);
    va_end(args2);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5RS_acat(H5RS_str_t *rs, const char *s)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(rs);
    assert(s);

    
    if (*s) {
        size_t len = strlen(s);

        
        if (H5RS__prepare_for_append(rs) < 0)
            HGOTO_ERROR(H5E_RS, H5E_CANTINIT, FAIL, "can't initialize ref-counted string");

        
        if ((rs->len + len) >= rs->max)
            if (H5RS__resize_for_append(rs, len) < 0)
                HGOTO_ERROR(H5E_RS, H5E_CANTRESIZE, FAIL, "can't resize ref-counted string buffer");

        
        H5MM_memcpy(rs->end, s, len);
        rs->end += len;
        *rs->end = '\0';
        rs->len += len;
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5RS_ancat(H5RS_str_t *rs, const char *s, size_t n)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(rs);
    assert(s);

    
    if (n && *s) {
        size_t len = strlen(s);

        
        n = MIN(len, n);

        
        if (H5RS__prepare_for_append(rs) < 0)
            HGOTO_ERROR(H5E_RS, H5E_CANTINIT, FAIL, "can't initialize ref-counted string");

        
        if ((rs->len + n) >= rs->max)
            if (H5RS__resize_for_append(rs, n) < 0)
                HGOTO_ERROR(H5E_RS, H5E_CANTRESIZE, FAIL, "can't resize ref-counted string buffer");

        
        H5MM_memcpy(rs->end, s, n);
        rs->end += n;
        *rs->end = '\0';
        rs->len += n;
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5RS_aputc(H5RS_str_t *rs, int c)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(rs);
    assert(c);

    
    if (H5RS__prepare_for_append(rs) < 0)
        HGOTO_ERROR(H5E_RS, H5E_CANTINIT, FAIL, "can't initialize ref-counted string");

    
    if ((rs->len + 1) >= rs->max)
        if (H5RS__resize_for_append(rs, 1) < 0)
            HGOTO_ERROR(H5E_RS, H5E_CANTRESIZE, FAIL, "can't resize ref-counted string buffer");

    
    *rs->end++ = (char)c;
    rs->len++;
    *rs->end = '\0';

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5RS_decr(H5RS_str_t *rs)
{
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(rs);
    assert(rs->n > 0);

    
    if ((--rs->n) == 0) {
        if (!rs->wrapped)
            rs->s = (char *)H5FL_BLK_FREE(str_buf, rs->s);
        rs = H5FL_FREE(H5RS_str_t, rs);
    } 

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5RS_incr(H5RS_str_t *rs)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(rs);
    assert(rs->n > 0);

    
    if (rs->wrapped) {
        if (H5RS__xstrdup(rs, rs->s) < 0)
            HGOTO_ERROR(H5E_RS, H5E_CANTCOPY, FAIL, "can't copy string");
        rs->wrapped = false;
    } 

    
    rs->n++;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

H5RS_str_t *
H5RS_dup(H5RS_str_t *ret_value)
{
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    if (ret_value != NULL)
        
        ret_value->n++;

    FUNC_LEAVE_NOAPI(ret_value)
} 

int
H5RS_cmp(const H5RS_str_t *rs1, const H5RS_str_t *rs2)
{
    
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(rs1);
    assert(rs1->s);
    assert(rs2);
    assert(rs2->s);

    FUNC_LEAVE_NOAPI(strcmp(rs1->s, rs2->s))
} 

size_t
H5RS_len(const H5RS_str_t *rs)
{
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(rs);
    assert(rs->s);

    FUNC_LEAVE_NOAPI(strlen(rs->s))
} 

char *
H5RS_get_str(const H5RS_str_t *rs)
{
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(rs);
    assert(rs->s);

    FUNC_LEAVE_NOAPI(rs->s)
} 

unsigned
H5RS_get_count(const H5RS_str_t *rs)
{
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(rs);
    assert(rs->n > 0);

    FUNC_LEAVE_NOAPI(rs->n)
} 
