/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#ifndef H5Fpublic_H
#define H5Fpublic_H

#include "H5public.h"   
#include "H5ACpublic.h" 
#include "H5Ipublic.h"  

#define H5F_ACC_RDONLY (0x0000u) 
#define H5F_ACC_RDWR   (0x0001u) 
#define H5F_ACC_TRUNC  (0x0002u) 
#define H5F_ACC_EXCL   (0x0004u) 

#define H5F_ACC_CREAT (0x0010u) 
#define H5F_ACC_SWMR_WRITE                                                                                   \
    (0x0020u) 
#define H5F_ACC_SWMR_READ                                                                                    \
    (0x0040u) 

#define H5F_ACC_DEFAULT (0xffffu)

#define H5F_OBJ_FILE     (0x0001u) 
#define H5F_OBJ_DATASET  (0x0002u) 
#define H5F_OBJ_GROUP    (0x0004u) 
#define H5F_OBJ_DATATYPE (0x0008u) 
#define H5F_OBJ_ATTR     (0x0010u) 

#define H5F_OBJ_ALL (H5F_OBJ_FILE | H5F_OBJ_DATASET | H5F_OBJ_GROUP | H5F_OBJ_DATATYPE | H5F_OBJ_ATTR)

#define H5F_OBJ_LOCAL (0x0020u)

#define H5F_FAMILY_DEFAULT 0 

#define H5F_PAGE_BUFFER_SIZE_DEFAULT SIZE_MAX

#ifdef H5_HAVE_PARALLEL

#define H5F_MPIO_DEBUG_KEY "H5F_mpio_debug_key"
#endif 

typedef enum H5F_scope_t {
    H5F_SCOPE_LOCAL  = 0, 
    H5F_SCOPE_GLOBAL = 1  
} H5F_scope_t;

#define H5F_UNLIMITED HSIZE_UNDEF

typedef enum H5F_close_degree_t {
    H5F_CLOSE_DEFAULT = 0, 
    H5F_CLOSE_WEAK    = 1, 
    H5F_CLOSE_SEMI    = 2, 
    H5F_CLOSE_STRONG  = 3  
} H5F_close_degree_t;

//! <!-- [H5F_info2_t_snip] -->
typedef struct H5F_info2_t {
    struct {
        unsigned version;        
        hsize_t  super_size;     
        hsize_t  super_ext_size; 
    } super;
    struct {
        unsigned version;   
        hsize_t  meta_size; 
        hsize_t  tot_space; 
    } free;
    struct {
        unsigned     version;   
        hsize_t      hdr_size;  
        H5_ih_info_t msgs_info; 
    } sohm;
} H5F_info2_t;
//! <!-- [H5F_info2_t_snip] -->

typedef enum H5F_mem_t {
    H5FD_MEM_NOLIST = -1, 
    H5FD_MEM_DEFAULT = 0, 
    H5FD_MEM_SUPER = 1,   
    H5FD_MEM_BTREE = 2,   
    H5FD_MEM_DRAW  = 3,   
    H5FD_MEM_GHEAP = 4,   
    H5FD_MEM_LHEAP = 5,   
    H5FD_MEM_OHDR  = 6,   

    H5FD_MEM_NTYPES 
} H5F_mem_t;

//! <!-- [H5F_sect_info_t_snip] -->
typedef struct H5F_sect_info_t {
    haddr_t addr; 
    hsize_t size; 
} H5F_sect_info_t;
//! <!-- [H5F_sect_info_t_snip] -->

typedef enum H5F_libver_t {
    H5F_LIBVER_ERROR    = -1,
    H5F_LIBVER_EARLIEST = 0, 
    H5F_LIBVER_V18      = 1, 
    H5F_LIBVER_V110     = 2, 
    H5F_LIBVER_V112     = 3, 
    H5F_LIBVER_V114     = 4, 
    H5F_LIBVER_V200     = 5, 
    H5F_LIBVER_LATEST   = 5, 
    H5F_LIBVER_NBOUNDS       
} H5F_libver_t;

//! <!-- [H5F_fspace_strategy_t_snip] -->
typedef enum H5F_fspace_strategy_t {
    H5F_FSPACE_STRATEGY_FSM_AGGR = 0, 
    H5F_FSPACE_STRATEGY_PAGE =
        1, 
    H5F_FSPACE_STRATEGY_AGGR = 2, 
    H5F_FSPACE_STRATEGY_NONE = 3, 
    H5F_FSPACE_STRATEGY_NTYPES    
} H5F_fspace_strategy_t;
//! <!-- [H5F_fspace_strategy_t_snip] -->

typedef enum H5F_file_space_type_t {
    H5F_FILE_SPACE_DEFAULT     = 0, 
    H5F_FILE_SPACE_ALL_PERSIST = 1, 
    H5F_FILE_SPACE_ALL         = 2, 
    H5F_FILE_SPACE_AGGR_VFD = 3,    
    H5F_FILE_SPACE_VFD      = 4,    
    H5F_FILE_SPACE_NTYPES           
} H5F_file_space_type_t;

//! <!-- [H5F_retry_info_t_snip] -->

#define H5F_NUM_METADATA_READ_RETRY_TYPES 21

typedef struct H5F_retry_info_t {
    unsigned  nbins;
    uint32_t *retries[H5F_NUM_METADATA_READ_RETRY_TYPES];
} H5F_retry_info_t;
//! <!-- [H5F_retry_info_t_snip] -->

typedef herr_t (*H5F_flush_cb_t)(hid_t object_id, void *udata);

#define H5F_RFIC_UNUSUAL_NUM_UNUSED_NUMERIC_BITS                                                             \
    (0x0001u) 
#define H5F_RFIC_ALL                                                                                         \
    (H5F_RFIC_UNUSUAL_NUM_UNUSED_NUMERIC_BITS) 

#ifdef __cplusplus
extern "C" {
#endif

H5_DLL htri_t H5Fis_accessible(const char *container_name, hid_t fapl_id);

H5_DLL hid_t H5Fcreate(const char *filename, unsigned flags, hid_t fcpl_id, hid_t fapl_id);

#ifndef H5_DOXYGEN
H5_DLL hid_t H5Fcreate_async(const char *app_file, const char *app_func, unsigned app_line,
                             const char *filename, unsigned flags, hid_t fcpl_id, hid_t fapl_id, hid_t es_id);
#else
H5_DLL hid_t H5Fcreate_async(const char *filename, unsigned flags, hid_t fcpl_id, hid_t fapl_id, hid_t es_id);
#endif

H5_DLL hid_t H5Fopen(const char *filename, unsigned flags, hid_t fapl_id);

#ifndef H5_DOXYGEN
H5_DLL hid_t H5Fopen_async(const char *app_file, const char *app_func, unsigned app_line,
                           const char *filename, unsigned flags, hid_t access_plist, hid_t es_id);
#else
H5_DLL hid_t H5Fopen_async(const char *filename, unsigned flags, hid_t access_plist, hid_t es_id);
#endif

H5_DLL hid_t H5Freopen(hid_t file_id);

#ifndef H5_DOXYGEN
H5_DLL hid_t H5Freopen_async(const char *app_file, const char *app_func, unsigned app_line, hid_t file_id,
                             hid_t es_id);
#else
H5_DLL hid_t H5Freopen_async(hid_t file_id, hid_t es_id);
#endif

H5_DLL herr_t H5Fflush(hid_t object_id, H5F_scope_t scope);

#ifndef H5_DOXYGEN
H5_DLL herr_t H5Fflush_async(const char *app_file, const char *app_func, unsigned app_line, hid_t object_id,
                             H5F_scope_t scope, hid_t es_id);
#else
H5_DLL herr_t H5Fflush_async(hid_t object_id, H5F_scope_t scope, hid_t es_id);
#endif

H5_DLL herr_t H5Fclose(hid_t file_id);

#ifndef H5_DOXYGEN
H5_DLL herr_t H5Fclose_async(const char *app_file, const char *app_func, unsigned app_line, hid_t file_id,
                             hid_t es_id);
#else
H5_DLL herr_t H5Fclose_async(hid_t file_id, hid_t es_id);
#endif

H5_DLL herr_t H5Fdelete(const char *filename, hid_t fapl_id);

H5_DLL hid_t H5Fget_create_plist(hid_t file_id);

H5_DLL hid_t H5Fget_access_plist(hid_t file_id);

H5_DLL herr_t H5Fget_intent(hid_t file_id, unsigned *intent);

H5_DLL herr_t H5Fget_fileno(hid_t file_id, unsigned long *fileno);

H5_DLL ssize_t H5Fget_obj_count(hid_t file_id, unsigned types);

H5_DLL ssize_t H5Fget_obj_ids(hid_t file_id, unsigned types, size_t max_objs, hid_t *obj_id_list);

H5_DLL herr_t H5Fget_vfd_handle(hid_t file_id, hid_t fapl, void **file_handle);

H5_DLL herr_t H5Fmount(hid_t loc_id, const char *name, hid_t child, hid_t plist);

H5_DLL herr_t H5Funmount(hid_t loc_id, const char *name);

H5_DLL hssize_t H5Fget_freespace(hid_t file_id);

H5_DLL herr_t H5Fget_filesize(hid_t file_id, hsize_t *size);

H5_DLL herr_t H5Fget_eoa(hid_t file_id, haddr_t *eoa);

H5_DLL herr_t H5Fincrement_filesize(hid_t file_id, hsize_t increment);

H5_DLL ssize_t H5Fget_file_image(hid_t file_id, void *buf_ptr, size_t buf_len);

H5_DLL herr_t H5Fget_mdc_config(hid_t file_id, H5AC_cache_config_t *config_ptr);

H5_DLL herr_t H5Fset_mdc_config(hid_t file_id, const H5AC_cache_config_t *config_ptr);

H5_DLL herr_t H5Fget_mdc_hit_rate(hid_t file_id, double *hit_rate_ptr);

H5_DLL herr_t H5Fget_mdc_size(hid_t file_id, size_t *max_size_ptr, size_t *min_clean_size_ptr,
                              size_t *cur_size_ptr, int *cur_num_entries_ptr);

H5_DLL herr_t H5Freset_mdc_hit_rate_stats(hid_t file_id);

H5_DLL ssize_t H5Fget_name(hid_t obj_id, char *name, size_t size);

H5_DLL herr_t H5Fget_info2(hid_t obj_id, H5F_info2_t *file_info);

H5_DLL herr_t H5Fget_metadata_read_retry_info(hid_t file_id, H5F_retry_info_t *info);

H5_DLL herr_t H5Fstart_swmr_write(hid_t file_id);

H5_DLL ssize_t H5Fget_free_sections(hid_t file_id, H5F_mem_t type, size_t nsects,
                                    H5F_sect_info_t *sect_info );

H5_DLL herr_t H5Fclear_elink_file_cache(hid_t file_id);

H5_DLL herr_t H5Fset_libver_bounds(hid_t file_id, H5F_libver_t low, H5F_libver_t high);

H5_DLL herr_t H5Fstart_mdc_logging(hid_t file_id);

H5_DLL herr_t H5Fstop_mdc_logging(hid_t file_id);

H5_DLL herr_t H5Fget_mdc_logging_status(hid_t file_id, bool *is_enabled, bool *is_currently_logging);

H5_DLL herr_t H5Freset_page_buffering_stats(hid_t file_id);

H5_DLL herr_t H5Fget_page_buffering_stats(hid_t file_id, unsigned accesses[2], unsigned hits[2],
                                          unsigned misses[2], unsigned evictions[2], unsigned bypasses[2]);

H5_DLL herr_t H5Fget_mdc_image_info(hid_t file_id, haddr_t *image_addr, hsize_t *image_size);

H5_DLL herr_t H5Fget_dset_no_attrs_hint(hid_t file_id, bool *minimize);

H5_DLL herr_t H5Fset_dset_no_attrs_hint(hid_t file_id, bool minimize);

#ifdef H5_HAVE_PARALLEL

H5_DLL herr_t H5Fset_mpi_atomicity(hid_t file_id, bool flag);

H5_DLL herr_t H5Fget_mpi_atomicity(hid_t file_id, bool *flag);
#endif 

/// \cond DEV

H5_DLL herr_t H5Fformat_convert(hid_t fid);
/// \endcond

/// \cond DEV

#ifndef H5F_MODULE
#define H5Fcreate_async(...) H5Fcreate_async(__FILE__, __func__, __LINE__, __VA_ARGS__)
#define H5Fopen_async(...)   H5Fopen_async(__FILE__, __func__, __LINE__, __VA_ARGS__)
#define H5Freopen_async(...) H5Freopen_async(__FILE__, __func__, __LINE__, __VA_ARGS__)
#define H5Fflush_async(...)  H5Fflush_async(__FILE__, __func__, __LINE__, __VA_ARGS__)
#define H5Fclose_async(...)  H5Fclose_async(__FILE__, __func__, __LINE__, __VA_ARGS__)

#define H5Fcreate_async_wrap H5_NO_EXPAND(H5Fcreate_async)
#define H5Fopen_async_wrap   H5_NO_EXPAND(H5Fopen_async)
#define H5Freopen_async_wrap H5_NO_EXPAND(H5Freopen_async)
#define H5Fflush_async_wrap  H5_NO_EXPAND(H5Fflush_async)
#define H5Fclose_async_wrap  H5_NO_EXPAND(H5Fclose_async)
#endif 
/// \endcond

#ifndef H5_NO_DEPRECATED_SYMBOLS

#define H5F_ACC_DEBUG (0x0000u) 

//! <!-- [H5F_info1_t_snip] -->
typedef struct H5F_info1_t {
    hsize_t super_ext_size; 
    struct {
        hsize_t      hdr_size;  
        H5_ih_info_t msgs_info; 
    } sohm;
} H5F_info1_t;
//! <!-- [H5F_info1_t_snip] -->

H5_DLL herr_t H5Fget_info1(hid_t obj_id, H5F_info1_t *file_info);

H5_DLL herr_t H5Fset_latest_format(hid_t file_id, bool latest_format);

H5_DLL htri_t H5Fis_hdf5(const char *file_name);

#endif 

#ifdef __cplusplus
}
#endif
#endif 
