% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/esp-get-tiles.R, R/esp-get-attributions.R
\name{esp_get_tiles}
\alias{esp_get_tiles}
\alias{esp_get_attributions}
\alias{esp_getTiles}
\title{Get static tiles from public administrations of Spain}
\source{
\url{https://dieghernan.github.io/leaflet-providersESP/} leaflet plugin,
\strong{v1.3.3}.
}
\usage{
esp_get_tiles(
  x,
  type = "IDErioja",
  zoom = NULL,
  zoommin = 0,
  crop = TRUE,
  res = 512,
  bbox_expand = 0.05,
  transparent = TRUE,
  mask = FALSE,
  update_cache = FALSE,
  cache_dir = NULL,
  verbose = FALSE,
  options = NULL
)

esp_get_attributions(type, options = NULL)
}
\arguments{
\item{x}{An \code{\link[sf:sf]{sf}} or \code{\link[sf:sfc]{sfc}} object.}

\item{type}{This argument can be either:
\itemize{
\item The name of one of the pre-defined providers (see
\link{esp_tiles_providers}).
\item A list with two named elements \code{id} and \code{q} with your own arguments. See
\code{\link[=esp_make_provider]{esp_make_provider()}} and examples.
}}

\item{zoom}{character string or number. Only valid for WMTS providers, zoom
level to be downloaded. If \code{NULL}, it is determined automatically. If set,
it overrides \code{zoommin}. If a single \code{sf} \code{POINT} and \code{zoom = NULL}, the
function sets a zoom level of 18. See \strong{Details}.}

\item{zoommin}{character string or number. Delta on default \code{zoom}.
The default value is designed to download fewer tiles than you probably
want. Use \code{1} or \code{2} to increase the resolution.}

\item{crop}{logical. If \code{TRUE}, the results will be cropped to the specified
\code{x} extent. If \code{x} is an \code{\link[sf:sf]{sf}} object with one \code{POINT},
\code{crop} is set to \code{FALSE}. See \code{\link[terra:crop]{terra::crop()}}.}

\item{res}{character string or number. Only valid for WMS providers.
Resolution (in pixels) of the final tile.}

\item{bbox_expand}{number. Expansion percentage of the bounding box of \code{x}.}

\item{transparent}{logical. Provides transparent background, if supported.}

\item{mask}{logical. \code{TRUE} if the result should be masked to \code{x}. See
\code{\link[terra:mask]{terra::mask()}}.}

\item{update_cache}{logical. Should the cached file be refreshed? Default
is \code{FALSE}. When set to \code{TRUE}, it will force a new download.}

\item{cache_dir}{character string. A path to a cache directory. See
\strong{Caching strategies} section in \code{\link[=esp_set_cache_dir]{esp_set_cache_dir()}}.}

\item{verbose}{logical. If \code{TRUE} displays informational messages.}

\item{options}{A named list containing additional options to pass to the
query.}
}
\value{
A \code{SpatRaster} with 3 (RGB) or 4 (RGBA) layers, depending on
the provider. See \code{\link[terra:rast]{terra::rast()}}.
}
\description{
Get static map tiles based on a spatial object. Maps can be fetched from
various open map servers.

This function is an implementation of the javascript plugin
\href{https://dieghernan.github.io/leaflet-providersESP/}{leaflet-providersESP}
\strong{v1.3.3}.

\code{esp_get_attributions} gets the attribution of a tile provider defined as
the \code{type} argument.
}
\details{
Zoom levels are described on the
\href{https://wiki.openstreetmap.org/wiki/Zoom_levels}{OpenStreetMap wiki}:\tabular{rl}{
   zoom \tab area to represent \cr
   0 \tab whole world \cr
   3 \tab large country \cr
   5 \tab state \cr
   8 \tab county \cr
   10 \tab metropolitan area \cr
   11 \tab city \cr
   13 \tab village or suburb \cr
   16 \tab streets \cr
   18 \tab some buildings, trees \cr
}


For a complete list of providers see \link{esp_tiles_providers}.

Most WMS/WMTS providers provide tiles on
\href{https://epsg.io/3857}{\code{"EPSG:3857"}}. In case that the tile looks deformed,
try projecting first \code{x}:

\code{x <- sf::st_transform(x, 3857)}
}
\examples{
\dontshow{if (esp_check_access()) withAutoprint(\{ # examplesIf}
\dontrun{

# This example downloads data to your local computer!

segovia <- esp_get_prov_siane("segovia", epsg = 3857)
tile <- esp_get_tiles(segovia, "IGNBase.Todo")

library(ggplot2)
library(tidyterra)

ggplot(segovia) +
  geom_spatraster_rgb(data = tile, maxcell = Inf) +
  geom_sf(fill = NA, linewidth = 1)

# Another provider

tile2 <- esp_get_tiles(segovia, type = "MDT")

ggplot(segovia) +
  geom_spatraster_rgb(data = tile2, maxcell = Inf) +
  geom_sf(fill = NA, linewidth = 1, color = "red")

# A custom WMS provided

custom_wms <- esp_make_provider(
  id = "an_id_for_caching",
  q = "https://idecyl.jcyl.es/geoserver/ge/wms?",
  service = "WMS",
  version = "1.3.0",
  format = "image/png",
  layers = "geolog_cyl_litologia"
)

custom_wms_tile <- esp_get_tiles(segovia, custom_wms)

autoplot(custom_wms_tile, maxcell = Inf) +
  geom_sf(data = segovia, fill = NA, color = "red", linewidth = 1)

# A custom WMTS provider

custom_wmts <- esp_make_provider(
  id = "cyl_wmts",
  q = "https://www.ign.es/wmts/pnoa-ma?",
  service = "WMTS",
  layer = "OI.OrthoimageCoverage"
)

custom_wmts_tile <- esp_get_tiles(segovia, custom_wmts)

autoplot(custom_wmts_tile, maxcell = Inf) +
  geom_sf(data = segovia, fill = NA, color = "white", linewidth = 1)

# Example from https://leaflet-extras.github.io/leaflet-providers/preview/
cartodb_dark <- list(
  id = "CartoDB_DarkMatter",
  q = "https://a.basemaps.cartocdn.com/dark_all/{z}/{x}/{y}{r}.png"
)
cartodb_dark_tile <- esp_get_tiles(segovia, cartodb_dark,
  zoommin = 1,
  update_cache = TRUE
)

autoplot(cartodb_dark_tile, maxcell = Inf) +
  geom_sf(data = segovia, fill = NA, color = "white", linewidth = 1)
}
\dontshow{\}) # examplesIf}
}
\seealso{
\code{\link[terra:rast]{terra::rast()}}, \link{esp_tiles_providers}, \code{\link[maptiles:get_tiles]{maptiles::get_tiles()}}

\code{\link[giscoR:gisco_attributions]{giscoR::gisco_attributions()}}

Other functions for creating maps with images:
\code{\link{addProviderEspTiles}()},
\code{\link{esp_make_provider}()}
}
\concept{images}
