#' @title One-Compartment IV Bolus Pharmacokinetic Model (Nonlinear)
#' @name one_compartment_iv_bolus_nl
#' @description
#' Fits plasma concentration-time data to a one-compartment intravenous (IV) bolus
#' pharmacokinetic model using nonlinear regression. The model assumes instantaneous
#' drug distribution throughout a single, well-mixed compartment and first-order
#' elimination kinetics.
#'
#' Model parameters are estimated by nonlinear least squares:
#'   - Elimination rate constant (k_el)
#'   - Initial plasma concentration (C0)
#'   - Apparent volume of distribution (Vd = Dose / C0)
#'
#' Secondary pharmacokinetic parameters are derived:
#'   - Elimination half-life (t1/2 = ln(2)/k_el)
#'   - Clearance (CL = k_el * Vd)
#'
#' The function supports optional grouping (e.g., subjects, conditions). Publication-quality
#' plots with fitted curves are generated, and annotations summarizing key PK parameters
#' appear in the upper-right corner when <= 2 groups.
#'
#' Model:
#' C(t) = C0 * exp(-k_el * t)
#'
#' @param data A data frame containing plasma concentration-time data.
#' @param time_col Character string specifying the column name for time.
#' @param conc_col Character string specifying the column name for plasma concentration.
#' @param dose Numeric value specifying the administered IV bolus dose.
#' @param group_col Optional character string specifying a grouping variable (e.g., subject, condition).
#' @param plot Logical; if TRUE, generates a concentration-time plot with fitted curves.
#' @param annotate Logical; if TRUE, annotates the plot with PK parameters (only if <= 2 groups).
#'
#' @import stats
#' @import ggplot2
#' @importFrom stats na.omit nls
#' @importFrom ggplot2 ggplot aes geom_point geom_line geom_text labs theme
#' theme_bw element_text element_blank
#'
#' @return A list containing:
#' \describe{
#'   \item{\code{fitted_parameters}}{Data frame with C0, k_el, t1/2, Vd, CL, RMSE, AIC, and BIC.}
#'   \item{\code{data}}{Processed data used for fitting and plotting.}
#' }
#' @examples
#' # Example I: Single subject one-compartment IV bolus data
#' df <- data.frame(
#'   time = c(0.08, 0.25, 0.5, 1, 2, 4, 6, 8, 12),
#'   concentration = c(18.2, 16.1, 13.5, 10.2, 6.8, 4.9, 3.6, 2.1, 1.2)
#' )
#' one_compartment_iv_bolus_nl(
#'   data = df,
#'   time_col = "time",
#'   conc_col = "concentration",
#'   dose = 100
#' )
#'
#' # Example II: Condition-dependent pharmacokinetics (e.g., pH or physiological state)
#' df_cond <- data.frame(
#'   time = rep(c(0.25, 0.5, 1, 2, 4, 6), 2),
#'   concentration = c(
#'     17.8, 15.6, 13.1, 9.8, 6.4, 4.8,   # Condition A
#'     14.9, 13.0, 10.9, 8.0, 5.2, 3.9    # Condition B
#'   ),
#'   condition = rep(c("Condition A", "Condition B"), each = 6)
#' )
#' one_compartment_iv_bolus_nl(
#'   data = df_cond,
#'   time_col = "time",
#'   conc_col = "concentration",
#'   dose = 100,
#'   group_col = "condition"
#' )
#'
#' # Example III: Multiple subjects (population-style one-compartment IV bolus pharmacokinetics)
#' df_subjects <- data.frame(
#'   time = rep(c(0.25, 0.5, 1, 2, 4, 6, 8), 6),
#'   concentration = c(
#'     18.6, 16.3, 13.9, 10.5, 7.0, 5.1, 3.8,   # Subject 1
#'     17.9, 15.7, 13.2, 9.9, 6.6, 4.9, 3.6,    # Subject 2
#'     17.1, 15.0, 12.6, 9.4, 6.3, 4.7, 3.4,    # Subject 3
#'     16.4, 14.4, 12.0, 9.0, 6.0, 4.4, 3.2,    # Subject 4
#'     15.8, 13.9, 11.6, 8.7, 5.8, 4.2, 3.1,    # Subject 5
#'     15.2, 13.3, 11.0, 8.3, 5.5, 4.0, 2.9     # Subject 6
#'   ),
#'   subject = rep(paste0("S", 1:6), each = 7)
#' )
#' one_compartment_iv_bolus_nl(
#'   data = df_subjects,
#'   time_col = "time",
#'   conc_col = "concentration",
#'   dose = 100,
#'   group_col = "subject"
#' )
#' @references Widmark, E. M. P. (1919) Studies in the concentration of indifferent
#' narcotics in blood and tissues. Acta Medica Scandinavica, 52(1), 87–164.
#' @references Gibaldi, M. & Perrier, D. (1982) <isbn:9780824710422> Pharmacokinetics,
#' 2nd Edition. Marcel Dekker, New York.
#' @references Gabrielsson, J. & Weiner, D. (2000) <isbn:9186274929> Pharmacokinetic/Pharmacodynamic
#' Data Analysis: Concepts and Applications, 3rd Edition, Revised and Expanded.
#' Swedish Pharmaceutical Press, Stockholm.
#' @author Paul Angelo C. Manlapaz
#' @export

utils::globalVariables(c("time", "conc", "C0", "kel", "t_half", "Vd", "CL", "RMSE",
                         "AIC", "BIC", "label", "x_pos", "y_pos", "hjust", "vjust",
                         "group"))

one_compartment_iv_bolus_nl <- function(data,
                                        time_col = "time",
                                        conc_col = "concentration",
                                        dose,
                                        group_col = NULL,
                                        plot = TRUE,
                                        annotate = TRUE) {

  if (!requireNamespace("ggplot2", quietly = TRUE))
    stop("Package 'ggplot2' is required.")

  # -------------------------
  # Prepare data
  # -------------------------
  df <- data[, c(time_col, conc_col, group_col), drop = FALSE]
  df <- stats::na.omit(df)
  colnames(df)[1:2] <- c("time", "conc")
  df <- df[df$time >= 0 & df$conc > 0, ]

  # -------------------------
  # Group handling
  # -------------------------
  if (!is.null(group_col)) {
    df$group <- as.factor(df[[group_col]])
  } else {
    df$group <- factor("All")
  }

  # -------------------------
  # Nonlinear fitting by group
  # -------------------------
  pk_fun <- function(time, C0, kel) C0 * exp(-kel * time)

  fit_results <- do.call(rbind, lapply(split(df, df$group), function(d) {

    # Starting values
    C0_start <- max(d$conc)
    kel_start <- 0.1

    fit <- stats::nls(
      conc ~ C0 * exp(-kel * time),
      data = d,
      start = list(C0 = C0_start, kel = kel_start),
      algorithm = "port",
      lower = c(C0 = 1e-4, kel = 1e-4),
      control = stats::nls.control(maxiter = 300)
    )

    coef_fit <- coef(fit)
    C0 <- coef_fit["C0"]
    kel <- coef_fit["kel"]

    t_half <- log(2)/kel
    Vd <- dose / C0
    CL <- kel * Vd

    pred <- pk_fun(d$time, C0, kel)
    rmse <- sqrt(mean((d$conc - pred)^2))

    n <- length(d$conc)
    p <- length(coef_fit)
    rss <- sum((d$conc - pred)^2)
    aic <- n * log(rss/n) + 2 * p
    bic <- n * log(rss/n) + log(n) * p

    data.frame(
      group = unique(d$group),
      C0 = C0,
      kel = kel,
      t_half = t_half,
      Vd = Vd,
      CL = CL,
      RMSE = rmse,
      AIC = aic,
      BIC = bic
    )
  }))

  # -------------------------
  # Plotting
  # -------------------------
  if (plot) {
    pred_df <- do.call(rbind, lapply(split(df, df$group), function(d) {
      pars <- fit_results[fit_results$group == unique(d$group), ]
      t_seq <- seq(min(d$time), max(d$time), length.out = 200)
      data.frame(
        time = t_seq,
        conc = pk_fun(t_seq, pars$C0, pars$kel),
        group = pars$group
      )
    }))

    p <- ggplot2::ggplot(df, ggplot2::aes(time, conc, color = group)) +
      ggplot2::geom_point(size = 3) +
      ggplot2::geom_line(data = pred_df, ggplot2::aes(time, conc, color = group), linewidth = 1) +
      ggplot2::labs(
        title = "One-Compartment IV Bolus Pharmacokinetic Model",
        subtitle = "First-order elimination (nonlinear fit)",
        x = "Time (hours)",
        y = "Plasma Concentration (mg/L)",
        color = "Group"
      ) +
      ggplot2::theme_bw(base_size = 14) +
      ggplot2::theme(
        plot.title = ggplot2::element_text(face = "bold", hjust = 0.5),
        plot.subtitle = ggplot2::element_text(hjust = 0.5),
        panel.grid.major = ggplot2::element_blank(),
        panel.grid.minor = ggplot2::element_blank()
      )

    # Upper-right annotation
    if (annotate && nrow(fit_results) <= 1) {
      ann <- fit_results
      ann$label <- paste0(
        "C0 = ", round(ann$C0, 2), "\n",
        "k_el = ", signif(ann$kel, 3), "\n",
        "t1/2 = ", round(ann$t_half, 2), "\n",
        "Vd = ", round(ann$Vd, 2), "\n",
        "CL = ", round(ann$CL, 2), "\n",
        "RMSE = ", round(ann$RMSE, 3), "\n",
        "AIC = ", round(ann$AIC, 1), "\n",
        "BIC = ", round(ann$BIC, 1)
      )
      ann$x_pos <- max(df$time) * 0.95
      ann$y_pos <- max(df$conc) * 0.95
      ann$hjust <- 1
      ann$vjust <- 1

      p <- p + ggplot2::geom_text(
        data = ann,
        ggplot2::aes(x = x_pos, y = y_pos, label = label, color = group),
        hjust = ann$hjust, vjust = ann$vjust,
        show.legend = FALSE
      )
    }

    print(p)
  }

  return(list(
    fitted_parameters = fit_results,
    data = df
  ))
}
