#' @title Two-Compartment IV Bolus Pharmacokinetic Model (Linear)
#' @name two_compartment_iv_bolus
#' @description
#' Fits plasma concentration-time data following an intravenous (IV) bolus dose
#' using a **log-linear approximation of a two-compartment pharmacokinetic model**.
#'
#' This function applies linear regression to the logarithm of plasma
#' concentrations versus time to estimate the **terminal elimination phase**
#' parameters. The approach provides an empirical approximation to a
#' biexponential decline but does **not explicitly decompose** the curve into
#' distribution and elimination exponentials.
#'
#' Under the two-compartment IV bolus model, concentration-time profiles are
#' classically described by:
#'
#' \deqn{
#'   C(t) = A e^{-\alpha t} + B e^{-\beta t}
#' }
#'
#' In this implementation, the terminal log-linear phase is approximated as:
#'
#' \deqn{
#'   \log(C) = \log(C_0) - k_{el} \, t
#' }
#'
#' where \eqn{C_0} is the extrapolated initial concentration and
#' \eqn{k_{el}} is the apparent elimination rate constant.
#'
#' From the fitted log-linear model, secondary pharmacokinetic parameters are
#' derived, including terminal half-life, apparent volume of distribution, and
#' clearance.
#'
#' @param data A data frame with plasma concentration-time data.
#' @param time_col Character string for the time column.
#' @param conc_col Character string for plasma concentration column.
#' @param dose Numeric value specifying IV bolus dose.
#' @param group_col Optional character string specifying grouping variable.
#' @param plot Logical; if TRUE, generates concentration-time plots.
#' @param annotate Logical; if TRUE, annotates plot (only if <=2 groups).
#'
#' @import stats
#' @import ggplot2
#' @importFrom stats na.omit lm
#' @importFrom ggplot2 ggplot aes geom_point geom_line geom_smooth geom_text
#' scale_y_log10 labs theme theme_bw element_text element_blank
#'
#' @return A list containing:
#' \describe{
#'   \item{\code{fitted_parameters}}{Data frame with C0, kel, t_half, Vd, CL, and R^2 for each group.}
#'   \item{\code{data}}{Processed data used for fitting and plotting.}
#' }
#' @examples
#' # Example I: Single subject two-compartment IV bolus data
#' df <- data.frame(
#'   time = c(0.08, 0.25, 0.5, 1, 2, 4, 6, 8, 12),
#'   concentration = c(40.0, 30.5, 25.0, 17.5, 10.2, 6.4, 4.1, 2.8, 1.5)
#' )
#' two_compartment_iv_bolus(
#'   data = df,
#'   time_col = "time",
#'   conc_col = "concentration",
#'   dose = 100
#' )
#'
#' # Example II: Condition-dependent pharmacokinetics (e.g., physiological state)
#' df_cond <- data.frame(
#'   time = rep(c(0.25, 0.5, 1, 2, 4, 6, 8), 2),
#'   concentration = c(
#'     25.3, 22.1, 18.5, 13.2, 8.5, 5.6, 3.8,   # Condition A
#'     20.7, 18.0, 14.9, 11.3, 7.1, 4.7, 3.2    # Condition B
#'   ),
#'   condition = rep(c("Condition A", "Condition B"), each = 7)
#' )
#' two_compartment_iv_bolus(
#'   data = df_cond,
#'   time_col = "time",
#'   conc_col = "concentration",
#'   dose = 100,
#'   group_col = "condition"
#' )
#'
#' # Example III: Multiple subjects (population-style two-compartment IV bolus pharmacokinetics)
#' df_subjects <- data.frame(
#'   time = rep(c(0.25, 0.5, 1, 2, 4, 6, 8), 5),
#'   concentration = c(
#'     26.1, 23.2, 19.6, 14.0, 9.0, 6.0, 4.0,   # Subject 1
#'     24.8, 21.8, 18.4, 13.3, 8.8, 5.8, 3.9,   # Subject 2
#'     25.5, 22.5, 19.0, 13.8, 8.7, 5.7, 3.7,   # Subject 3
#'     23.9, 20.9, 17.7, 12.8, 8.4, 5.5, 3.5,   # Subject 4
#'     24.4, 21.5, 18.0, 13.0, 8.5, 5.6, 3.6    # Subject 5
#'   ),
#'   subject = rep(paste0("S", 1:5), each = 7)
#' )
#' two_compartment_iv_bolus(
#'   data = df_subjects,
#'   time_col = "time",
#'   conc_col = "concentration",
#'   dose = 100,
#'   group_col = "subject"
#' )
#' @references Gibaldi, M. & Perrier, D. (1982) <isbn:9780824710422> Pharmacokinetics,
#' 2nd Edition. Marcel Dekker, New York.
#' @references Gabrielsson, J. & Weiner, D. (2000) <isbn:9186274929> Pharmacokinetic/Pharmacodynamic
#' Data Analysis: Concepts and Applications, 3rd Edition, Revised and Expanded. Swedish Pharmaceutical
#' Press, Stockholm.
#' @author Paul Angelo C. Manlapaz
#' @export

utils::globalVariables(c("time", "conc", "log_conc", "group", "C0", "kel", "t_half",
                         "Vd", "CL", "R2", "label", "x_pos", "y_pos", "hjust", "vjust",
                         "head", "tail"))

two_compartment_iv_bolus <- function(data,
                                            time_col = "time",
                                            conc_col = "concentration",
                                            dose,
                                            group_col = NULL,
                                            plot = TRUE,
                                            annotate = TRUE) {

  if (!requireNamespace("ggplot2", quietly = TRUE))
    stop("Package 'ggplot2' is required.")

  # -------------------------
  # Prepare data
  # -------------------------
  df <- data[, c(time_col, conc_col, group_col), drop = FALSE]
  df <- stats::na.omit(df)
  colnames(df)[1:2] <- c("time", "conc")

  df <- df[df$time >= 0 & df$conc > 0, ]
  df$log_conc <- log(df$conc)

  # -------------------------
  # Grouping
  # -------------------------
  if (!is.null(group_col)) {
    df$group <- as.factor(df[[group_col]])
  } else {
    df$group <- "Experimental"
  }

  # -------------------------
  # Linearized biexponential fitting
  # -------------------------
  fit_results <- do.call(rbind, lapply(split(df, df$group), function(d) {

    # Simple linear regression on log-transformed data
    # (This is a linearization approximation)
    fit <- stats::lm(log_conc ~ time, data = d)
    s <- summary(fit)

    kel <- -coef(fit)[2]
    logC0 <- coef(fit)[1]
    C0 <- exp(logC0)

    t_half <- log(2) / kel
    auc <- sum(diff(d$time) * (head(d$conc, -1) + tail(d$conc, -1)) / 2)
    CL <- dose / auc
    Vd <- CL / kel

    data.frame(
      group = unique(d$group),
      C0 = C0,
      kel = kel,
      t_half = t_half,
      Vd = Vd,
      CL = CL,
      R2 = s$r.squared
    )
  }))

  # -------------------------
  # Plot
  # -------------------------
  if (plot) {
    p <- ggplot2::ggplot(df, ggplot2::aes(x = time, y = conc, color = group)) +
      ggplot2::geom_point(size = 3) +
      ggplot2::geom_line(linewidth = 1) +
      ggplot2::geom_smooth(
        ggplot2::aes(y = exp(log_conc)),
        method = "lm",
        se = FALSE,
        color = "black",
        linewidth = 1
      ) +
      ggplot2::scale_y_log10() +
      ggplot2::labs(
        title = "Two-Compartment IV Bolus Pharmacokinetic Model",
        subtitle = "Log-linear approximation",
        x = "Time (hours)",
        y = "Plasma Concentration (log scale)",
        color = "Group"
      ) +
      ggplot2::theme_bw(base_size = 14) +
      ggplot2::theme(
        plot.title = ggplot2::element_text(face = "bold", hjust = 0.5),
        plot.subtitle = ggplot2::element_text(hjust = 0.5),
        panel.grid.major = ggplot2::element_blank(),
        panel.grid.minor = ggplot2::element_blank()
      )

    # -------------------------
    # Annotation: lower-left and upper-right
    # -------------------------
    if (annotate && nlevels(df$group) <= 2) {
      ann <- fit_results
      ann$label <- paste0(
        "C0 = ", round(ann$C0, 2), "\n",
        "k_el = ", signif(ann$kel, 3), "\n",
        "t1/2 = ", round(ann$t_half, 2), "\n",
        "Vd = ", round(ann$Vd, 2), "\n",
        "CL = ", round(ann$CL, 2), "\n",
        "R2 = ", round(ann$R2, 3)
      )

      x_min <- min(df$time)
      x_max <- max(df$time)
      y_min <- min(df$conc)
      y_max <- max(df$conc)

      # Assign positions: first group lower-left, second group upper-right
      ann$x_pos <- c(x_min + 0.05 * (x_max - x_min),
                     x_max - 0.05 * (x_max - x_min))[seq_len(nrow(ann))]

      ann$y_pos <- c(y_min * 0.5,
                     y_max * 0.95)[seq_len(nrow(ann))]

      ann$hjust <- c(0, 1)[seq_len(nrow(ann))]
      ann$vjust <- c(0, 1)[seq_len(nrow(ann))]

      p <- p + ggplot2::geom_text(
        data = ann,
        ggplot2::aes(x = x_pos, y = y_pos, label = label, color = group),
        hjust = ann$hjust,
        vjust = ann$vjust,
        size = 4,
        show.legend = FALSE
      )
    }

    print(p)
  }

  return(list(
    fitted_parameters = fit_results,
    data = df
  ))
}
