% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/michaelis_menten.R
\name{michaelis_menten}
\alias{michaelis_menten}
\title{Michaelis-Menten Kinetics (Linear Form) with Inhibition Comparison}
\arguments{
\item{data}{A data.frame containing concentration and rate data.}

\item{conc_col}{Column name for substrate or drug concentration.}

\item{rate_col}{Column name for reaction rate or elimination rate.}

\item{group_col}{Column indicating different conditions (e.g., inhibitor levels).}

\item{transform}{Linearization method: "Lineweaver-Burk" (default),
"Eadie-Hofstee", or "Hanes-Woolf".}

\item{inhibition_type}{Expected inhibition type:
"competitive", "noncompetitive", "uncompetitive", "multi-inhibition" or "none".}

\item{plot}{Logical; if TRUE, generates linearized comparison plot.}
}
\value{
A list containing:
\describe{
  \item{\code{fitted_parameters}}{A data frame summarizing linear regression
        results for each group or condition, including the estimated slope,
        intercept, coefficient of determination (\eqn{R^2}), and derived
        Michaelis-Menten parameters (\code{Km} and \code{Vmax}) computed according
        to the selected linear transformation.}
  \item{\code{transformed_data}}{A data frame containing the processed and
        linearized concentration and rate data (\code{x} and \code{y}) used for
        model fitting and visualization, along with the original group labels.}
  \item{\code{interpretation}}{A character string describing the expected
        inhibition pattern based on the specified \code{inhibition_type} and
        the comparison of slopes and intercepts across groups.}
}
}
\description{
Performs Michaelis-Menten kinetic analysis using linearized transformations
(Lineweaver-Burk by default) to compare multiple datasets and infer inhibition type.

The function fits linear regressions to transformed data and compares slopes
and intercepts across conditions.

Interpretation of inhibition patterns (Lineweaver-Burk):
\itemize{
  \item Competitive inhibition:
    Same y-intercept (1/Vmax), increased slope -> apparent Km increases
  \item Noncompetitive inhibition:
    Same x-intercept (-1/Km), increased y-intercept -> Vmax decreases
  \item Uncompetitive inhibition:
    Parallel lines -> both Km and Vmax decrease proportionally
}
}
\examples{
# Example I: Single Michaelis-Menten dataset (no inhibition)
df1 <- data.frame(
  concentration = c(0.5, 1, 2, 4, 6, 8, 10),
  rate = c(0.48, 0.85, 1.45, 2.20, 2.70, 3.05, 3.25),
  group = "No Inhibitor"
)
# Lineweaver-Burk
michaelis_menten(
  data = df1,
  conc_col = "concentration",
  rate_col = "rate",
  group_col = "group",
  transform = "Lineweaver-Burk",
  inhibition_type = "none",
  plot = TRUE
)
# Eadie-Hofstee
michaelis_menten(
  data = df1,
  conc_col = "concentration",
  rate_col = "rate",
  group_col = "group",
  transform = "Eadie-Hofstee",
  inhibition_type = "none",
  plot = TRUE
)
# Hanes-Woolf
michaelis_menten(
  data = df1,
  conc_col = "concentration",
  rate_col = "rate",
  group_col = "group",
  transform = "Hanes-Woolf",
  inhibition_type = "none",
  plot = TRUE
)

# Example II: Two datasets compared (inhibition analysis)
df2 <- data.frame(
  concentration = rep(c(0.5, 1, 2, 4, 6, 8, 10), 2),
  rate = c(
    # Reference (no inhibitor)
    0.50, 0.90, 1.50, 2.30, 2.80, 3.10, 3.30,
    # Condition B (possible inhibitor)
    0.35, 0.65, 1.10, 1.70, 2.10, 2.40, 2.55
  ),
  group = rep(c("No Inhibitor", "Inhibitor"), each = 7)
)
# Lineweaver-Burk
michaelis_menten(
  data = df2,
  conc_col = "concentration",
  rate_col = "rate",
  group_col = "group",
  transform = "Lineweaver-Burk",
  inhibition_type = "uncompetitive",
  plot = TRUE
)
# Eadie-Hofstee
michaelis_menten(
  data = df2,
  conc_col = "concentration",
  rate_col = "rate",
  group_col = "group",
  transform = "Eadie-Hofstee",
  inhibition_type = "competitive",
  plot = TRUE
)
# Hanes-Woolf
michaelis_menten(
  data = df2,
  conc_col = "concentration",
  rate_col = "rate",
  group_col = "group",
  transform = "Hanes-Woolf",
  inhibition_type = "competitive",
  plot = TRUE
)

# Example III: Six datasets compared (one reference, five test conditions)
df3 <- data.frame(
  concentration = rep(c(0.5, 1, 2, 4, 6, 8, 10), 6),
  rate = c(
    # Reference
    0.50, 0.90, 1.50, 2.30, 2.80, 3.10, 3.30,
    # Mixed Noncompetitive inhibitor A
    0.35, 0.65, 1.10, 1.70, 2.10, 2.40, 2.55,
    # Uncompetitive inhibitor
    0.40, 0.70, 1.15, 1.75, 2.15, 2.45, 2.60,
    # Mixed Noncompetitive inhibitor B
    0.30, 0.55, 0.95, 1.45, 1.85, 2.10, 2.20,
    # Mixed Noncompetitive + higher dose
    0.28, 0.50, 0.85, 1.30, 1.65, 1.85, 1.95,
    # Uncompetitive + higher dose
    0.38, 0.65, 1.10, 1.65, 2.05, 2.30, 2.40
  ),
  group = rep(c(
    "Reference",
    "Noncompetitive (Mixed) A",
    "Uncompetitive",
    "Noncompetitive (Mixed) B",
    "Noncompetitive (Mixed) High",
    "Uncompetitive High"
  ), each = 7)
)
# Lineweaver-Burk
michaelis_menten(
  data = df3,
  conc_col = "concentration",
  rate_col = "rate",
  group_col = "group",
  transform = "Lineweaver-Burk",
  inhibition_type = "multi-inhibition",
  plot = TRUE
)
# Eadie-Hofstee
michaelis_menten(
  data = df3,
  conc_col = "concentration",
  rate_col = "rate",
  group_col = "group",
  transform = "Eadie-Hofstee",
  inhibition_type = "multi-inhibition",
  plot = TRUE
)
# Hanes-Woolf
michaelis_menten(
  data = df3,
  conc_col = "concentration",
  rate_col = "rate",
  group_col = "group",
  transform = "Hanes-Woolf",
  inhibition_type = "multi-inhibition",
  plot = TRUE
)
}
\references{
Michaelis, L. and Menten, M. (1913) Die kinetik der invertinwirkung.
Biochemistry Zeitung, 79, 333-369.
}
\author{
Paul Angelo C. Manlapaz
}
