#' @title Main function to simulate expected yield using the Beverton-Holt Yield-per-Recruit model for a range of input parameters
#'
#' @description Estimate yield using the Beverton-Holt Yield-per-Recruit (YPR) model using a range of values for conditional fishing (\code{cf}) and natural (\code{cm}) mortality and a single fixed minimum length limit for harvest (\code{minLL}).
#'
#' @inheritParams yprBH_func
#' @param minLL The minimum length limit for harvest in mm
#' @param cfmin A single numeric for minimum conditional fishing mortality.
#' @param cfmax A single numeric for maximum conditional fishing mortality.
#' @param cfinc A single numeric for increment to cycle from minimum to maximum conditional fishing mortality.
#' @param cmmin A single numeric for minimum conditional natural mortality.
#' @param cmmax A single numeric for maximum conditional natural mortality.
#' @param cminc A single numeric for increment to cycle from minimum to maximum conditional natural mortality.
#' @param loi A numeric vector for lengths of interest. Used to determine number of fish that reach desired lengths.
#'
#' @details Details will be filled out later
#'
#' @return  A data.frame with the following calculated values:
#' \itemize{
#' \item \code{yield} is the estimated yield (in g).
#' \item \code{exploitation} is the exploitation rate.
#' \item \code{Nharvest} is the number of harvested fish.
#' \item \code{Ndie} is the number of fish that die of natural deaths.
#' \item \code{Nt} is the number of fish at time tr (time they become harvestable size).
#' \item \code{avgwt} is the average weight of fish harvested.
#' \item \code{avglen} is the average length of fish harvested.
#' \item \code{tr} is the time for a fish to recruit to a minimum length limit (i.e., time to enter fishery).
#' \item \code{F} is the instantaneous rate of fishing mortality.
#' \item \code{M} is the instantaneous rate of natural mortality.
#' \item \code{Z} is the instantaneous rate of total mortality.
#' \item \code{S} is the (total) annual rate of survival.
#' \item \code{N at xxx mm} is the number that reach the length of interest supplied. There will be one column for each length of interest.
#' }
#'
#' For convenience the data.frame also contains the model input values (\code{minLL}; \code{cf} derived from \code{cfmin}, \code{cfmax}, and \code{cfinc}; \code{cm} derived from \code{cmmin}, \code{cmmax}, and \code{cminc}; \code{N0}; \code{Linf}; \code{K}; \code{t0}; \code{LWalpha}; \code{LWbeta}; and \code{tmax}).
#'
#' The data.frame also contains a \code{notes} value which may contain abbreviations for "issues" that occurred when computing the results and were adjusted for. The possible abbreviates are defined under "values" in the documentation for \code{\link{yprBH_func}}.
#'
#' @seealso \code{\link{yprBH_func}} for estimating yield from single values of \code{cf}, \code{cm}, and \code{minLL}, and \code{\link{yprBH_minLL_var}} for simulating yield with multiple values of \code{cf}, \code{cm}, and \code{minLL}.
#'
#' @author Jason C. Doll, \email{jason.doll@fmarion.edu}
#'
#' @examples
#' # Life history parameters to be used below
#' LH <- makeLH(N0=100,tmax=15,Linf=592,K=0.20,t0=-0.3,LWalpha=-5.528,LWbeta=3.273)
#'
#' # Estimate yield for multiple values of minLL, cf, and cm
#' # # This is a minimal example, lengthinc, cfinc, cminc would likely be smaller
#' # #   to produce finer-scaled results
#' Res_1 <- yprBH_minLL_fixed(minLL=200,
#'                          cfmin=0.1,cfmax=0.9,cfinc=0.1,
#'                          cmmin=0.1,cmmax=0.9,cminc=0.1,
#'                          loi=c(200,250,300,350),lhparms=LH)
#'
#' # Load other required packages for organizing output and plotting
#' library(dplyr)    ## for filter
#' library(ggplot2)  ## for ggplot et al.
#' library(tidyr)    ## for pivot_longer
#'
#' # Custom theme for plots (to make look nice)
#' theme_FAMS <- function(...) {
#'   theme_bw() +
#'   theme(
#'     panel.grid.major=element_blank(),panel.grid.minor=element_blank(),
#'     axis.text=element_text(size=14,color="black"),
#'     axis.title=element_text(size=16,color="black"),
#'     axis.title.y=element_text(angle=90),
#'     axis.line=element_line(color="black"),
#'     panel.border=element_blank()
#'   )
#' }
#'
#' # Yield curve (yield vs exploitation)
#' # Extract results for cm=0.40
#' plot_dat <- Res_1 |> dplyr::filter(cm==0.40)
#'
#' ggplot(data=plot_dat,mapping=aes(x=u,y=yield)) +
#'   geom_point() +
#'   geom_line() +
#'   labs(y="Yield (g)",x="Exploitation (u)") +
#'   theme_FAMS()
#'
#' #Plot number of fish reaching 300 mm as a function of exploitation with cm = 0.40
#' ggplot(data=plot_dat,mapping=aes(x=u,y=`N at 300 mm`)) +
#'   geom_point() +
#'   geom_line() +
#'   labs(y="Number of fish at 300 mm",x="Exploitation (u)") +
#'   theme_FAMS()
#'# Plot number of fish reaching 300 mm as a function of exploitation with cm = 0.40
#' # Select columns for plotting and convert to long
#' plot_data_long <- plot_dat %>%
#'   select(u,`N at 200 mm`, `N at 250 mm`, `N at 300 mm`, `N at 350 mm`) %>%
#'   pivot_longer(!u, names_to="loi",values_to="number")
#'
#' # Generate plot
#' ggplot(data=plot_data_long,mapping=aes(x=u,y=number,group=loi,color=loi)) +
#'   geom_point() +
#'   scale_color_discrete(name="Yield",labels=c("N at 200 mm",
#'                        "N at 250 mm", "N at 300 mm", "N at 350 mm"))+
#'   geom_line() +
#'   labs(y="Number of fish",x="Exploitation (u)") +
#'   theme_FAMS() +
#'   theme(legend.position = "top")+
#'   guides(color=guide_legend(title="Length of interest"))
#'
#' @rdname yprBH_minLL_fixed
#' @export
yprBH_minLL_fixed<-function(minLL,cfmin,cfmax,cfinc,cmmin,cmmax,cminc,loi=NA,
                            lhparms,matchRicker=FALSE){

  # ---- Check inputs
  iCheckMLH(minLL)
  iCheckLLinf(minLL,lhparms$Linf)
  iCheckcf(cfmin,"minimum")
  iCheckcf(cfmax,"maximum")
  cf <- iCheckcfminc(cfinc,cfmin,cfmax)
  iCheckcm(cmmin,"minimum")
  iCheckcm(cmmax,"maximum")
  cm <- iCheckcfminc(cminc,cmmin,cmmax)

  # ---- Compute Yield et al. for varying minLL, cf, and cm
  # Setup data.frame of input values ... cf, and cm sequences were
  #   created in checks above
  res <- expand.grid(minLL=minLL,cf=cf,cm=cm)

  # Send each row to yprBH_func() ...
  #   i.e., calculate yield et al for all cf, and cm combos
  res <- purrr::pmap_df(res,yprBH_func,matchRicker=matchRicker,loi=loi,lhparms=lhparms)

  # ---- Return data.frame with both output values and input parameters
  res
}
