/*****************************************************************************
* (c) Copyright 2012-2013 F.Hoffmann-La Roche AG                             *
* Contact: bioinfoc@bioinfoc.ch, Detlef.Wolf@Roche.com.                      *
*                                                                            *
* This file is part of BIOINFO-C. BIOINFO-C is free software: you can        *
* redistribute it and/or modify it under the terms of the GNU Lesser         *
* General Public License as published by the Free Software Foundation,       *
* either version 3 of the License, or (at your option) any later version.    *
*                                                                            *
* BIOINFO-C is distributed in the hope that it will be useful, but           *
* WITHOUT ANY WARRANTY; without even the implied warranty of                 *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU          *
* Lesser General Public License for more details. You should have            *
* received a copy of the GNU Lesser General Public License along with        *
* BIOINFO-C. If not, see <http://www.gnu.org/licenses/>.                     *
*****************************************************************************/
/** @file html.c
    @brief Parsing HTML CGI POST data, various other CGI routines
    and generating HTML pages.
    Module prefices cgi_, html_
*/
#include <ctype.h>
#include <unistd.h>
#include <time.h>
#include "log.h"
#include "hlrmisc.h"
#include "format.h"
#include "linestream.h"
#include "html.h"

#include "R.h"

#define UNUSED(x) (void)(x)

static int hexdigit (register int c) {
  /*
    Convert a hex digit (character) to a numeric value.
    The character must be a valid hex digit.  The result is undefined
    if it is not.  Digits > 9 can be either upper or lower case.
  */
  c &= 0x7f;
  if (c >= 'a' && c <= 'f')
    return (c - 'a' + 10);
  if (c >= 'A' && c <= 'F')
    return (c - 'A' + 10);
  return (c - '0');
}

static char x2c (char *const x) {
  /*
    Convert a 2 digit hex string to a char.
    For example, x2c ("3f") ==> '?'.  No error checking is done.
  */
  return (hexdigit (x[0]) * 16) + hexdigit (x[1]);
}

/* ##### Section: CGI specific stuff:initializing/headers ##### */

static FILE *gHeaderDestination = NULL;

static void cgiLogImpl (char *format,va_list args) {
  /**
     The routine ensures that the HTTP protocol header is completed
     before writing any error message using functions die(), warn(),
     usage(), or romsg() from module log.<br>
     Usage: like printf()
  */
  UNUSED(format);
  UNUSED(args);
  cgiHeader (NULL);
}

void cgiInit (void) {
  /**
     Initialize the cgi module.
     Within a cgi program, this function should be called as early
     as possible.<br>
     cgiInit ensures that error messages generated by
     die() and warn() will output a cgi header, if this has not
     yet happened. This makes sure that die/warn in lower level
     routines (e.g. from this library) will not result in a
     "server error" message displayed in the user's browser.
  */
  char *server = getenv ("SERVER_SOFTWARE");
  if (server != NULL && strNCaseEqual (server,"apache",6)) {
    int fd;

    if (PLABLA_CLOSE (2) == -1)
      perror ("PROBLEM: cgiInit: close(2)");
    fd = dup (1);
    if (fd == -1) {
      REprintf ("PROBLEM: could not duplicate stdout\n");
      perror ("PROBLEM: cgiInit: dup");
    }
    if (fd != 2)
      REprintf ("PROBLEM: cgiInit(): could not redirect stderr (fd %d) - possibly cgiInit() several calls to cgiInit().\n",
              fd);
  }
  log_registerDie (&cgiLogImpl);
  log_registerWarn (&cgiLogImpl);
  log_registerUsage (&cgiLogImpl);
  log_registerRomsg (&cgiLogImpl);
}

int cgiIsCGI (void) {
  /**
     @return 1, if process is a CGI program called by the HTTP deamon
             0 else
  */
  return ((getenv ("HTTP_HOST") != NULL) ||
          (getenv ("HTTP_USER_AGENT") != NULL) ||
          (getenv ("HTTP_ACCEPT") != NULL));
}

static int gHeaderPrinted = 0;
static int gHeaderExpires = -1; // # of seconds the page is valid; -1 = eternity
static Stringa gHeaderRedirUrl = NULL; // "Location:" for HTTP redirection
static Stringa gHeaderCharSet = NULL; // character set specification
static Stringa gContentDisposition = NULL;

void cgiContentDispositionSet (char *filename) {
  /**
     Instruct the next cgiHeader() to include a content disposition 'attachment'
     line, which will prompt the user with a 'save file as' 'filename' dialog.
     @param[in] filename - the file name
  */
  stringCreateOnce (gContentDisposition,50);
  stringPrintf (gContentDisposition,
                "Content-disposition: attachment; filename=%s",filename);
}

void cgiExpiresSet (int seconds_valid) {
  /**
     Sets the lifetime of the next html page to be constructed.<br>
     Precondition: cgiHeader() has not yet been called.<br>
     Postcondition: cgiHeader()  will set an expires-flag.<br>
     Note: uses a standard HTTP 1.0 protocol feature that should work everywhere.<br>
     @param[in] seconds_valid - if a WWW browser is directed back to a page
                                visited before it has the choice to load the
                                page from its cache or retrieve it from the
                                HTTP server. 'seconds_valid' decides that
                                choice: if the more than 'seconds_valid'
                                seconds have passed, the browser is forced to
                                reload the page; else it can get if from its
                                cache. 0 for 'seconds_valid' is ok.
  */
  if (seconds_valid < -1)
    die ("cgiExpiresSet: %d",seconds_valid);
  if (gHeaderPrinted)
    die ("cgiExpiresSet: called after cgiHeader()");
  gHeaderExpires = seconds_valid;
}

void cgiRedirSet (char *url) {
  /**
     Instruct the next cgiHeader call to immediately redirect the
     browser to address 'url'. This is usefull for setting a cookie
     and redirecting at the same time.
     @param[in] url - the URL to re-direct to
  */
  stringCreateOnce (gHeaderRedirUrl,50);
  stringCpy (gHeaderRedirUrl,url);
}

void cgiEncodingSet (char *charset) {
  /**
     Sets the character set specification which is written into the cgi
     header.<br>
     Precondition: cgiHeader() has not yet been called.<br>
     Postcondition: cgiHeader()  will set the character set specification.<br>
     !param[in] charset - character set, e.g. "iso-8859-1", "utf-8", ...
                          If NULL, no character set specification is written
                          into the cgi header.
  */
  stringCreateOnce (gHeaderCharSet,15);
  if (charset != NULL && *charset != '\0')
    stringCpy (gHeaderCharSet,charset);
  else
    stringDestroy (gHeaderCharSet);
}

void cgiHeader (char *mimeType) {
  /**
     Print html header of type, but only if running under WWW;
     print only once during a process.<br>
     The behaviour of this function can be customized by cgiRedirSet(),
     cgiEncodingSet() and cgiExpiresSet()<br>
     Precondition: gHeaderDestination - is set (where to write the header)
     @param[in] mimeType - if NULL, text/html is assumed
                           if empty string, do not print header and prevent
                           die, warn etc. to print header (useful if header
                           has been printed outside of this process)
  */
  char exp[70];
  static Stringa charset = NULL;
  stringCreateOnce (charset,25);

  if (!cgiIsCGI ())
    return;
  if (mimeType != NULL && mimeType[0] == '\0')
    gHeaderPrinted = 1;
  if (gHeaderPrinted)
    return;
  gHeaderPrinted = 1;
  if (gHeaderExpires > -1) {
    char date[40];
    time_t t = time(0) + ((gHeaderExpires == 0) ? -36000 : gHeaderExpires);
    /* 0 means immediate refresh: since the clocks of the server running
       this cgi and the clock on the computer where the brower is sitting
       are usually off by a few minutes, we go back a while in time to force
       a reload */
    struct tm *gmt = gmtime (&t);
    strftime (date,39,"%a, %d %b %Y %H:%M:%S GMT",gmt);
    snprintf (exp,sizeof exp,"\r\nExpires: %.40s",date);
  }
  else
    exp[0] = '\0';
  if (gHeaderCharSet)
    stringPrintf (charset,"; charset=%s",string (gHeaderCharSet));
  else
    stringClear (charset);
  if(gHeaderDestination) {
    fprintf (gHeaderDestination,"Content-Type: %s%s%s\r\n",
             mimeType ? mimeType : "text/html",string (charset),exp);
  } else {
    REprintf("Content-Type: %s%s%s\r\n",
               mimeType ? mimeType : "text/html",string (charset),exp);
  }
  if (gContentDisposition != NULL && stringLen (gContentDisposition) > 0) {
    if(gHeaderDestination) {
      fprintf (gHeaderDestination,"%s\r\n",string (gContentDisposition));
    } else {
      REprintf("%s\r\n",string (gContentDisposition));
    }
    stringClear (gContentDisposition);
  }
  if (gHeaderRedirUrl) {
    if(gHeaderDestination) {
      fprintf (gHeaderDestination,"Location: %s\r\n",string (gHeaderRedirUrl));
    } else {
      REprintf("Location: %s\r\n",string (gHeaderRedirUrl));
    }
  }
  if(gHeaderDestination) {
    fputs ("\r\n",gHeaderDestination);
  } else {
    REprintf("\r\n");
  }
  fflush (gHeaderDestination); // to be sure this comes before any error msg
}

int cgiHeaderIsPrinted (void) {
  /**
     @return whether the CGI header was printed
  */
  return gHeaderPrinted;
}

void cgiHeaderCookie (char *mimeType,char *cookieSpec) {
  /**
     Like cgiHeader, but include a cookie for transmission to the client.<br>
     Precondition: cgiHeader()  has not yet been called;
                                if it was called before, cgiHeaderCookie()
                                has no effect
  */
  Stringa s = stringCreate (40);
  stringCpy (s,(mimeType ? mimeType : "text/html"));
  stringCat (s,"\r\nSet-Cookie: ");
  stringCat (s,cookieSpec);
  cgiHeader (string (s)); // treat like an extended mime type
  stringDestroy (s);
}

/* ##### Section: handling cookies ##### */
/*
   Cookie routines:<br>
   Usage:<br>
   In the process sending the page with the cookie in its header, use<br>
   char *c = cgiConstructCookie ("myparam","somevalue",0);<br>
   cgiHeaderCookie (0,c);<br>
   In the process receiving input use<br>
   char *v = cgiReadCookie ("myparam")<br>

   NAME=VALUE<br>
   This string is a sequence of characters excluding semi-colon, comma
   and white space. If there is a need to place such data in the name
   or value, some encoding method such as URL style %XX encoding is
   recommended, though no encoding is defined or required.<br>
   However, experience has shown that quoted strings with
   embedded blanks work (see rfc 2109)!<br>
   Implementation notes:<br>
   Support syntax above and if VALUE is quote delimited, unquote it.<br>
   Sample value of HTTP_COOKIE:<br>
   xxx_cookie_name="apple tree"; zwei=zwei; zwei="drei"<br>
   then<br>
   name            -> value<br>
   ---------------    ----------<br>
   xxx_cookie_name -> apple tree<br>
   zwei            -> drei<br>
   drei            -> null<br>
   <br>
   Algorithm should be:<br>
   look for 'name=', <br>
   if value starts with '"'<br>
   then return until terminating '"'<br>
   else return until eos
*/

static char *g_domain = NULL;

void cgiDomainSet (char *domain) {
  /**
     Set the domain to be used in the next cookie to be constructed<br>
     Postcondition: cgiConstructCookie() will add the domain spec
     @param[in] domain - domainname, or NULL to quit setting a domain
                         e.g. ".roche.com"
  */
  strReplace (&g_domain,domain);
}

char *cgiConstructCookie (char *name,char *value,int lifelength) {
  /**
     Construct a string for a transient or permanent cookie.
     The returned string can be used in cgiHeaderCookie().
     @param[in] name - name of the cookie for use in cgiReadCookie
     @param[in] value - value to be stored
     @param[in] lifelength - 0: transient cookie (bound to browser process);
                             >0 : go to disk and stay there for 'lifelength' seconds;
                             <0 : delete this cookie
     @return pointer to constructed text; this memory may be modified
             but not free'd nor realloc'ed by the caller of this routine.
             Stable until next call to this routine
  */
  static Stringa s = NULL;
  static Stringa e = NULL;
  stringCreateOnce (s,40);
  stringCreateOnce (e,40);

  cgiEncodeWord (value,e);
  stringCpy (s,name);
  stringCat (s,"=");
  stringCat (s,string (e));
  if (lifelength != 0) {
    // date format according to http://www.faqs.org/rfcs/rfc822.html, section 5.1
    char niceDate[40];
    time_t t;
    struct tm *tmp;
    t = time(NULL) + lifelength;
    tmp = localtime (&t);
    strftime (niceDate, sizeof (niceDate),"%a, %d-%b-%y 00:00:00 GMT",tmp);
    stringCat (s,"; expires=");
    stringCat (s,niceDate);
  }
  stringCat (s,"; path=/;");
  if (g_domain != NULL && *g_domain != '\0') {
    stringCat (s," domain=");
    stringCat (s,g_domain);
    stringCat (s,";");
  }
  return string (s);
}

char *cgiReadCookie (char *name) {
  /**
     If the current process is a CGI process and received a cookie
     from the client, the cookie is examined for a parameter called
     'name', whose value is extracted and returned.
     @param[in] name - name of the cookie whose value is the be returned
     @return pointer to value of cookie named 'name' if available; else 0<br>
             This memory may be modified but not free'd nor realloc'ed by the
             caller of this routine it is kept stable until the next call to
             cgiReadCookie()
  */
  char *cp;
  char *cp1;
  static Stringa cookie = NULL;
  static Stringa value  = NULL;

  if (getenv ("HTTP_COOKIE") == NULL)
    return 0; // no cookies to examine
  if (cookie == NULL)
    cookie = stringCreate (100);
  stringCpy (cookie,getenv ("HTTP_COOKIE")); // don't alter original:  allows multiple entry
  stringCreateOnce (value,40);
  stringCpy (value,name);
  stringCat (value,"=");

  cp = NULL;
  cp1 = string (cookie);
  while ((cp1 = strstr (cp1,string (value))) != NULL) { // find start of name=
    cp1 = cp1+strlen (name)+1; // skip name=
    cp  = cp1; // save start
  }
  if (cp == NULL)
    return 0;
  if (*cp == '"') {
    stringCpy (value,cp+1); // this value is quoted
    if ((cp = strchr (string (value),'"'))) // terminating '"' found
      *cp = '\0'; // remove it
    else
      die ("Cannot parse cookie ]]%s[[. Trailing \" missing.",
           string (cookie));
  }
  else {
    stringCpy (value,cp); // this value is not quoted
    if ((cp = strchr (string (value),';'))) // terminating ';' found
      *cp = '\0'; // remove it
  }
  stringAdjust (value);
  cgiDecodeWord (value);
  return string (value);
}

/* ##### Section: getting fields from form/post ##### */

static char *gCgiBuffer = NULL;
static int gCgiBufferSize;

char *cgiGet2Post (void) {
  /**
     cgiGet2Post() enables access to data generated by method GET
     via cgiGetInit()/cgiGetNext(). The data generated by method POST is
     not accessible after this call, unless cgiGet2PostReset() is used.<br>
     Precondition: none<br>
     Postconditon: if 'QUERY_STRING' is defined, cgiGetNextPair()
     can be called on name/value pairs of 'QUERY_STRING' (method GET),
     but not on name/value pairs of stdin (method POST) anymore
     until you call cgiGet2PostReset() afterwards.<br>
     Implementation: content of 'QUERY_STRING' is copied to gCgiBuffer,
     the allocated memory is not freed until the program exits.
     @return pointer to value of non-empty 'QUERY_STRING';
             returns NULL, if QUERY_STRING is not defined or empty
  */
  char *queryString = getenv ("QUERY_STRING");
  int len;
  if (queryString != NULL && (len = strlen (queryString)) > 0) {
    gCgiBufferSize = len;
    strReplace (&gCgiBuffer,queryString);
    return queryString;
  }
  return NULL;
}

void cgiGet2PostReset (void) {
  /**
     Allows to use the functions cgiGetInit()/cgiGetNext() after cgiGet2Post()
     was called, by resetting the CGI buffer.
  */
  hlr_free (gCgiBuffer);
  gCgiBufferSize = 0;
}

static int cgiGetNextPair (int *first,Stringa item,Stringa value) {
  /**
     This function is the basis for all other calls here to read
     from a form (cgiGetInit/cgiGetNext, cgiGetByName, etc)
     - this function is necessary, all the others are convenience.<br>
     Usage:<br>
     int first = 1;<br>
     Stringa item  = stringCreate (10);<br>
     String value = stringCreate (10);<br>
     while (cgiGetNextPair (&first,item,value)) { ... }
     @param[in] first - on first call should be 1
     @param[in] item - a valid Stringa
     @param[in] value - a valid Stringa
     @param[out] first - changed to 0
     @param[out] item - contains an item
     @param[out] value - contains a value
     @return 1 if one more item/value pair was found, 0 if not
  */
  static char *bufferPtr;
  int mode;
  int c;
  int freadRes;

  if (gCgiBuffer == NULL) {
    char *contentLength = getenv ("CONTENT_LENGTH");
    gCgiBufferSize = contentLength ? atoi (contentLength) : 0;
    if (gCgiBufferSize != 0) {
      gCgiBuffer = (char *)hlr_malloc (gCgiBufferSize);
      freadRes = fread (gCgiBuffer,gCgiBufferSize,1,stdin);
      if (freadRes < 0) {
	die("cgiGetNextPair fread error - please report the bug");
      }
    }
    bufferPtr = gCgiBuffer;
  }
  if (*first) {
    *first = 0;
    bufferPtr = gCgiBuffer;
  }
  if (item == NULL || value == NULL)
    die ("cgiGetNextPair: Array for item or value not initialized");
  stringClear (item);
  stringClear (value);
  mode = 'i';
  while (bufferPtr - gCgiBuffer < gCgiBufferSize) {
    c = *(bufferPtr++);
    if (c == '+')
      c = ' ';
    if (c == '=') {
      cgiDecodeWord (item);
      mode = 'v';
      if (bufferPtr - gCgiBuffer == gCgiBufferSize)
        return 1;
    }
    else if (c == '&' || bufferPtr - gCgiBuffer == gCgiBufferSize) {
      if (mode == 'i')
        return 0;
      else {
        if (bufferPtr - gCgiBuffer == gCgiBufferSize)
          stringCatChar (value,c);
        cgiDecodeWord (value);
        return 1;
      }
    }
    else {
      if (mode == 'i')
        stringCatChar (item,c);
      else if (mode == 'v' &&  c != '\r')
        stringCatChar (value,c);
    }
  }
  return 0;
}

/* ----- functions based on cgiGetNextPair() ----- */
static int gFirst = -1;
static Stringa gItem = NULL;

void cgiGetInit (void) {
  /**
     Precondition: running in a CGI
     Postcondition: cgiGetNext() can be called and will
                    return the first field of the POSTed data, if any
  */
  gFirst = 1;
  stringCreateOnce (gItem,20);
}

char *cgiGetNext (Stringa value) {
  /**
     Get the name and value of the next form field.<br>
     Precondition: successful cgiGetInit() or cgiGetNext()<br>
     Postcondition: next call to cgiGetNext() returns next form field<br>
     Note: cgiGetInit/cgiGetNext do not add functionality to the<br>
           basic routine cgiGetNextPair() but are usually more<br>
           convenient to use.
     @param[in] value - existing Stringa
     @param[out] value - filled with contents of field;
                         contents undefined if no more field
     @return name of field, NULL if no more field;
             the memory returned is managed by this routine; it may
             be written to, but not free'd or realloc'd by the user;
             it stays stable until the next call to this routine.
  */
  if (gFirst == -1)
    die ("cgiGetNext() without cgiGetInit()");
  if (cgiGetNextPair (&gFirst,gItem,value))
    return string (gItem);
  gFirst = -1;
  return NULL;
}

void cgiGetNextNamed (char *name, Stringa value) {
  /**
     Same as cgiGetNext(), but die()s if form fields are
     exhausted or the field found does not match 'name'
     @param[in] name - expected name of form field
     @param[in] value - existing Stringa
     @param[out] value - filled with contents of field
  */
  char *nameFound = cgiGetNext (value);
  if (nameFound == NULL)
    die ("cgiGetNextNamed(): no more form fields (expected form field '%s')",
         name);
  if (name == NULL || name[0] == '\0')
    die ("cgiGetNextNamed: no or empty name supplied");
  if (strDiffer (nameFound,name))
    die ("cgiGetNextNamed(): expected form field '%s' but found '%s'",
         name,nameFound);
}

char *cgiGetByName (char *name) {
  /**
     Get and return value of name from POST<br>
     Note: The memory returend is managed by this routine.<br>
           The user may change it, but not free or realloc it.<br>
           The value returned is stable until the next call
           to this routine.
     @param[in] name  -- name of item posted
     @return NULL if no item named 'name' found, else
             pointer to value associated with 'name'
  */
  int first = 1;
  static Stringa item = NULL;
  static Stringa value = NULL;

  stringCreateOnce (item,10);
  stringCreateOnce (value,10);
  while (cgiGetNextPair (&first,item,value))
    if (strEqual (name,string (item)))
      return string (value);
  return NULL;
}

char *cgiGetByNameM (char *name) {
  /**
     Like cgiGetByName(), but die()s if field not found (M = mandatory)
  */
  char *cp = cgiGetByName (name);
  if (cp == NULL)
    die ("cgiGetByNameM: name %s not found",name);
  return cp;
}

/* ##### Section: encoding/decoding strings ##### */

void cgiEncodeWord (char *s,Stringa a) {
  /**
     Encode string for use in a URL
     @param[in] s - null terminated string
     @param[in] a - Stringa (must exist)
     @param[out] a - with URL-conform translation of s
  */
  char *cp = s - 1;
  unsigned char c;
  char hex[3];

  if (a == NULL)
    die ("cgiEncodeWord");
  stringClear (a);
  while ((c = *++cp) != '\0') {
    if (!isalnum(c) && c != '_' && c != '-' && c != '.' && c != ':') {
      snprintf (hex,sizeof hex,"%02X",c);
      stringCatChar (a,'%');
      stringCatChar (a,hex[0]);
      stringCatChar (a,hex[1]);
    }
    else
      stringCatChar (a,c);
  }
}

char *cgiEncodeW (char *s) {
  /**
     Same as cgiEncodeWord(), but returns pointer to result.<br>
     Result memory is owned by this routine.<br>
     Don't free() or realloc() the memory returned.<br>
     Contents stay stable until next call to this routine.<br>
  */
  static Stringa b = NULL;
  stringCreateOnce (b,64);
  cgiEncodeWord (s,b);
  return string (b);
}

void cgiDecodeWord (Stringa a) {
  /**
     URL-decodes a Stringa<br>
     Encoded hex characters are turned into their ASCII equivalents, and
     +'s are turned back into spaces.
     @param[in]  a - Stringa
     @param[out] a - modified
  */
  char *cp = string (a) - 1; // one less for start of while loop
  char c;
  char *c2 = string (a);

  while ((c = *++cp) != '\0') {
    switch (c) {
    case '%':
      *c2++ = x2c (++cp);
      ++cp; // skip second hex digit
      if (*cp == '\0')
        die ("cgiDecodeWord: % in post not followed by two characters.");
      break;
    case '+':
      *c2++ = ' ';
      break;
    default:
      *c2++ = c;
    }
  }
  *c2 = '\0';
  stringAdjust (a);
}

void html_encode (char *inText,Stringa outText,int withExceptions) {
  /**
     HTML-encode HTML-special chars (>, <, &, ");<br>
     There are three operating modes:<br>
     - standard: 'inText' is assumed to be non-HTML; every HTML-special char
       is encoded. This mode is turned on by setting 'withExceptions' to 0;<br>
     - a "half-HTML-mode": assume the text is non-HTML,
       except for hyperlinks (<a href>) and the two basic
       formatting markups \<b\> and \<em\>.<br>
       This mode is turned on by setting 'withExceptions' to 1
       and 'inText' IS NOT starting with "<html>".<br>
     - a full HTML-mode.
       'inText' is treated as being already in HTML format - i.e. this routine
       just copies the 'inText' to 'outText' without change;<br>
       This mode is turned on by setting 'withExceptions' to 1
       and 'inText' starts with "<html>"
       @param[in] inText - original text to be encoded
       @param[in] outText - a valid Stringa
       @param[out] outText - the Stringa filled with encoded version of 'inText'
       @param[in] withExceptions - if 1, then exceptions are left alone,
                                   else every special char is encoded
  */
  char *cp = inText;
  int i = 0;
  int j = 0;
  int inAnchorExcep = 0;
  int doReplace = 0;
  int step = 1;

  static struct {
    char orgChar;
    char *encoded;
  } et[] = { // table of chars needing encoding
    {'<',"&lt;"},
    {'>',"&gt;"},
    {'&',"&amp;"},
    {'"',"&quot;"},
    {'\0',NULL}
  };

  static struct {
    char *text;
    int len;
  } exceptions[] = {
    {"<A HREF",7},
    {"</A>",4},
    {"<EM>",4},
    {"</EM>",5},
    {"<B>",3},
    {"</B>",4},
    {NULL,0}
  };

  if (inText == NULL || outText == NULL)
    die ("html_encode: null input");
  if (withExceptions && strNCaseEqual (inText,"<html>",6)) {
    stringCpy (outText,inText);
    return;
  }
  stringClear (outText);
  while (*cp != '\0') {
    doReplace = 0;
    step = 1;
    for (i=0;et[i].orgChar;i++) {
      if (*cp == et[i].orgChar) {
        if (withExceptions) {
          for (j=0;exceptions[j].text;j++) {
            if (strNCaseEqual (cp,exceptions[j].text,exceptions[j].len)) {
              if (j == 0) // remember anchor start was found
                inAnchorExcep = 1;
              step = exceptions[j].len;
              break;
            }
          }
          if (!exceptions[j].text)
            if (!inAnchorExcep)
              doReplace = 1;
          if (inAnchorExcep && *cp == '>')
            inAnchorExcep = 0;
        }
        else
          doReplace = 1;
        break;
      }
    }
    if (doReplace)
      stringCat (outText,et[i].encoded);
    else // copy as is
      stringNCat (outText,cp,step);
    cp += step;
  }
}

char *html_encodeS (char *s) {
  /**
     Like html_encode(), but manages the buffer;<br>
     Note that code like
     printf ("%s %s",html_encodeS (s1),html_encodeS (s2))
     doesn't work because there is only one buffer.
  */
  static Stringa b = NULL;
  stringCreateOnce (b,100);
  html_encode (s,b,/*withExceptions*/0);
  return string (b);
}

/* ##### Section: cgiURL: mini-module for constructing parameterized URLs ##### */

static Stringa cgiurl = NULL;
static int cgiHasParams;
static Stringa cgiword = NULL;

void cgiURLCreate (char *host,int port,char *program) {
  /**
     Start creation of an URL.<br>
     Precondition : none<br>
     Postcondition: cgiURLAdd*() and cgiURLGet() can be called<br>
     Typycial sequence of calls:<br>
     cgiURLCreate - cgiURLAdd - cgiURLAddInt - cgiURLGet - cgiURLCreate ...<br>
     @param[in] host - e.g. bioinfo.bas.roche.com
     @param[in] port -- e.g. 8080, 0 means no port
     @param[in] program - e.g. /htbin/fetch_noform.cgi
  */
  char portStr[20];
  stringCreateOnce (cgiurl,50);
  stringCreateOnce (cgiword,50);
  if (!host)
    die ("cgiURLCreate: NULL host");
  stringCpy (cgiurl,"http://");
  stringCat (cgiurl,host);
  if (port != 0) {
    stringCat (cgiurl,":");
    snprintf (portStr,sizeof portStr,"%d",port);
    stringCat (cgiurl,portStr);
  }
  if (program[0] != '/')
    stringCat (cgiurl,"/");
  stringCat (cgiurl,program);
  cgiHasParams = 0;
}

void cgiURLCreate2 (char *cgiServerUrl,char *program) {
  /**
     This is an alternative to cgiURLCreate, in case you already
     have an URL stub.<br>
     Precondition : none<br>
     Postcondition: cgiURLAdd() and cgiURLGet() can be called<br>
     Typycial sequence of calls:<br>
     cgiURLCreate2 - cgiURLAdd - cgiURLAdd - cgiURLGet - cgiURLCreate ...<br>
     @param[in] cgiServerUrl - e.g. http://bioinfo:8080/htbin
     @param[in] program - e.g. fetch_noform
                          or NULL (if NULL, then only cgiServerUrl is used)
  */
  stringCreateOnce (cgiurl,50);
  stringCreateOnce (cgiword,50);
  stringCpy (cgiurl,cgiServerUrl);
  if (program) {
    stringCat (cgiurl,"/");
    stringCat (cgiurl,program);
  }
  cgiHasParams = 0;
}

void cgiURLAdd (char *param) {
  /**
     Add parameter to url currently under construction.<br>
     Precondition: last call was cgiURLCreate*() or cgiURLAdd() or
                   cgiURLAddInt()<br>
     Postcondition: cgiURLGet() will return URL with param included
     @param[in] param - NULL terminated string
  */
  if (cgiurl == NULL)
    die ("cgiURLAdd without cgiURLCreate");
  stringCat (cgiurl,(cgiHasParams ? "+" : "?"));
  cgiHasParams = 1;
  cgiEncodeWord (param,cgiword);
  stringCat (cgiurl,string (cgiword));
}

void cgiURLAddNV (char *name,char *value) {
  /**
     Add name-value pair to url currently under construction<br>
     Precondition: last call was cgiURLCreate*() or cgiURLAddNV()<br>
     Postcondition: cgiURLGet() will return URL with param included
     @param[in] name - null terminated string
     @param[in] value - null terminated string; if NULL
                        then this call has no effect
  */
  if (cgiurl == NULL)
    die ("cgiURLAddNV without cgiURLCreate");
  stringCat (cgiurl,(cgiHasParams ? "&" : "?"));
  cgiHasParams = 1;
  cgiEncodeWord (name,cgiword);
  stringCat (cgiurl,string (cgiword));
  if (value != NULL) {
    stringCat (cgiurl,"=");
    cgiEncodeWord (value,cgiword);
    stringCat (cgiurl,string (cgiword));
  }
}

void cgiURLAddNVInt (char *name,int value) {
  /**
     Linke cgiURLAddNV but value is an int
  */
  char s[HLR_ITOA_SIZE];
  hlr_itoa (s,value);
  cgiURLAddNV (name,s);
}

void cgiURLAddInt (int param) {
  /**
     Add integer parameter to url currently under construction<br>
     Precondition: last call was cgiURLCreate*() or cgiURLAdd() or
                   cgiURLAddInt()<br>
     Postcondition: cgiURLGet() will return URL with param included
     @param[in] param - number to be added
  */
  char s[HLR_ITOA_SIZE];
  hlr_itoa (s,param);
  cgiURLAdd (s);
}

char *cgiURLGet (void) {
  /**
     Returns the constructed URL.<br>
     Precondition: last call was cgiURLCreate*() or cgiURLAdd*()
                   (for calling sequence see cgiURLCreate())<br>
     Postcondition: next allowed calles to cgiURL*() functions are
                    cgiURLCreate*() or repeated call of cgiURLGet()<br>
     Note: The memory is managed by this routine.<br>
           It may by written to, but not free'd or realloc'd by the
           caller of this this routine. It contents stay stable until
           the next call to cgiURLCreate*()
     Returns: URL constructed.
  */
  return string(cgiurl);
}

/* ##### Section: construct HTML text e.g. for hyperlinks  ##### */

static char *html_host = NULL;
static char *html_program = NULL;
static char *html_option = NULL;
static int html_port = 0; /* 0  = default HTTP port (80),
                             -1 = port and cgi path contained in html_host
                             >0 = use this port in constructing the URL
                          */

void html_URLSet (char *host,int port,char *program) {
  /**
     Fix URL parts for subsequent calls to html_hlink() and html_clink()<br>
     Typical usage:<br>
     html_URLSet ("bioinfo.bas.roche.com",8080,"/htbin/proteomiscgidev");<br>
     printf ("<FORM ACTION=%s METHOD=POST>\n",html_clink ("login","do"));<br>
     html_hlink ("menu","display","Goto Menu");<br>
     Inputs: like cgiURLCreate()
  */
  if (port < 0)
    die ("html_URLSet");
  strReplace (&html_host,host);
  html_port = port;
  strReplace (&html_program,program);
  hlr_free (html_option);
}

void html_URLSet2 (char *cgiServerUrl,char *program) {
  /**
     Same function as html_URLSet(), but different interface<br>
     Inputs: like cgiURLCreate2()
  */
  html_port = -1; // indicate mode
  strReplace (&html_host,cgiServerUrl);
  strReplace (&html_program,program);
  hlr_free (html_option);
}

void html_URLOptSet (char *option) {
  /**
     Sets optional argument to be added to the URL<br>
     Precondition: html_URLSet<br>
     Postcondition: html_clink*() will insert 'option' as first parameter
     @param[in] option - can also be NULL, can be changed after this
                         function has been called without affecting
                         the postcondition
  */
  if (html_host == NULL)
    die ("html_URLOptSet: call html_URLSet() first");
  strReplace (&html_option,option);
}

char *html_clink4 (char *class1,char *method,
                   char *p1,char *p2,char *p3,char *p4) {
  /**
     Construct parameterized URL to the program set in html_URLSet()
     (c for cgi)<br>
     Precondition: html_URLSet() was called.<br>
  */
  if (html_port == -1)
    cgiURLCreate2 (html_host,html_program);
  else
    cgiURLCreate (html_host,html_port,html_program);
  if (html_option != NULL)
    cgiURLAdd (html_option);
  cgiURLAdd (class1);
  cgiURLAdd (method);
  if (p1 != NULL)
    cgiURLAdd (p1);
  if (p2 != NULL)
    cgiURLAdd (p2);
  if (p3 != NULL)
    cgiURLAdd (p3);
  if (p4 != NULL)
    cgiURLAdd (p4);
  return cgiURLGet ();
}

void html_hlink4 (char *class1,char *method,char *label,
                  char *p1,char *p2,char *p3,char *p4) {
  /**
     Construct a parameterized hyperlink to the program set in html_URLSet()
     (h for hyperlink).<br>
     Precondition: html_URLSet() was called.<br>
  */
  Rprintf ("<a href=\"%s\">%s</a>",
          html_clink4 (class1,method,p1,p2,p3,p4),label);
}

/* ##### Section: generate hyperlinks to named windows such that these pop up when navigated to ##### */

static char *gCSS_class = NULL;

void html_setCSSclass (char *css_class) {
  /**
     Set the CSS class for the urlLabel in html_getPopupLinkAdv()
     @param[in] css_class - full HTML CSS class spec, e.g. 'class=bih_menubar'.
                            Can be NULL. Stays in effect until cleared by
                            html_setCSSclass(NULL).
  */
  strReplace (&gCSS_class,css_class);
}

char *html_getPopupLinkAdv (char *url,char *urlLabel,char *windowName,
                            int isButton,int width,int height,
                            int showScrollbars,int isResizeable,int showBars) {
  /**
     Generates a html hyperlink/button which opens an URL in a popup
     window.<br>
     The memory is managed by this routine and valid until next function call.
     @param[in] url - e.g. "http://xxx.com:8080/bicgi/bioinfo_homepage_cgi"
     @param[in] urlLabel - e.g. "Bioinfo Homepage Link"
     @param[in] windowName - name of window to be opened, e.g. "biHomePopup1"
     @param[in] isButton - if 1 create button, else create regular looking
                           hyperlink
     @param[in] width/height - window width/height in pixels. If 0, default is
                               used.
     @param[in] showScrollbars - if 0/1 hide/show scrollbars
     @param[in] isResizeable - if 0 make window non-resizeable, else set to 1
     @param[in] showBars - if 0/1 hide/show statusbar, toolbar, menubar and
                           addressbar
     @return html hyperlink/button.
  */
  char c;
  char *cp = NULL;
  static Stringa urlStr=NULL;
  static Stringa retVal=NULL;
  static Stringa js=NULL;
  static Stringa heightStr=NULL;
  static Stringa widthStr=NULL;
  stringCreateClear (urlStr,200);
  stringCreateOnce (js,200);
  stringCreateOnce (retVal,300);
  stringCreateClear (heightStr,12);
  stringCreateClear (widthStr,12);
  if (width)
    stringPrintf (widthStr,"width=%d,",width);
  if (height)
    stringPrintf (heightStr,"height=%d,",height);
  /* To avoid that javascript decodes a encoded url, the url is split
     after a '%' and reassembled with '+' operator.
     In general the url is encoded automatically, but if a value of a post
     parameter contains a '&' it is necessary to encode before, e.g.
     http://.../a=123&b=4&5 --> http://.../a=123&b=4%265
  */
  cp = url - 1;
  while ((c = *++cp) != '\0') {
    stringCatChar (urlStr,c);
    if (*(cp+1) != '\0' && c=='%')
      stringCat (urlStr,"'+'");
  }

  stringPrintf (js,"var w=window.open('%s','%s','%s%s"
                "scrollbars=%d,resizable=%d,status=%d,toolbar=%d,location=%d,menubar=%d'); "
                "w.focus();",
                string (urlStr),windowName,string (widthStr),
                string (heightStr),showScrollbars,isResizeable,
                showBars,showBars,showBars,showBars);

  if (isButton)
    stringPrintf (retVal,"<button name=\"%s\" type=button onClick=\"%s\">%s</button>",
                  windowName,string (js),urlLabel);
  else
    stringPrintf (retVal,"<a href=\"javascript:%s\" %s>%s</a>",
                  string (js),gCSS_class ? gCSS_class : "",urlLabel);
  return string (retVal);
}

char *html_getPopupLink (char *url,char *urlLabel,char *windowName,
                         int isButton) {
  /**
     Generates an html hyperlink/button which opens an URL in a popup window .<br>
     The memory is managed by this routine and valid until next function call.
     @param[in] url - e.g. "http://xxx.com:8080/bicgi/bioinfo_homepage_cgi"
     @param[in] urlLabel - e.g. "Bioinfo Homepage Link"
     @param[in] windowName - name of window to be opened, e.g. "biHomePopup1"
     @param[in] isButton - if 1 create button, else create regular looking
                           hyperlink
     @return html hyperlink/button.
  */
  return html_getPopupLinkAdv (url,urlLabel,windowName,isButton,
                               0/*width*/,0/*height*/,
                               1/*showScrollbars*/, 1/*isResizeable*/,
                               1/*showBars*/);
}

void html_printPopupLinkAdv (char *url,char *urlLabel,char *windowName,
                             int isButton,int width,int height,
                             int showScrollbars,int isResizeable,int showBars) {
  /**
     Prints a html hyperlink/button which opens an URL in a popup window.
     @param[in] url - e.g. "http://xxx.com:8080/bicgi/bioinfo_homepage_cgi"
     @param[in] urlLabel - e.g. "Bioinfo Homepage Link"
     @param[in] windowName - name of window to be opened, e.g. "biHomePopup1"
     @param[in] isButton - if 1 create button, else create regular looking
                           hyperlink
     @param[in] width/height - window width/height in pixels. If 0, default is
                               used.
     @param[in] showScrollbars - if 0/1 hide/show scrollbars
     @param[in] isResizeable - if 0 make window non-resizeable, else set to 1
     @param[in]  showBars - if 0/1 hide/show statusbar, toolbar, menubar and
                            addressbar
  */
  Rprintf ("%s",
          html_getPopupLinkAdv (url,urlLabel,windowName,isButton,width,height,
                                showScrollbars,isResizeable,showBars));
}

void html_printPopupLink (char *url,char *urlLabel,char *windowName,
                          int isButton) {
  /**
     Prints a html hyperlink/button which opens an URL in a popup window
     @param[in] url - e.g. "http://xxx.com:8080/bicgi/bioinfo_homepage_cgi"
     @param[in] urlLabel - e.g. "Bioinfo Homepage Link"
     @param[in] windowName - name of window to be opened, e.g. "biHomePopup1"
     @param[in] isButton - if 1 create button, else create regular looking
                           hyperlink
  */
  html_printPopupLinkAdv (url,urlLabel,windowName,isButton,
                          0/*width*/,0/*height*/,1/*showScrollbars*/,
                          1/*isResizeable*/,1/*showBars*/);
}

int html_uniqueIntGet (void) {
  /**
     Generate an integer that is unique for this process.<br>
     Typcial usage: <br>
     printf ("<a href=\"http://adr\" target=%d>",html_uniqueIntGet ());<br>
     This will make the browser to open a new window when the
     user clicks the hyperlink.<br>
     BUGS: processes running in the same second will have
           the same "unique" id.
  */
  static int uniqueInt = 0;

  if (uniqueInt == 0)
    uniqueInt = time (NULL);
  return uniqueInt;
}

/* ##### Section: formatting HTML tables ##### */

char *html_tab2table (char *tab,int firstLineIsHeader,int borderWidth,
                      int withMarkup) {
  /**
     Converts tab delimited ASCII data into HTML tables;<br>
     @param[in] tab - buffer containing data; newline delimites rows, tab
                      separates items within a row;<br>
                      HTML options can be included in the <td> or <th> tags:
                      these options must be enclosed by backslash characters
                      and occur directly at the beginning of an item
     @param[in] firstLineIsHeader - if 1, for the first non-empty line TH tags
                                    will be used
     @param[in] borderWidth - 0 means no border
     @param[in] withMarkup - 1 to parse for /.../ at the beginning of each
                             field, 0 to output each field unchanged
     @param[out] tab - contents of 'tab' destroyed
     @return a buffer containing an HTML formatted table;
             read/write access OK; memory managed by this routine.
  */
  /*
    Example:

    "Name \t Age \n
    Adam \t 45 \n
    Eve \t\bgcolor=red\ 42 \n"

    would be transformed into (assuming firstLineIsHeader==1, withMarkup==1)

    <table border=0>
    <tr><th>Name</th><th>Age</th></tr>\n
    <tr><td>Adam</td><td>45</td></tr>\n
    <tr><td>Eva</td><td bgcolor=red>42</td></tr>\n
    </table>
  */
  static Stringa html = NULL;
  LineStream ls;
  char *line;
  char hd = firstLineIsHeader ? 'H' : 'D'; // hd = <tH> or <tD>
  WordIter wi;
  char *word;
  char *endOfMarkup;
  char *value;

  stringCreateClear (html,1000);
  ls = ls_createFromBuffer (tab);
  stringAppendf (html,"<TABLE BORDER=%d>\n",borderWidth);
  while ((line = ls_nextLine (ls)) != NULL) {
    if (line[0] == '\0')
      continue;
    stringCat (html,"<TR>");
    wi = wordFldIterCreate (line,"\t");
    while ((word = wordNext (wi)) != NULL) {
      if (withMarkup && *word=='\\' && (endOfMarkup=strchr (word+1,'\\')) ) {
        *endOfMarkup='\0';
        value = (endOfMarkup[1] ? endOfMarkup+1 : "&nbsp;");
        stringAppendf (html,"<T%c %s>%s</T%c>",
                       hd,word+1,value,hd);
      }
      else {
        value = (word[0] ? word : "&nbsp;");
        stringAppendf (html,"<T%c>%s</T%c>",
                       hd,value,hd);
      }
    }
    wordIterDestroy (wi);
    stringCat (html,"</TR>\n");
    hd = 'D';
  }
  ls_destroy (ls);
  stringCat (html,"</TABLE>\n");
  return string (html);
}

char *html_text2tables (char *tab,int firstLineIsHeader,int borderWidth,
                        int withMarkup) {
  /**
     Like html_tab2table(), but allows for several tables separted by lines
     not containing tab<br>
     Input format: one or more table sections can be embedded in normal text;
                   the beginning and end of these sections is determined
                   automatically (lines in tables contain at least one tab);
     @param[in] tab - buffer containing data; newline delimites rows, tab
                      separates items within a row;<br>
                      HTML options can be included in the <td> or <th> tags:
                      these options must be enclosed by backslash characters
                      and occur directly at the beginning of an item
     @param[in] firstLineIsHeader - if 1, for the first non-empty line TH tags
                                    will be used
     @param[in] borderWidth - 0 means no border
     @param[in] withMarkup - 1 to parse for /.../ at the beginning of each
                             field, 0 to output each field unchanged
     @param[out] tab - contents of 'tab' destroyed
     @return a buffer containing an HTML formatted table;
             read/write access OK; memory managed by this routine.
  */
  /*
    Example:

    "This is just ordinary text\n
    The table starts in the next line\n
    Name \t Age \n
    Adam \t 45 \n
    Eve \t\bgcolor=red\ 42 \n
    This is a footer"

    would be transformed into (assuming firstLineIsHeader==1, withMarkup==1)

    "This is just ordinary text<br>\n
    The table starts in the next line<br>\n
    <table border=0>
    <tr><th>Name</th><th>Age</th></tr>\n
    <tr><td>Adam</td><td>45</td></tr>\n
    <tr><td>Eva</td><td bgcolor=red>42</td></tr>\n
    </table>
    This is a footer<br>"
  */
  static Stringa html = NULL;
  static Stringa tabbuf = NULL;
  LineStream ls;
  char *line;

  stringCreateClear (html,1000);
  stringCreateClear (tabbuf,1000);
  ls = ls_createFromBuffer (tab);
  while ((line = ls_nextLine (ls)) != NULL) {
    if (line[0] == '\0')
      continue;
    if (!strchr (line,'\t')) {
      if (stringLen (tabbuf)) {
        stringCat (html,html_tab2table (string (tabbuf),firstLineIsHeader,
                                        borderWidth,withMarkup));
        stringClear (tabbuf);
      }
      stringCat (html,line);
      stringCat (html,"<br>\n");
      continue;
    }
    // line with 2 or more columns
    stringCat (tabbuf,line);
    stringCat (tabbuf,"\n");
  }
  ls_destroy (ls);
  if (stringLen (tabbuf))
    stringCat (html,html_tab2table (string (tabbuf),firstLineIsHeader,
                                    borderWidth,withMarkup));
  return string (html);
}

/* ##### Section: creating applet tags ##### */

static FILE *gAppletTagOutFileP = NULL;
static Texta gParamNames = NULL;
static Texta gParamValues = NULL;
static Stringa gEmbedTag = NULL;

void html_appletTagOpen (FILE *fp,char *jarFileUrls,char *appletClass,
                         int width,int height) {
  /**
     Construct html that will start a java applet under Internet Explorer
     and in case this is not possible, show a suitable diagnostic message to
     the user.<br>
     Postcondition: HTML for loading applet has been printed;
                    html_appletParam() and html_appletTagClose()
                    can be called<br>
     Known bugs: - forces restart of Java Runtime Environment if mixed use
                   of several Java versions.<br>
                 - under some circumstances starts a second Java console
                   window
     @param[in] fp - output destination, e.g. stdout
     @param[in] jarFileUrls - e.g. http://xxx.com:8080/genome/genomeviewer.jar
                              (separate several JARs with ',' (comma))
     @param[in] appletClass - e.g. GenomeApplet.class
     @param[in] width - in pixels
     @param[in] height - in pixels
  */
  gAppletTagOutFileP = fp;
  fprintf (fp,"<object "
           "classid=\"clsid:8AD9C840-044E-11D1-B3E9-00805F499D93\" "
           "width=%d height=%d>\n"
           "<param name=code value=\"%s\">\n"
           "<param name=archive value=\"%s\">\n",
           width,height,appletClass,jarFileUrls);
  textCreateClear (gParamNames,10);
  textCreateClear (gParamValues,10);
  stringCreateClear (gEmbedTag,100);
  stringCat (gEmbedTag,"<comment>\n");
  stringAppendf (gEmbedTag,"<embed code=\"%s\" archive=\"%s\"\n"
                 "type=\"application/x-java-applet;version=1.6\"\n"
                 "width=%d height=%d\n",
                 appletClass,jarFileUrls,width,height);
}

void html_appletParam (char *name,char *value) {
  /**
     Include a parameter to an open applet tag.<br>
     Precondition: html_appletTagOpen()<br>
     Known bugs: neither name nor value must contain a double quote (")
                 (how would one encode those properly?)<br>
     @param[in] name
     @param[in] value - can be NULL
  */
  if (strchr (name, '"') || (value && strchr (value, '"')))
    die ("html_appletParam: \" found.");
  textAdd (gParamNames,name);
  if (value)
    textAdd (gParamValues,value);
  else
    textAdd (gParamValues,"");
}

void html_appletTagClose (void) {
  /**
  Precondition: html_appletTagOpen().<br>
  Postcondition: HTML closing applet tag has been printed
                 html_appletTagOpen() can be called again.
  */
  int i;

  for (i=0;i<arrayMax (gParamNames);i++)
    fprintf (gAppletTagOutFileP,"<param name=\"%s\" value=\"%s\">\n",
             textItem (gParamNames,i),textItem (gParamValues,i));
  fprintf (gAppletTagOutFileP,"%s",string (gEmbedTag));
  for (i=0;i<arrayMax (gParamNames);i++)
    fprintf (gAppletTagOutFileP,"%s=\"%s\"\n",
             textItem (gParamNames,i),textItem (gParamValues,i));
  fprintf (gAppletTagOutFileP,
           "><noembed>No Java support</noembed>\n</embed>\n"
           "</comment></object>\n");
}

/* ##### Section: creating web start jnlp tags ##### */

static FILE *gFp = NULL;
static Texta gJars = NULL;
static Texta gArgs = NULL;
static char *gMainClass = NULL;

void html_webstartOpen (FILE *fp,char *codebase,char *title,char *homepage,
                        char *description,char *icon,int allPermissions,
                        char *heap,char *mainClass) {
  /**
     @param[in] fp - where to write the XML text
     @param[in] codebase - code base of JAVA application
     @param[in] title - title of the application
     @param[in] homepage - home page of the application (can be NULL)
     @param[in] description - description of the application (can be NULL)
     @param[in] icon - icon of the application (can be NULL)
     @param[in] allPermissions - whether the application has all permissions
                                 (essentially whether it is signed)
     @param[in] heap - will be used in the j2se tag: max-heap-size, e.g. 400m
     @param[in] mainClass - main class of the application
  */
  gFp = fp;
  fprintf (fp,"<jnlp spec='6.0+' codebase='%s'>\n",codebase);
  fprintf (fp,"  <information>\n");
  if (title == NULL)
    die ("html_webstartOpen: a title is required");
  fprintf (fp,"    <title>%s</title>\n",title);
  fprintf (fp,"    <vendor>bioinfoc.ch</vendor>\n");
  if (homepage != NULL)
    fprintf (fp,"    <homepage href='%s'/>\n",homepage);
  if (icon != NULL)
    fprintf (fp,"    <icon href='%s'/>\n",icon);
  if (description != NULL)
    fprintf (fp,"    <description>%s</description>\n",description);
  fprintf (fp,"  </information>\n");
  if (allPermissions) {
    fprintf (fp,"  <security>\n");
    fprintf (fp,"    <all-permissions/>\n");
    fprintf (fp,"  </security>\n");
  }
  fprintf (fp,"  <resources>\n");
  if (heap != NULL)
    fprintf (fp,"    <j2se version='1.6.+' max-heap-size='%s'/>\n",
             heap);
  else
    fprintf (fp,"    <j2se version='1.6.+'/>\n");
  gMainClass = hlr_strdup (mainClass);
  textCreateClear (gJars,5);
  textCreateClear (gArgs,5);
}

void html_webstartAddJar (char *jar) {
  /**
     Call after html_webstartOpen;
     call this function for every jar used by the application
     @param[in] jar - name of a jar
  */
  textAdd (gJars,jar);
}

void html_webstartAddArg (char *arg) {
  /**
     Call after html_webstartOpen;
     call this function for every argument used to invoke the application
     @param[in] arg - an argument of the application
  */
  static Stringa enc = NULL;

  stringCreateOnce (enc,20);
  html_encode (arg,enc,0);
  textAdd (gArgs,string (enc));
}

void html_webstartClose (void) {
  /**
     Must be called to finish writing the jnlp XML
  */
  int i;

  for (i=0;i<arrayMax (gJars);i++)
    fprintf (gFp,"    <jar href='%s'/>\n",arru (gJars,i,char *));
  fprintf (gFp,"  </resources>\n");
  fprintf (gFp,"  <application-desc main-class='%s'>\n",gMainClass);
  hlr_free (gMainClass);
  for (i=0;i<arrayMax (gArgs);i++)
    fprintf (gFp,"    <argument>%s</argument>\n",arru (gArgs,i,char *));
  fprintf (gFp,"  </application-desc>\n");
  fprintf (gFp,"</jnlp>\n");
}

/* ##### Section: multi part stuff ##### */
/*
  Mini-module mp (multipart) for reading data from multipart forms
  these forms MUST have ENCTYPE="multipart/form-data" and can contain
  separate parts where each part may contain text or binary data.
  Functions from this mini-module use prefix cgiMp
  Example program using these functions:
  http://bioinfo.bas.roche.com:8080/bios/common/bioCompLibSeminar.html
  see 'Code example for file upload via WWW browser'
*/

static int gMpUsed = 0;
static int gMpBufferSize = 0;
static char *gMpBuffer = NULL;
static char gMpDelimiter[] = {0x0d,0x0a};
static int gMpBoundarySize = 0;
static char *gMpBoundary = NULL;
static char *gMpReadPos = NULL;

static int cgiIsMpInitialized (void) {
  /**
     @return 1 if mini module mp is initialized and 0 else
  */
  return gMpBufferSize != 0 && gMpBuffer != NULL &&
    gMpBoundarySize != 0 && gMpBoundary != NULL;
}

static char *mymemmem (char *s1,int sz1,char *s2,int sz2) {
  /**
    Equivalent of strstr() on memory, returns pointer to first byte
    of first occurrence of s2 in s1
  */
  char *result = NULL;
  char *cp = s1;
  int sz = sz1 - sz2;

  while (cp - s1 <= sz) {
    if (!memcmp (cp,s2,sz2)) {
      result = cp;
      break;
    }
    cp++;
  }
  return result;
}

/* useful for debugging when using mem stuff instead of zero-terminated strings
static void memputs (char *s,int sz) {
  int i = 0;
  while (i < sz)
    putchar ((int)s[i++]);
}
*/

int cgiMpInit (void)  {
  /**
     Initialize mini module mp.<br>
     Precondition: mp not initialized.<br>
     Postcondition: mp initialized, other cgiMp... functions can be called.<br>
     Note: example call sequence:<br>
     cgiMpInit();<br>
     while (cgiMpNext (a,b,c,d))<br>
     ...<br>
     cgiMpReset();<br>
     while (cgiMpNext (a,b,c,d))<br>
     ...<br>
     cgiMpDeinit();<br>
     Deinitialization may be crucial e.g. if VERY LARGE files are read
     since these functions are reading from stdin initialization will
     be allowed only once; but contents can be read multiple times by
     using cgiMpReset()
     @return 1 if mp initialization successful, 0 if form is not multipart or
             there is no form at all
  */
  char *contentType = getenv ("CONTENT_TYPE");
  char *delimiterPos = (char *)ILLADR;

  if (contentType == NULL || strNDiffer (contentType,"multipart/form-data",19))
    return 0;
  if (cgiIsMpInitialized ())
    die ("cgiMpInit(): illegal use - already initialized");
  if (gMpUsed)
    die ("cgiMpInit(): illegal use - has been used before");
  if (contentType == NULL || strNDiffer (contentType,"multipart/form-data",19))
    die ("cgiMpInit(): invalid content-type: %s",s0 (contentType));
  gMpBufferSize = atoi (getenv ("CONTENT_LENGTH"));
  gMpBuffer = (char *)hlr_malloc (gMpBufferSize);
  if (fread (gMpBuffer,gMpBufferSize,1,stdin) != 1)
    die ("cgiMpInit(): input length does not match content length %d",
         gMpBufferSize);
  delimiterPos = mymemmem (gMpBuffer,gMpBufferSize,gMpDelimiter,
                           sizeof (gMpDelimiter));
  if (delimiterPos == NULL || delimiterPos == gMpBuffer)
    die ("cgiMpInit(): invalid content format: %s",gMpBuffer);
  gMpBoundarySize = delimiterPos - gMpBuffer;
  gMpBoundary = (char *)hlr_malloc (gMpBoundarySize);
  memcpy (gMpBoundary,gMpBuffer,gMpBoundarySize);
  gMpUsed = 1;
  cgiMpReset ();
  return 1;
}

int cgiMpNext (Stringa item,Array value,Stringa filename,
               Stringa contentType) {
  /**
     Returns next item/value pair plus optional information for file upload.<br>
     Precondition: mp initialized (cgiMpInit ())
     @param[in] item - Stringa to store item name in, must not be NULL
     @param[in] value - Array of char to store item value in, must not be NULL,
                        not Stringa because data might be binary
     @param[in] filename - Stringa to store filename in if any;
                           may be NULL if not interested
     @param[in] contentType - Stringa to store content MIME type in;
                              may be NULL if not interested
     @param[out] item - name of the item read
     @param[out] value - value of the item; a Stringa if form data or
                         a text file; binary buffer (no trailing '\0') when
                         data from a binary file
     @param[out] filename - the filename if value is from a file,
                            an empty string "" if value is not from a file
                            but a form field.
     @param[out] contentType - MIME type of content; empty ("") if form data or
                               a text file (MIME type="text/plain");
                               a binary file else.
     @return 0 if all items used up, 1 else
  */

  int result = 0;
  char *readPos = (char *)ILLADR;
  char *boundaryPos = (char *)ILLADR;
  char *delimiterPos = (char *)ILLADR;
  char *namePos = (char *)ILLADR;
  char *filenamePos = (char *)ILLADR;
  Stringa myContentType = stringCreate (32);

  if (!cgiIsMpInitialized ())
    die ("cgiMpNext(): illegal use - not initialized");
  stringClear (item);
  stringClear (value);
  if (filename != NULL)
    stringClear (filename);
  if (contentType != NULL)
    stringClear (contentType);
  boundaryPos = mymemmem (gMpReadPos,gMpBufferSize - (gMpReadPos - gMpBuffer),
                          gMpBoundary,gMpBoundarySize);
  if (boundaryPos != NULL) {
    result = 1;
    readPos = gMpReadPos;
    gMpReadPos = boundaryPos + gMpBoundarySize + sizeof (gMpDelimiter);
    // parse content-disposition line
    delimiterPos = mymemmem (readPos, boundaryPos - readPos,
                             gMpDelimiter,sizeof (gMpDelimiter));
    if (delimiterPos == NULL)
      die ("cgiMpNext(): no line delimiter found until next boundary: %s",
           readPos);
    if (strNDiffer (readPos,"Content-Disposition: form-data",30))
      die ("cgiMpNext(): invalid part format",readPos);
    if ((namePos = mymemmem (readPos,delimiterPos - readPos,"name=",5)) != NULL)
      strCopySubstr (namePos,'"','"',item);
    else
      die ("cgiMpNext(): invalid part format: %s",readPos);
    if (filename != NULL)
      if ((filenamePos = mymemmem (namePos,delimiterPos - readPos,
                                   "filename=",9)) != NULL)
        strCopySubstr (filenamePos,'"','"',filename);
    // parse optional content-type line
    readPos = delimiterPos + sizeof (gMpDelimiter);
    delimiterPos = mymemmem (readPos,boundaryPos - readPos,
                             gMpDelimiter,sizeof (gMpDelimiter));
    if (delimiterPos == NULL)
      die ("cgiMpNext(): no line delimiter found until next boundary: %s",
           readPos);
    if (strNEqual (readPos,"Content-Type: ",14)) {
      strCopySubstr (readPos + 13,' ',0x0d,myContentType);
      if (strEqual (string (myContentType),"text/plain"))
        stringClear (myContentType);
      readPos = delimiterPos + sizeof (gMpDelimiter);
      delimiterPos = mymemmem (readPos,boundaryPos - readPos,
                               gMpDelimiter,sizeof (gMpDelimiter));
    }
    // require blank line separation of part header and part data
    if (delimiterPos != readPos)
      die ("cgiMpNext(): invalid part format: %s",readPos);
    readPos += sizeof (gMpDelimiter);
    // read part data (without final delimiter)
    if (isEmptyString (myContentType)) {
      stringNCpy (value,readPos,boundaryPos - readPos - sizeof (gMpDelimiter));
      stringTranslate (value,"\r","");
    }
    else {
      arraySetMax (value,0);
      while (readPos < boundaryPos - sizeof (gMpDelimiter)) {
        array (value,arrayMax (value),char) = *readPos;
        readPos++;
      }
    }
  }
  if (contentType != NULL)
    stringCpy (contentType,string (myContentType));
  stringDestroy (myContentType);
  return result;
}

void cgiMpReset (void) {
  /**
     Reset multi-part reading to first part<br>
     Precondition: mp initialized, 0 - all parts have been read<br>
     Postcondition: ready for reading from first part again
  */
  if (!cgiIsMpInitialized ())
    die ("cgiMpReset(): illegal use - not initialized");
  gMpReadPos = gMpBuffer + gMpBoundarySize + sizeof (gMpDelimiter);
}

void cgiMpDeinit (void) {
  /**
     Deinitialize mp and release all temporarily allocated resources.<br>
     Precondition: mp initialized.<br>
     Postcondition: mp not initialized.<br>
     Note: important to use if some parts are files which can be VERY LARGE.<br>
  */
  if (!cgiIsMpInitialized ())
    die ("cgiMpDeinit(): illegal use - not initialized");
  gMpReadPos = NULL;
  gMpBoundarySize = 0;
  hlr_free (gMpBoundary);
  gMpBufferSize = 0;
  hlr_free (gMpBuffer);
}
