% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sofar.R
\name{sofar}
\alias{sofar}
\title{Sparse orthogonal factor regression}
\usage{
sofar(
  Y,
  X,
  nrank = 1,
  su = NULL,
  sv = NULL,
  ic.type = c("GIC", "BIC", "AIC", "GCV"),
  modstr = list(),
  control = list(),
  screening = FALSE
)
}
\arguments{
\item{Y}{response matrix}

\item{X}{covariate matrix}

\item{nrank}{an integer specifying the desired rank/number of factors}

\item{su}{a scaling vector for U such that \eqn{U^TU = diag(s_u)}.}

\item{sv}{a scaling vector for V such that \eqn{V^TV = diag(s_v)}.}

\item{ic.type}{select tuning method; the default is GIC}

\item{modstr}{a list of internal model parameters controlling the model
fitting}

\item{control}{a list of internal computation parameters controlling
optimization}

\item{screening}{If TRUE, marginal screening via lasso is performed before
sofar fitting.}
}
\value{
A \code{sofar} object containing
  \item{call}{original function call}
  \item{Y}{input response matrix}
  \item{X}{input predictor matrix}
  \item{Upath}{solution path of U}
  \item{Dpath}{solution path of D}
  \item{Vpath}{solution path of D}
  \item{Rpath}{path of estimated rank}
  \item{icpath}{path of information criteria}
  \item{lam.id}{ids of selected lambda for GIC, BIC, AIC and GCV}
  \item{p.index}{ids of predictors which passed screening}
  \item{q.index}{ids of responses which passed screening}
  \item{lamA}{tuning sequence for A}
  \item{lamB}{tuning sequence for B}
  \item{lamD}{tuning sequence for D}
  \item{U}{estimated left singular matrix that is orthogonal (factor weights)}
  \item{V}{estimated right singular matrix that is orthogonal (factor loadings)}
  \item{D}{estimated singular values}
  \item{rank}{estimated rank}
}
\description{
Compute solution paths of sparse orthogonal factor regression
}
\details{
The model parameters can be specified through argument \code{modstr}.
The available elements include
\itemize{
    \item{mu}: parameter in the augmented Lagrangian function.
    \item{mugamma}: increament of mu along iterations to speed up
        computation.
    \item{WA}: weight matrix for A.
    \item{WB}: weight matrix for B.
    \item{Wd}: weight matrix for d.
    \item{wgamma}: power parameter in constructing adaptive weights.
}
The model fitting can be controled through argument \code{control}.
The avilable elements include
\itemize{
   \item{nlam}: number of lambda triplets to be used.
   \item{lam.min.factor}: set the smallest lambda triplets as a fraction of the
    estimation lambda.max triplets.
   \item{lam.max.factor}: set the largest lambda triplets as a multiple of the
    estimation lambda.max triplets.
   \item{lam.AB.factor}: set the relative penalty level between A/B and D.
   \item{penA,penB,penD}: if TRUE, penalty is applied.
   \item{lamA}: sequence of tuning parameters for A.
   \item{lamB}: sequence of tuning parameters for B.
   \item{lamD}: sequence of tuning parameters for d.
   \item{methodA}: penalty for penalizing A.
   \item{methodB}: penalty for penalizing B.
   \item{epsilon}: convergence tolerance.
   \item{maxit}: maximum number of iterations.
   \item{innerEpsilon}: convergence tolerance for inner subroutines.
   \item{innerMaxit}: maximum number of iterations for inner subroutines.
   \item{sv.tol}: tolerance for singular values.
}
}
\examples{
\dontrun{
library(rrpack)
## Simulate data from a sparse factor regression model
p <- 100; q <- 50; n <- 100; nrank <- 3
mydata <- rrr.sim1(n, p, q, nrank, s2n = 1,
                   sigma = NULL, rho_X = 0.5, rho_E = 0.3)
Y <- mydata$Y
X <- mydata$X

fit1 <- sofar(Y, X, ic.type = "GIC", nrank = nrank + 2,
              control = list(methodA = "adlasso", methodB = "adlasso"))
summary(fit1)
plot(fit1)

fit1$U
crossprod(fit1$U) #check orthogonality
fit1$V
crossprod(fit1$V) #check orthogonality
}

}
\references{
Uematsu, Y., Fan, Y., Chen, K., Lv, J., & Lin, W. (2019). SOFAR: large-scale
association network learning. \emph{IEEE Transactions on Information
Theory}, 65(8), 4924--4939.
}
