################################################################################
# ---------------------- DATA TRANSFORMATION FUNCTIONS ----------------------- #
################################################################################

#' Data Transformation
#'
#' Applies the Discrete Cosine I transform, data binning and the variance stabi-
#' lizing transform function to the data.
#' 
#' @param x       \code{p} dimensional data vector.
#' @param D       DCT-I matrix of dimension \code{p}.
#' @param Te      number of bins for data binning.
#' @param m       approximate number of observations per bin. 
#' 
#' @return \code{y_star} \code{2Te-2} dimensional vector with 
#'              binned, variance stabilized and mirrored data. The bin number 
#'              \code{Te} may be modified to guarantee at least two data points 
#'              per bin. If \code{p/Te} is not an integer, the vector dimension 
#'              is \code{2*floor(p/round(p/Te))-2}.
#'         
#' @keywords internal
#' @noRd
data_transformation <- function(x, D, Te, m) {
  w <- (D%*%x)^2
  w_star <- data_binning(w, Te)
  y_star <- VST(w_star, m)
  TTe <- length(y_star)
  y_star <- c(y_star[2:(TTe-1)], y_star[TTe:1])
  
  return(y_star)
}

#' Data Binning
#'
#' Each \code{p}-dimensional vector is divided into \code{Te} consecutive bins. 
#' The data points in each bin and across the \code{n} samples are summed.
#' 
#' @param w   \code{p} dimensional data vector.
#' @param Te  number of bins for data binning. \code{Te} should be smaller than 
#'            the vector length \code{p}.
#' 
#' @return \code{w.star}: \code{Te}-dimensional vector with at approxi-
#'              mately \code{m = round(p/Te)} summed data points for each bin.
#'
#' @keywords internal
#' @noRd
data_binning <- function(w, Te) {
  p <- length(w)
  if (Te > p) {
    stop("Te must be smaller or equal than p.")
  }
  start <- round(seq(1, p-floor(p/Te), length.out = Te))
  end <- c(start[-1]-1, p)
  w_star <- vapply(
    seq_along(start), 
    function(i) sum(w[start[i]:end[i]]), 
    numeric(1)
  )
  
  return(w_star)
}

#' Variance Stabilizing Transform
#'
#' Applies the variance stabilizing transform for gamma distribution to a 
#' vector. For details see (Brown et al., \emph{"Nonparametric regression in 
#' exponential families."}, The Annals of Statistics 38.4 (2010): 2005-2046).
#' 
#' @param w_star  vector with binned data points.
#' @param m       number of summed data points per bin.
#' 
#' @return vector with binned and variance stabilized data, i.e. 
#'          \code{log(./m)/sqrt(2)} is applied entrywise to \code{w_star}.
#' 
#' @keywords internal
#' @noRd
VST <- function(w_star, m) {
  y_star <- log(w_star/m)/sqrt(2)
  return(y_star)
}