################################################################################
# ------------------------- TRIGONOMETRIC FUNCTIONS -------------------------- #
################################################################################

#' Discrete Cosine Transform matrix
#'
#' Computes the Discrete Cosine Transform I (DCT-I) matrix.
#' 
#' @param N matrix dimension.
#' 
#' @return the \code{NxN}-dimensional DCT-I matrix.
#' 
#' @keywords internal
#' @noRd
DCT.matrix <- function(N) {
  seq <- seq(0,N-1)
  arg <- (pi*seq/(N-1))
  DCT <- sapply(seq, function(j){cos(arg*j)})
  fac <- matrix(1/sqrt(2),N,N)
  fac[2:(N-1), 2:(N-1)] <- 1
  fac[1,1]=fac[1,N]=fac[N,1]=fac[N,N]=1/2
  DCT <- DCT*fac*sqrt(2)/sqrt(N-1)
  return(DCT)
}

#' Auto-covariance to Spectral Density Function
#'
#' @param x vector with the auto-covariance function.
#' 
#' @return y  vector with the spectral density function.
#' 
#' @keywords internal
#' @noRd
acf2sdf <- function(x) {
  N <- length(x)
  y <- 2*dtt::dtt(x, type="dct", variant = 1)
  return(y)
}

#' Spectral Density to Auto-covariance Function
#'
#' @param x vector with the spectral density function.
#' 
#' @return y vector with the auto-covariance function.
#' 
#' @keywords internal
#' @noRd
sdf2acf <- function(x){
  N <- length(x)
  y <- dtt::dtt(x, type="dct", variant = 1)/(N-1)
  return(y)
}

#' Sinc Function
#'
#' @param x vector.
#' 
#' @return y sinc(x).
#' 
#' @keywords internal
#' @noRd
sinc <- function(x){
  y <- sin(x)/x
  y[is.na(y)] <- 1
  return(y)
}

#' Q_p-Polynomials
#'
#' @param pminus1 degree of polynomial.
#' @param x       \code{r}-dim. vector of evaluation points.
#' 
#' @return res \code{r}-dim. vector with Q polynomial of degree \code{pminus1}
#'             at \code{x}, for details see (Schwarz, Krivobokova; 2016).
#' 
#' @keywords Internal
#' @noRd
Q_pminus1 <- function(pminus1, x) {
  if (pminus1 < 0) {
    stop ("pminus1 must be non-negative")
  }
  res <- switch(
    pminus1+1, #+1 to get cases 0,1,... instead of 1,2,...
    
    # Q_(p-1) = Q0
    x^0,
    
    # Q_(p-1) = Q1
    1/2+cos(pi*x)^2/2,
    
    # Q_(p-1) = Q2
    1/3+2*cos(pi*x)^2/3,
    
    # Q_(p-1) = Q3
    5/24+3*cos(pi*x)^2/4+cos(pi*x)^4/24,
    
    # Q_(p-1) = Q4
    2/15+11*cos(pi*x)^2/15+2*cos(pi*x)^4/15,
    
    # Q_(p-1) = Q5
    (61+479*cos(pi*x)^2+179*cos(pi*x)^4+cos(pi*x)^6)/720,
    
    # Q_(p-1) = Q6
    17/315+4*cos(pi*x)^2/7+38*cos(pi*x)^4/105+4*cos(pi*x)^6/315,
    
    # Q_(p-1) = Q7
    (1385+19028*cos(pi*x)^2+18270*cos(pi*x)^4+1636*cos(pi*x)^6+
       cos(pi*x)^8)/40320,
    
    # Q_(p-1) = Q8
    (62+1072*cos(pi*x)^2+1452*cos(pi*x)^4+247*cos(pi*x)^6+2*cos(pi*x)^8)/2835,
    
    # Q_(p-1) = Q9
    (50521+1073517*cos(pi*x)^2+1949762*cos(pi*x)^4+540242*cos(pi*x)^6+
       14757*cos(pi*x)^8+cos(pi*x)^10)/3628800,
    
    # Q_(p-1) = Q10
    (1382+35396*cos(pi*x)^2+83021*cos(pi*x)^4+34096*cos(pi*x)^6+
       2026*cos(pi*x)^8+4*cos(pi*x)^10)/155925
  )
  
  return(res)
}