% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sim_contacts.R
\name{sim_contacts}
\alias{sim_contacts}
\title{Simulate contacts for an infectious disease outbreak}
\usage{
sim_contacts(
  contact_distribution = function(x) stats::dpois(x = x, lambda = 2),
  infectious_period = function(n) stats::rlnorm(n = n, meanlog = 2, sdlog = 0.5),
  prob_infection = 0.5,
  outbreak_start_date = as.Date("2023-01-01"),
  anonymise = FALSE,
  outbreak_size = c(10, 10000),
  population_age = c(1, 90),
  contact_tracing_status_probs = c(under_followup = 0.7, lost_to_followup = 0.2, unknown
    = 0.1),
  config = create_config()
)
}
\arguments{
\item{contact_distribution}{A \code{function} or an \verb{<epiparameter>} object to
generate the number of contacts per infection.

The function can be defined or anonymous. The function must have a single
argument in the form of an \code{integer} vector with elements representing the
number of contacts, and return a \code{numeric} vector where each element
corresponds to the probability of observing the number of contacts in the
vector passed to the function. The index of the \code{numeric} vector returned
is offset by one to the corresponding probability of observing the number
of contacts, i.e. the first element of the output vector is the probability
of observing zero contacts, the second element is the probability of
observing one contact, etc.

An \verb{<epiparameter>} can be provided. This will be converted into a
probability mass function internally.

The default is an anonymous function with a Poisson probability mass function
(\code{\link[=dpois]{dpois()}}) with a mean (\eqn{\lambda}) of 2 contacts per infection.}

\item{infectious_period}{A \code{function} or an \verb{<epiparameter>} object for the
infectious period. This defines the duration from becoming infectious to
no longer infectious. In the simulation, individuals are assumed to
become infectious immediately after being infected (the latency period is
assumed to be zero). The time intervals between an infected individual and
their contacts are assumed to be uniformly distributed within the
infectious period. Infectious periods must be strictly positive.

The function can be defined or anonymous. The function must return a vector
of randomly generated real numbers representing sampled infectious periods.
The function must have a single argument, the number of random infectious
periods to generate.

An \verb{<epiparameter>} can be provided. This will be converted into random
number generator internally.

The default is an anonymous function with a lognormal distribution random
number generator (\code{\link[=rlnorm]{rlnorm()}}) with \code{meanlog = 2} and \code{sdlog = 0.5}.}

\item{prob_infection}{A single \code{numeric} for the probability of a secondary
contact being infected by an infected primary contact.}

\item{outbreak_start_date}{A \code{date} for the start of the outbreak.}

\item{anonymise}{A \code{logical} boolean for whether case names should be
anonymised. Default is \code{FALSE}.}

\item{outbreak_size}{A \code{numeric} vector of length 2 defining the minimum and
the maximum number of infected individuals for the simulated outbreak.
Default is \code{c(10, 1e4)}, so the minimum outbreak size is 10 infected
individuals, and the maximum outbreak size is 10,000 infected individuals.
Either number can be changed to increase or decrease the maximum or minimum
outbreak size to allow simulating larger or smaller outbreaks. If the
minimum outbreak size cannot be reached after running the simulation for
many iterations (internally) then the function errors, whereas if the
maximum outbreak size is exceeded the function returns the data early and a
warning stating how many cases and contacts are returned.}

\item{population_age}{Either a \code{numeric} vector with two elements or a
\verb{<data.frame>} with age structure in the population. Use a \code{numeric} vector
to specific the age range of the population, the first element is the lower
bound for the age range, and and the second is the upper bound for the age
range (both inclusive, i.e. [lower, upper]). The \verb{<data.frame>} with
age groups and the proportion of the population in that group. See details
and examples for more information.}

\item{contact_tracing_status_probs}{A named \code{numeric} vector with the
probability of each contact tracing status. The names of the vector must
be \code{"under_followup"}, \code{"lost_to_followup"}, \code{"unknown"}. Values of each
contact tracing status must sum to one.}

\item{config}{A list of settings to adjust the randomly sampled delays and
Ct values. See \code{\link[=create_config]{create_config()}} for more information.}
}
\value{
A contacts \verb{<data.frame>}.

The structure of the output is:

\describe{
\item{\code{from}}{\code{character} column with name of case.}
\item{\code{to}}{\code{character} column with name of contacts of case.}
\item{\code{age}}{\code{integer} with age of infectee.}
\item{\code{sex}}{\code{character} column with either \code{"m"} or \code{"f"} for the sex
of the contact.}
\item{\code{date_first_contact}}{\verb{<Date>} column for the first contact between
case and contacts.}
\item{\code{date_last_contact}}{\verb{<Date>} column for the last contact between
case and contacts.}
\item{\code{was_case}}{\code{logical} boolean column with either \code{TRUE} or \code{FALSE}
for if the contact becomes a case.}
\item{\code{status}}{\code{character} column with the status of each contact. By
default it is either \code{"case"}, \code{"under_followup"} \code{"lost_to_followup"}, or
\code{"unknown"}.}
}
}
\description{
Simulate contacts for an infectious disease outbreak
}
\examples{
# quickly simulate contact tracing data using the function defaults
contacts <- sim_contacts()
head(contacts)

# to simulate more realistic contact tracing data load epiparameters from
# {epiparameter}
library(epiparameter)
contact_distribution <- epiparameter(
  disease = "COVID-19",
  epi_name = "contact distribution",
  prob_distribution = create_prob_distribution(
    prob_distribution = "pois",
    prob_distribution_params = c(mean = 2)
  )
)

infectious_period <- epiparameter(
  disease = "COVID-19",
  epi_name = "infectious period",
  prob_distribution = create_prob_distribution(
    prob_distribution = "gamma",
    prob_distribution_params = c(shape = 1, scale = 1)
  )
)

contacts <- sim_contacts(
  contact_distribution = contact_distribution,
  infectious_period = infectious_period,
  prob_infection = 0.5
)
}
\author{
Joshua W. Lambert, Carmen Tamayo
}
