% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/subsample.R
\name{subsample}
\alias{subsample}
\title{Convert a daily record to discrete}
\usage{
subsample(
  dates,
  values,
  thresh_ref,
  season_start = 10,
  n_seasons = 4,
  half_win = 2,
  threshold = 0.8,
  n_samples = 1,
  freq = "month",
  n_et_samples = 8,
  season_weights = rep(1, n_seasons),
  target = "none",
  look_behind = 2,
  look_ahead = 2,
  look_units = "days",
  seed = 123
)
}
\arguments{
\item{dates}{A vector of dates of 'Date' class.}

\item{values}{Numeric values. The values must be in correspondence with \code{dates}, meaning the
\emph{i}th element in \code{values} must correspond to the \emph{i}th date in \code{dates}.}

\item{thresh_ref}{Numeric values to calculate a threshold. See 'Details'. The values must be in
correspondence with \code{dates}, meaning the \code{i}th element in \code{thresh_ref} must correspond to the
\code{i}th date in \code{dates}.}

\item{season_start}{The starting month of the first season, specified as an integer from 1 to
12. Default is 10 (October).}

\item{n_seasons}{The number of seasons in a year. Must be a factor of 12. Default is 4.}

\item{half_win}{The half width of the window of years to group \code{thresh_ref} by. See 'Details'.}

\item{threshold}{The quantile of \code{thresh_ref} above which \code{values} is sampled \code{n_et_samples}
times per year.}

\item{n_samples}{Integer of the number of below-threshold samples to be selected from \code{values}
at a frequency defined by \code{freq}.}

\item{freq}{Character of the frequency at which below-threshold samples are selected. May be
"week", "month", "quarter", or "year". See 'Details'.}

\item{n_et_samples}{Integer of the number of yearly exceeds-threshold samples to be selected
from \code{values}. See 'Details'.}

\item{season_weights}{A vector of integers of the weights to assign to seasons for random sampling of exceeds-threshold
\code{values}. Based on the rank of the seasonal average \code{thresh_ref} (from highest to lowest). Must
have length equal to the number of seasons (\code{n_seasons}). See 'Details'.}

\item{target}{One of "none" or "peaks". See 'Details'.}

\item{look_behind, look_ahead}{When \code{target} is "peaks", the number of \code{look_units} before and
after the center date to include in the sliding window to determine local maxima.}

\item{look_units}{One of "days", "weeks", or "months". The units to give \code{look_ahead} and
\code{look_behind}.}

\item{seed}{An integer which determines the state for the random number generator. Ensures
random sampling is reproducible.}
}
\value{
A data.frame with the following columns:
\tabular{lll}{
\strong{Name} \tab \strong{Type} \tab \strong{Description} \cr
date \tab Date \tab Date \cr
adj_year \tab integer \tab Adjusted year \cr
season \tab integer \tab Season number '1':n_seasons \cr
value \tab numeric \tab Input \code{values} \cr
thresh_ref \tab numeric \tab Input \code{thresh_ref} values \cr
threshold \tab numeric \tab Seasonal \code{threshold} quantile of \code{thresh_ref} \cr
is_peak \tab logical \tab TRUE when \code{thresh_ref} value is local maximum. Only when \code{target} is
"peaks". TRUE/FALSE \cr
selection_type \tab character \tab Type of randomly selected value "not_selected" (a record not
sampled), "below_threshold" (sampled record with value at or below threshold), or
"exceeds_threshold" (sampled record with value above threshold) \cr
weight \tab integer \tab Weight assigned to the value  \cr
ys_rank \tab integer \tab Unique year-season rank of the seasonal average \code{thresh_ref} \cr
}
}
\description{
Sample a daily or near-daily data set to one containing infrequent but periodic records based on
a random sampling protocol. Intended to be used on a (near) daily water quality record.
}
\details{
\code{values} are randomly selected based on a provided sampling protocol using \code{dates} as an index
and \code{thresh_ref} as a classifier. Elements in \code{values} equal to or less than their seasonal
threshold are randomly sampled according the protocol set by \code{n_samples} and \code{freq}.
\code{n_et_samples} elements in \code{values} greater than the threshold are randomly sampled for each
year in \code{values}. This results in \code{n_samples} of below-threshold values for each unique \code{freq}
and \code{n_et_samples} of exceeds-threshold values for each unique year.

Elements in \code{values} and \code{thresh_ref} must correspond with their respective values in \code{dates}.

\code{subsample()} is psuedo-random across time in that values are selected randomly in rolling
chunks of time determined by \code{freq}. If, for example, \code{freq} is "week" and \code{n_samples} is 1,
then the result will be 1 randomly selected below-threshold, non-NA value for each week.
However, the selected values could be very close in time (e.g., Saturday and Sunday).

Thresholds are calculated based on groupings of seasonally adjusted years, accounting for
seasons split across years. For example, if \code{n_seasons = 4} and \code{season_start = 12}, then
season 1 includes December of e.g. 2020, January 2021, and February 2021. The year is considered
to begin in December and is designated by the year in which it ends (i.e., the seasonally
adjusted year); 2021 in this example. If \code{half_win} is \code{2}, the default, then a total of 5 years
is used to calculate the threshold. For example, when \code{half_win} is 2, the threshold for season
1 of 2021 is the quantile defined by \code{threshold} of all season 1 values in 2019, 2020, 2021,
2022, and 2023.

The selection of exceeds-threshold values is always across an entire year with no
guarantee of timing between selected values. Setting \code{threshold} to values near 1 would result
in a smaller sample pool since there would conceivably be fewer values above 0.9 than 0.8 - thus
increasing the likelihood of selected exceeds-threshold values being "far" apart in time.

Both \code{n_samples} and \code{n_et_samples} are adjusted lower when the number of unique \code{dates} in
the defined \code{freq} is less than the number of unique dates in a complete \code{freq}. This adjustment
is calculated by multiplying the number of unique dates in the given \code{freq} and the number of
\verb{*_samples}, dividing the number of dates in the complete \code{freq}, and then rounding to the
nearest whole number. For example, when \code{n_samples} is 2 and \code{freq} is “week” but only 1 unique
sample date exists for a given week, then \code{n_samples} is adjusted to 1 ((1 * 2) / 7) --> 1).

\code{season_weights} influences the random sampling of exceeds-threshold values by weighting the
values according to the rank of the seasonal average of \code{thresh_ref} for the respective adjusted
year. For example, if \code{n_seasons} is 2 and \code{season_weights} is c(2, 1) then each season with the
highest seasonal average of \code{thresh_ref} values is given a weight of 2 and each season with the
lowest is given a weight of 1 - making the exceeds-threshold \code{values} occurring in the highest
ranking seasons more likely to be selected than if the weight was 1. See the details for the
\code{prob} argument in \code{\link[=sample]{sample()}} for more information.

When \code{target} is "none", the random selection of exceeds-threshold values is influenced only by
\code{season_weights}. When "peaks", the weights are doubled for values corresponding to local
maxima, exceeds-threshold \code{thresh_ref} values.
}
\examples{
# Randomly sample using defaults
df <- subsample(
  dates = streamdat$date, values = streamdat$sc, thresh_ref = streamdat$q, 
)
# Plotting function
create_plot <- function(df, log_xfrm = "x", xlab, ylab) {
  
  plot(
    df$thresh_ref[df$selection_type == "not_selected"], 
    df$value[df$selection_type == "not_selected"], 
    pch = 21, col = "gray", 
    xlab = xlab, ylab = ylab, 
    main = "Subsampled Daily Data", 
    log = log_xfrm
  )
  points(
    df$thresh_ref[df$selection_type != "not_selected"], 
    df$value[df$selection_type != "not_selected"], 
    pch = 16, cex = 1.5, 
    col = c(
      "below_threshold" = "#42047E", 
      "exceeds_threshold" = "#07A49E"
    )[df$selection_type[df$selection_type != "not_selected"]]
  )
  legend(
    "topright", 
    legend = c("Not Sampled", "Below Threshold", "Exceeds Threshold"), 
    col = c("gray", "#42047E", "#07A49E"), 
    pch = c(21, 16, 16), 
    bty = "n"
  )
}
# Plot
create_plot(df, "x", "Discharge (CFS)", "Specific Conductivity (uS/cm)")


df <- subsample( 
  dates = streamdat$date, values = streamdat$sc, thresh_ref = streamdat$q, 
  target = "peaks", 
  season_weights = c(2, 1, 1, 1) # default is four seasons
)
create_plot(df, "x", "Discharge (CFS)", "Specific Conductivity (uS/cm)")

df <- subsample( 
  dates = streamdat$date, values = streamdat$sc, thresh_ref = streamdat$sc, 
  target = "peaks", 
  n_samples = 1, freq = "week", 
  n_et_samples = 12, 
  half_win = 3
)
df <- merge(streamdat, df, by.x = c("date", "sc"), by.y = c("date", "value"))
df <- df[, !colnames(df) \%in\% c("thresh_ref")]
colnames(df)[c(2, 3)] <- c("value", "thresh_ref")
create_plot(df, "x", "Discharge (CFS)", "Specific Conductivity (uS/cm)")
}
