################################################################################
# Performs two one-sided t tests for mean equivalence using one-sample, and two
# sample tests
# Author: Alexis Dinno <alexis.dinno@pdx.edu>
# version 3.1.9
# Date: Feb 06, 2029

equivalence.types <- c("delta", "epsilon")

tost.ti <- function(
    n1         = NA, 
    mean1      = NA, 
    sd1        = NA, 
    mu         = NA, 
    n2         = NA, 
    mean2      = NA, 
    sd2        = NA, 
    eqv.type   = equivalence.types, 
    eqv.level  = 1, 
    upper      = NA, 
    var.equal  = FALSE, 
    welch      = FALSE,
    conf.level = 0.95, 
    x.name     = "",
    y.name     = "",
    relevance  = TRUE) {
  # Sanitize boolean options
  var.equal <- as.bool(var.equal)
  welch     <- as.bool(welch)
  relevance <- as.bool(relevance)
  # Calculate alpha
  alpha <- (1 - conf.level)
  # default eqv.type is "delta" if unspecified
  if (length(eqv.type)>1) {
    eqv.type <- "delta"
    }
  # standardize eqv.type option
  eqv.type = tolower(eqv.type)
  # Default upper threshold for symmetric equivalence interval
  if (is.na(upper)) {
    upper <- eqv.level
    }
  # Set up formatting for confidence interval column heading
  cl.width <- nchar(toString(100*conf.level))
  if (cl.width == 1) {
    conf.level.display <- paste0(pad.spaces(6), "[",toString(100*conf.level),"% Conf. Interval]", collapse="")
    }
  if (cl.width == 2) {
    conf.level.display <- paste0(pad.spaces(5), "[",toString(100*conf.level),"% Conf. Interval]", collapse="")
    }
  if (cl.width == 4) {
    conf.level.display <- paste0(pad.spaces(3),"[",toString(100*conf.level),"% Conf. Interval]", collapse="")
    }
  if (cl.width == 5) {
    conf.level.display <- paste0(pad.spaces(2),"[",toString(100*conf.level),"% Conf. Interval]", collapse="")
    }
  if (cl.width > 5) {
    conf.level.display <- paste0(pad.spaces(5),"[",toString(100*conf.level),"% CI]", collapse="")
    }
  # Output relevance test header if necessary.
  if (!is.na(mu)) {
    relevance_header <- "One-sample"
    if (y.name!="") {
      rlang::inform(message=paste0("\nIgnoring y.name"))
      }
    }
  if (is.na(mu) & (!is.na(n2) & !is.na(mean2) & !is.na(sd2))) {
    relevance_header <- "Unpaired"
    }
  # Obtaining variable names, and group names
  # Name for x variable
  if (x.name!="") {
    x.name <- pad.left(abbreviate(x.name, 8), 8)
    }
   else {
    x.name <- pad.left("x",8)
    }
  x.output.name <- x.name
  # Unpaired case name for y variable
  if (relevance_header=="Unpaired") {
    if (y.name!="") {
      y.name <- pad.left(abbreviate(y.name, 8), 8)
      y.output.name <- y.name
      }
     else {
      y.name <- pad.left("y",8)
      }
    y.output.name <- y.name
    }
  if (relevance) {
    rlang::inform(message=paste0("\n",relevance_header," relevance t test of means\n",sep=""))
    }
  
  # Create top bar, mid bar, and bottom bar
  top.bar     <- paste0(paste0(rep("\U2500",9),collapse=""),"\U252C",paste0(rep("\U2500",70),collapse=""), collapse="")
  table.title <- paste0("Variable \U2502",pad.spaces(5),"Obs",pad.spaces(8),"Mean",pad.spaces(4),"Std. err.",pad.spaces(3),"Std. dev.",conf.level.display, collapse="")
  mid.bar     <- paste0(paste0(rep("\U2500",9),collapse=""),"\U253C",paste0(rep("\U2500",70),collapse=""), collapse="")
  bottom.bar  <- paste0(paste0(rep("\U2500",9),collapse=""),"\U2534",paste0(rep("\U2500",70),collapse=""), collapse="")
  # For one-sample tests
  if (!is.na(mu)) {
    # Ignore settings for unpaired test
    # Notify ignoring equal variances option
    if (var.equal) {
      rlang::inform(message=paste0("Ignoring var.equal=TRUE option, not relevant in one-sample tests\n"))
      var.equal <- FALSE
      }
    # Notify ignoring welch option
    if (var.equal) {
      rlang::inform(message=paste0("Ignoring welch=TRUE option, not relevant in one-sample tests\n"))
      welch <- FALSE
      }
    # Get n, accounting for missing values of x
    n <- n1
    nu <- n-1
    df.explain <- paste0(n," - 1")
    if (n <10000000) {
      # Format n and nu for very large sample sizes
      n.display <- pad.left(toString(n),7)
      nu.display <- toString(nu)    
      }
     else {
      # Format n and nu for very large sample sizes
      n.display <- toString(signif(n,2))
      nu.display <- toString(signif(nu,2))
      } 
    # Calculate the various sample statistics for this test
    x.bar <- mean1
    estimate <- x.bar
    theta <- x.bar - mu
    sd.x <- sd1
    se.x <- sd.x/sqrt(n)
    # Calculate and format positivist test statistics
    t.crit <- qt(p=alpha/2,df=nu,lower.tail=FALSE)
    t.pos <- theta/se.x
    se.crit <- se.x
    p.pos <- 2*pt(q=abs(t.pos), df=nu, lower.tail=FALSE)
    t.pos.display <- sprintf("%.4f",signif(t.pos,6))
    p.pos.display <- format.extreme.p.vals(p.pos)
    if (relevance) {
      # Prepare some output strings for the positivist test if needed
      t.pos.pad <- pad.spaces(80 - 8 - nchar("\U03B8^ = mean(") - nchar(trimws(x.name)) - nchar(") - ") - nchar(mu) - nchar(" = ") - nchar(trimws(sprintf("%8.4g",x.bar - mu))) - nchar("t = ") - 9)
      df.pad <- pad.spaces(80 - 5 - nchar("Ho: \U03B8 = 0") - nchar("Degrees of freedom = (n - 1) = ") - 9)
      }
    # Calculate the confidence interval for x, and format for output
    ci.x.lb <- x.bar - t.crit*se.x
    ci.x.ub <- x.bar + t.crit*se.x
    ci.x.lb.display <- pad.left(toString(signif(ci.x.lb,7)),11)
    ci.x.ub.display <- pad.left(toString(signif(ci.x.ub,7)),11)
    eqv.level = abs(eqv.level)
    if (abs(x.bar) <1000000) {
      # Format statistics for output if 'small' numbers
      x.bar.display <- pad.left(toString(signif(x.bar,6)),11)
      theta.display <- pad.left(toString(signif(theta,6)),11)
      }
     else {
      # Format statistics for output if 'large' numbers
      x.bar.display <- pad.left(toString(signif(x.bar,7)),11)
      theta.display <- pad.left(toString(signif(theta,7)),11)
      }
    sd.x.display <- pad.left(toString(signif(sd.x,7)),11)
    se.x.display <- pad.left(toString(signif(se.x,7)),11)
    if (upper!=abs(eqv.level)) {
      # Calculate test statistics and prepare output for asymmetric equivalence intervals 
      eqv.level = -1*abs(eqv.level)
      if (eqv.type=="delta") {
        upper.diff <- upper - theta
        upper.diff.name <- paste0(pad.spaces(3), "\U0394u-\U03B8\U02C6", collapse="")
        lower.diff <- theta - eqv.level
        lower.diff.name <- paste0(pad.spaces(3), "\U03B8\U02C6-\U0394l", collapse="")
        delta.lower.display <- sprintf("%#-8.5g",(signif(-1*abs(eqv.level),5)))
        delta.upper.display <- sprintf("%#-8.5g",signif(abs(upper),5))
        t1 <- (upper - theta)/se.x
        t2 <- (theta - eqv.level)/se.x
        }
       else {
        epsilon.lower.display <- sprintf("%#-8.5g",(signif(-1*abs(eqv.level),5)))
        epsilon.upper.display <- sprintf("%#-8.5g",signif(abs(upper),5))
        t1 <- upper - t.pos
        t2 <- t.pos - eqv.level
        }
      }
    else {
      # Calculate test statistics and prepare output for symmetric equivalence intervals 
     if (eqv.type=="delta") {
       upper.diff <- abs(eqv.level) - theta
       upper.diff.name <- paste0(pad.spaces(4), "\U0394-\U03B8\U02C6", collapse="")
       lower.diff <- theta + abs(eqv.level)
       lower.diff.name <- paste0(pad.spaces(4), "\U03B8\U02C6+\U0394")
       delta.display <- sprintf("%#-8.5g",signif(abs(eqv.level),5))
       t1 <- (abs(eqv.level) - theta)/se.x
       t2 <- (theta + abs(eqv.level))/se.x
       }
      else {
       epsilon.display <- sprintf("%#-8.5g",(signif(abs(eqv.level),5)))
       t1 <- abs(eqv.level) - t.pos
       t2 <- t.pos + abs(eqv.level)
       }
     }
    # Calculate p values and prepare them and test statistics for output
    p1 <- pt(q=t1, df=nu, lower.tail=FALSE)
    p2 <- pt(q=t2, df=nu, lower.tail=FALSE)
    t1.display <- sprintf("%-8.4g",signif(t1,5))
    t2.display <- sprintf("%-8.4g",signif(t2,5))
    p1.display <- format.extreme.p.vals(p1)
    p2.display <- format.extreme.p.vals(p2)
    if (t1 <= -10) {
      t.pad <- pad.spaces(13)
      }
    if (t1 < 0) {
      t.pad <- pad.spaces(14)
      }
    if (t1 >= 0) {
      t.pad <- pad.spaces(15)
      }
    if (eqv.type=="delta") {
      ci.upper.diff.lb <- upper.diff - t.crit*se.x
      ci.upper.diff.ub <- upper.diff + t.crit*se.x
      ci.lower.diff.lb <- lower.diff - t.crit*se.x
      ci.lower.diff.ub <- lower.diff + t.crit*se.x
      if (abs(x.bar) <1000000) {
        lower.diff.display <- pad.left(toString(signif(lower.diff,7)),11)
        upper.diff.display <- pad.left(toString(signif(upper.diff,7)),11)
        }
       else {
        lower.diff.display <- pad.left(toString(signif(lower.diff,7)),11)
        upper.diff.display <- pad.left(toString(signif(upper.diff,7)),11)
        }
      ci.lower.diff.lb.display <- pad.left(toString(signif(ci.lower.diff.lb,7)),11)
      ci.lower.diff.ub.display <- pad.left(toString(signif(ci.lower.diff.ub,7)),11)
      ci.upper.diff.lb.display <- pad.left(toString(signif(ci.upper.diff.lb,7)),11)
      ci.upper.diff.ub.display <- pad.left(toString(signif(ci.upper.diff.ub,7)),11)
      }
    if (relevance) {
      rlang::inform(message=paste0("One-sample t test for mean difference"))
      rlang::inform(message=top.bar)
      rlang::inform(message=table.title)
      rlang::inform(message=mid.bar)
      rlang::inform(message=paste0(paste0(x.name," \U2502 ",n.display," ",x.bar.display," ",se.x.display," ",sd.x.display,pad.spaces(3),ci.x.lb.display," ",ci.x.ub.display)))
      rlang::inform(message=bottom.bar)
      rlang::inform(message=paste0(paste0(pad.spaces(8), "\U03B8^ = mean(",trimws(x.name),") - ",mu," = ",trimws(sprintf("%8.4g",x.bar - mu)),t.pos.pad,"t = ",pad.left(t.pos.display,9))))
      rlang::inform(message=paste0(paste0("     Ho: \U03B8 = 0",df.pad,"Degrees of freedom = (n - 1) = ",pad.left(nu.display,9))))
      rlang::inform(message=paste0("\n",pad.spaces(35),"Ha: \U03B8 \U2260 0", collapse=""))
      rlang::inform(message=paste0(pad.spaces(28), "Pr(|T| > |t|) ",p.pos.display,"\n"), sep="")
      }
    rlang::inform(message=paste0("\nOne-sample t test for mean equivalence"))
    rlang::inform(message=top.bar)
    rlang::inform(message=table.title)
    rlang::inform(message=mid.bar)
    rlang::inform(message=paste0(paste0(x.name," \U2502 ",n.display," ",x.bar.display," ",se.x.display," ",sd.x.display,pad.spaces(3),ci.x.lb.display," ",ci.x.ub.display)))
    rlang::inform(message=mid.bar)
    if (eqv.type=="delta") {
      rlang::inform(message=paste0(paste0(upper.diff.name," \U2502",pad.spaces(9),upper.diff.display," ",se.x.display,pad.spaces(15),ci.upper.diff.lb.display," ",ci.upper.diff.lb.display)))
      rlang::inform(message=paste0(paste0(lower.diff.name," \U2502",pad.spaces(9),lower.diff.display," ",se.x.display,pad.spaces(15),ci.lower.diff.lb.display," ",ci.lower.diff.lb.display)))
      rlang::inform(message=bottom.bar)
      }
    rlang::inform(message=paste0(paste0(pad.spaces(8), "\U03B8\U02C6 = mean(",trimws(x.name),") - ",trimws(mu)," = ",trimws(sprintf("%8.4g",x.bar - mu)))))
    }

  # For two-sample unpaired tests
  if (is.na(mu) & (!is.na(n2) & !is.na(mean2) & !is.na(sd2))) {
    if (var.equal & welch) {
      rlang::inform(message=paste0("Ignoring welch option, not relevant when var.equal=TRUE\n"))
      }
    nx        <- n1
    ny        <- n2
    n         <- nx + ny
    x.bar     <- mean1
    var.x     <- sd1^2
    sd.x      <- sd1
    se.x      <- sd.x/sqrt(nx)
    y.bar     <- mean2
    var.y     <- sd2^2
    sd.y      <- sd2
    se.y      <- sd.y/sqrt(ny)
    c.bar     <- (x.bar*nx + y.bar*ny)/n
    var.c     <- (((n1-1)*var.x) + (n1*(c.bar - x.bar)^2) + ((n2-1)*var.y) + (n2*(c.bar - y.bar)^2))/(n-1)
    sd.c      <- sqrt(var.c)
    se.c      <- sd.c/sqrt(n)
    d.bar     <- x.bar - y.bar
    estimate  <- d.bar
    sd.d.bar  <- sqrt((var.x/nx)+(var.y/ny))
    # Calculate degrees of freedom, depending on whether equal variances are 
    # assumed, and whether Welch's degrees of freedom have been opted for.
    #
    # Satterthwaite's formula for degrees of freedom here!
    if (!var.equal) {
      nu <- (((var.x/nx) + (var.y/ny))^2)/((((var.x/nx)^2)/(nx-1)) + (((var.y/ny)^2)/(ny-1)))
      unpaired.option <- "with unequal variances"
      df.display      <- "Satterthwaite\U2019s degrees of freedom = " 
      df.explain      <- "Satterthwaite\U2019s formula"
      }
    # Welch's formula for degrees of freedom here!
    if (!var.equal & welch) {
      nu <- -2 + (((var.x/nx) + (var.y/ny))^2)/((((var.x/nx)^2)/(nx+1)) + (((var.y/ny)^2)/(ny+1)))
      unpaired.option <- "with unequal variances"
      df.display      <- "Welch\U2019s degrees of freedom = " 
      df.explain      <- "Welch\U2019s formula"
      }
    # Student's equal variances formula for degrees of freedom here!
    if (var.equal) {
      nu <- nx + ny - 2
      unpaired.option <- "with equal variances"
      df.display      <- "Degrees of freedom = (n - 1) = " 
      df.explain      <- paste0(nx," + ",ny," - 1")
      }
    # Calculate and format positivist test statistics
    t.crit.x    <- qt(p=(alpha/2),df=(nx-1),lower.tail=FALSE)
    t.crit.y    <- qt(p=(alpha/2),df=(ny-1),lower.tail=FALSE)
    t.crit.c    <- qt(p=(alpha/2),df=(n-1),lower.tail=FALSE)
    t.crit.d    <- qt(p=(alpha/2),df=nu,lower.tail=FALSE)
    t.pos       <- d.bar/sd.d.bar
    se.crit     <- sd.d.bar
    p.pos <- 2*pt(q=abs(t.pos), df=nu, lower.tail=FALSE)
    # Calculate the confidence interval for x, and format for output
    ci.x.lb <- x.bar - t.crit.x*se.x
    ci.x.ub <- x.bar + t.crit.x*se.x
    ci.y.lb <- y.bar - t.crit.y*se.y
    ci.y.ub <- y.bar + t.crit.y*se.y
    ci.c.lb <- c.bar - t.crit.c*se.c
    ci.c.ub <- c.bar + t.crit.c*se.c
    ci.d.lb <- d.bar - t.crit.d*sd.d.bar
    ci.d.ub <- d.bar + t.crit.d*sd.d.bar
    # Format sample statistics for display
    # Format nu and sample sizes for display
    if (nx <10000000 & ny <10000000) {
      if (var.equal) nu.display <- toString(nu)    
       else nu.display <- sprintf("%.4f",nu)
      }
     else {
      if (var.equal) nu.display <- toString(signif(nu,2))
       else nu.display <- sprintf("%.4f",signif(nu,2))
      } 
    if (nx <10000000) {
      nx.display <- pad.left(toString(nx),7)
      }
     else {
      nx.display <- toString(signif(nx,2))
      } 
    if (ny <10000000) {
      ny.display <- pad.left(toString(ny),7)
      }
     else {
      ny.display <- toString(signif(ny,2))
      } 
    if ((n) <10000000) {
      n.display <- pad.left(toString(n),7)
      }
     else {
      n.display <- toString(signif(n,2))
      } 
    if (abs(x.bar) <1000000) {
      # Format x.bar for output if 'small' numbers
      x.bar.display <- pad.left(toString(signif(x.bar,6)),11)
      }
     else {
      # Format x.bar for output if 'large' numbers
      x.bar.display <- pad.left(toString(signif(x.bar,7)),11)
      }
    if (abs(y.bar) <1000000) {
      # Format y.bar for output if 'small' numbers
      y.bar.display <- pad.left(toString(signif(y.bar,6)),11)
      }
     else {
      # Format y.bar for output if 'large' numbers
      y.bar.display <- pad.left(toString(signif(y.bar,7)),11)
      }
    if (abs(c.bar) <1000000) {
      # Format c.bar for output if 'small' numbers
      c.bar.display <- pad.left(toString(signif(c.bar,6)),11)
      }
     else {
      # Format c.bar for output if 'large' numbers
      c.bar.display <- pad.left(toString(signif(c.bar,7)),11)
      }
    if (abs(d.bar) <1000000) {
      # Format d.bar for output if 'small' numbers
      d.bar.display <- pad.left(toString(signif(d.bar,6)),11)
      }
     else {
      # Format d.bar for output if 'large' numbers
      d.bar.display <- pad.left(toString(signif(d.bar,7)),11)
      }
    sd.x.display <- pad.left(toString(signif(sd.x,7)),11)
    se.x.display <- pad.left(toString(signif(se.x,7)),11)
    sd.y.display <- pad.left(toString(signif(sd.y,7)),11)
    se.y.display <- pad.left(toString(signif(se.y,7)),11)
    sd.c.display <- pad.left(toString(signif(sd.c,7)),11)
    se.c.display <- pad.left(toString(signif(se.c,7)),11)
    sd.d.display <- pad.left(toString(signif(sd.d.bar,7)),11)
    se.d.display <- pad.left(toString(signif(sd.d.bar,7)),11)
    t.pos.display <- sprintf("%.4f",signif(t.pos,6))
    p.pos.display <- format.extreme.p.vals(p.pos)
    ci.x.lb.display <- pad.left(toString(signif(ci.x.lb,7)),11)
    ci.x.ub.display <- pad.left(toString(signif(ci.x.ub,7)),11)
    ci.y.lb.display <- pad.left(toString(signif(ci.y.lb,7)),11)
    ci.y.ub.display <- pad.left(toString(signif(ci.y.ub,7)),11)
    ci.c.lb.display <- pad.left(toString(signif(ci.c.lb,7)),11)
    ci.c.ub.display <- pad.left(toString(signif(ci.c.ub,7)),11)
    ci.d.lb.display <- pad.left(toString(signif(ci.d.lb,7)),11)
    ci.d.ub.display <- pad.left(toString(signif(ci.d.ub,7)),11)
    # Prepare equivalence notations and quantities for display
    if (upper!=abs(eqv.level)) {
      # Calculate test statistics and prepare output for asymmetric equivalence intervals 
      eqv.level = -1*abs(eqv.level)
      if (eqv.type=="delta") {
        upper.diff <- upper - d.bar
        upper.diff.name <- paste0(pad.spaces(3), "\U0394u-\U03B8\U02C6", collapse="")
        lower.diff <- d.bar - eqv.level
        lower.diff.name <- paste0(pad.spaces(3), "\U03B8\U02C6-\U0394l", collapse="")
        delta.lower.display <- sprintf("%#-8.5g",(signif(-1*abs(eqv.level),5)))
        delta.upper.display <- sprintf("%#-8.5g",signif(abs(upper),5))
        t1 <- (upper - d.bar)/sd.d.bar
        t2 <- (d.bar - eqv.level)/sd.d.bar
        }
       else {
        epsilon.lower.display <- sprintf("%#-8.5g",(signif(-1*abs(eqv.level),5)))
        epsilon.upper.display <- sprintf("%#-8.5g",signif(abs(upper),5))
        t1 <- upper - t.pos
        t2 <- t.pos - eqv.level
        }
      }
    else {
      # Calculate test statistics and prepare output for symmetric equivalence intervals 
     if (eqv.type=="delta") {
       upper.diff <- abs(eqv.level) - d.bar
       upper.diff.name <- paste0(pad.spaces(4), "\U0394-\U03B8\U02C6", collapse="")
       lower.diff <- d.bar + abs(eqv.level)
       lower.diff.name <- paste0(pad.spaces(4), "\U03B8\U02C6+\U0394", collapse="")
       delta.display <- sprintf("%#-8.5g",signif(abs(eqv.level),5))
       t1 <- (abs(eqv.level) - d.bar)/sd.d.bar
       t2 <- (d.bar + abs(eqv.level))/sd.d.bar
       }
      else {
       epsilon.display <- sprintf("%#-8.5g",(signif(abs(eqv.level),5)))
       t1 <- abs(eqv.level) - t.pos
       t2 <- t.pos + abs(eqv.level)
       }
     }
    # Calculate p values and prepare them and test statistics for output
    p1 <- pt(q=t1, df=nu, lower.tail=FALSE)
    p2 <- pt(q=t2, df=nu, lower.tail=FALSE)
    t1.display <- sprintf("%-8.4g",signif(t1,5))
    t2.display <- sprintf("%-8.4g",signif(t2,5))
    p1.display <- format.extreme.p.vals(p1)
    p2.display <- format.extreme.p.vals(p2)
    if (t1 <= -10) {
      t.pad <- pad.spaces(13)
      }
    if (t1 < 0) {
      t.pad <- pad.spaces(14)
      }
    if (t1 >= 0) {
      t.pad <- pad.spaces(15)
      }
    if (eqv.type=="delta") {
      ci.upper.diff.lb <- upper.diff - t.crit.d*sd.d.bar
      ci.upper.diff.ub <- upper.diff + t.crit.d*sd.d.bar
      ci.lower.diff.lb <- lower.diff - t.crit.d*sd.d.bar
      ci.lower.diff.ub <- lower.diff + t.crit.d*sd.d.bar
      if (abs(x.bar) <1000000) {
        lower.diff.display <- pad.left(toString(signif(lower.diff,7)),11)
        upper.diff.display <- pad.left(toString(signif(upper.diff,7)),11)
        }
       else {
        lower.diff.display <- pad.left(toString(signif(lower.diff,7)),11)
        upper.diff.display <- pad.left(toString(signif(upper.diff,7)),11)
        }
      ci.lower.diff.lb.display <- pad.left(toString(signif(ci.lower.diff.lb,7)),11)
      ci.lower.diff.ub.display <- pad.left(toString(signif(ci.lower.diff.ub,7)),11)
      ci.upper.diff.lb.display <- pad.left(toString(signif(ci.upper.diff.lb,7)),11)
      ci.upper.diff.ub.display <- pad.left(toString(signif(ci.upper.diff.ub,7)),11)
      }

    # Output positivist test if relevance=TRUE
    if (relevance) {
      t.pos.pad <- pad.spaces(80 - 8 - nchar("\U03B8^ = mean(") - nchar(trimws(x.output.name)) - nchar(" - mean(") - nchar(trimws(y.output.name)) - nchar(") = ") - nchar(trimws(d.bar.display)) - nchar("t =") - nchar(t.pos.display))
      df.pad <- pad.spaces(80 - nchar("  H0: \U03B8^ = 0") - nchar(df.display) - 7)
      # Output difference test results
      rlang::inform(message=paste0("Two-sample unpaired t test for mean difference ", unpaired.option, sep=""))
      rlang::inform(message=top.bar)
      rlang::inform(message=table.title)
      rlang::inform(message=mid.bar)
      rlang::inform(message=paste0(paste0(x.name," \U2502 ",nx.display," ",x.bar.display," ",se.x.display," ",sd.x.display,pad.spaces(3),ci.x.lb.display," ",ci.x.ub.display)))
      rlang::inform(message=paste0(paste0(y.name," \U2502 ",ny.display," ",y.bar.display," ",se.y.display," ",sd.y.display,pad.spaces(3),ci.y.lb.display," ",ci.y.ub.display)))
      rlang::inform(message=mid.bar)
      rlang::inform(message=paste0(paste0("Combined \U2502 ",n.display," ",c.bar.display," ",se.c.display," ",sd.c.display,pad.spaces(3),ci.c.lb.display," ",ci.c.ub.display)))
      rlang::inform(message=mid.bar)
      rlang::inform(message=paste0(paste0(pad.spaces(6), "\U03B8^ \U2502",pad.spaces(9),d.bar.display," ",sd.d.display,pad.spaces(15),ci.d.lb.display," ",ci.d.ub.display)))
      rlang::inform(message=bottom.bar)
      rlang::inform(message=paste0(paste0(pad.spaces(6), "\U03B8^ = mean(",trimws(x.output.name),") - mean(",trimws(y.output.name),") = ",trimws(d.bar.display),t.pos.pad,"t = ",t.pos.display)))
      rlang::inform(message=paste0(paste0("  H0: \U03B8^ = 0",df.pad,df.display,pad.left(nu.display,7))))
      rlang::inform(message=paste0("\n",pad.spaces(35),"Ha: \U03B8 \U2260 0", collapse=""))
      rlang::inform(message=paste0(pad.spaces(28),"Pr(|T| > |t|) ",p.pos.display,"\n"),sep="")
      }
    # Output equivalence test results
    rlang::inform(message=paste0("\nTwo-sample unpaired t test for mean equivalence ", unpaired.option, sep=""))
    rlang::inform(message=top.bar)
    rlang::inform(message=table.title)
    rlang::inform(message=mid.bar)
    rlang::inform(message=paste0(paste0(x.name," \U2502 ",nx.display," ",x.bar.display," ",se.x.display," ",sd.x.display,pad.spaces(3),ci.x.lb.display," ",ci.x.ub.display)))
    rlang::inform(message=paste0(paste0(y.name," \U2502 ",ny.display," ",y.bar.display," ",se.y.display," ",sd.y.display,pad.spaces(3),ci.y.lb.display," ",ci.y.ub.display)))
    rlang::inform(message=mid.bar)
    if (eqv.type=="delta") {
      rlang::inform(message=paste0(paste0(upper.diff.name," \U2502",pad.spaces(9),upper.diff.display," ",sd.d.display,pad.spaces(15),ci.upper.diff.lb.display," ",ci.upper.diff.ub.display)))
      rlang::inform(message=paste0(paste0(lower.diff.name," \U2502",pad.spaces(9),lower.diff.display," ",sd.d.display,pad.spaces(15),ci.lower.diff.lb.display," ",ci.lower.diff.ub.display)))
      }
     else {
      rlang::inform(message=paste0(paste0(pad.spaces(6), "\U03B8^ \U2502 ",pad.spaces(7)," ",d.bar.display," ",se.d.display,pad.spaces(15),ci.d.lb.display," ",ci.d.ub.display)))
      }
    rlang::inform(message=bottom.bar)
    rlang::inform(message=paste0(paste0(pad.spaces(8), "\U03B8^ = mean(",trimws(x.output.name),") - mean(",trimws(y.output.name),") = ",trimws(d.bar.display))))
    }
  if (!is.na(mu)) {
    as.units.display <- (trimws(x.name))
    }
   else {
    as.units.display <- paste0(trimws(x.name)," and ",trimws(y.name))
    }
  if (upper!=abs(eqv.level)) {
    if (eqv.type=="delta") {
      rlang::inform(message=paste0(paste0(pad.spaces(8), "\U0394l = ",pad.right(delta.lower.display, 8)," \U0394l expressed in same units as ",as.units.display)))
      rlang::inform(message=paste0(paste0(pad.spaces(8), "\U0394u =  ",pad.right(delta.upper.display, 8),"\U0394u expressed in same units as ",as.units.display)))
      nu.pad <- max(nchar(trimws(delta.lower.display)),nchar(pad.right(delta.upper.display,8)))
      }
     else {
      rlang::inform(message=paste0(paste0(pad.spaces(8), "\U03B5l = ",pad.right(epsilon.lower.display,8)," \U03B5l expressed in units of the t distribution")))
      rlang::inform(message=paste0(paste0(pad.spaces(8), "\U03B5u =  ",pad.right(epsilon.upper.display,8),"\U03B5u expressed in units of the t distribution")))
      nu.pad <- max(nchar(pad.right(epsilon.lower.display,8)),nchar(pad.right(epsilon.upper.display,8)))
      }
    }
   else {
    if (eqv.type=="delta") {
      rlang::inform(message=paste0(paste0(pad.spaces(9), "\U0394 = ",pad.right(delta.display,8),pad.spaces(2), "\U0394 expressed in same units as ",as.units.display)))
      nu.pad <- nchar(pad.right(delta.display,8))
      }
     else {
      rlang::inform(message=paste0(paste0(pad.spaces(9), "\U03B5 = ",pad.right(epsilon.display,8),pad.spaces(2), "\U03B5 expressed in units of the t distribution")))
      nu.pad <- nchar(pad.right(epsilon.display,8))
      }
    }
  rlang::inform(message=paste0(paste0(pad.spaces(8), "df = ",pad.right(trimws(nu.display),8),pad.spaces(2),"using ",df.explain)))
    # Warn if equivalence threshold has 0 power/100% beta error
    if (eqv.type=="delta") {
      criticalvalue <- se.crit*qt(p=alpha, df=nu, lower.tail=FALSE)
      criticalvalue.display <- trimws(sprintf("%-6.4g",signif(criticalvalue,6)))
      if (upper==abs(eqv.level) & abs(eqv.level) <= criticalvalue) {
        rlang::inform(message=paste0("\n Impossible to reject any Ho if \U0394 \U2264 t-crit\U00D7S.E. (", criticalvalue.display, "). See help(tost.ti).",sep=""))
        }
      if (upper!=abs(eqv.level) & abs(eqv.level)<=criticalvalue) {
        rlang::inform(message=paste0("\n Impossible to reject any Ho if |\U0394l| \U2264 t-crit\U00D7S.E. (", criticalvalue.display, "). See help(tost.ti).",sep=""))
        }
      if (upper!=abs(eqv.level) & upper <= criticalvalue) {
        rlang::inform(message=paste0("\n Impossible to reject any Ho if \U0394u \U2264 t-crit\U00D7S.E. (", criticalvalue.display, "). See help(tost.ti).",sep=""))
        }
      }
     else {
      criticalvalue <- qt(p=alpha, df=nu, lower.tail=FALSE)
      criticalvalue.display <- trimws(sprintf("%-5.3f",signif(criticalvalue,5)))
      if (upper==abs(eqv.level) & abs(eqv.level) <= criticalvalue) {
        rlang::inform(message=paste0("\n Impossible to reject any Ho if \U03B5 \U2264 t-crit (", criticalvalue.display,"). See help(tost.ti).",sep=""))
        }
      if (upper!=abs(eqv.level) & abs(eqv.level) <= criticalvalue) {
        rlang::inform(message=paste0("\n Impossible to reject any Ho if |\U03B5l| \U2264 t-crit (", criticalvalue.display, "). See help(tost.ti).",sep=""))
        }
      if (upper!=abs(eqv.level) & upper <= criticalvalue) {
        rlang::inform(message=paste0("\n Impossible to reject any Ho if \U03B5u \U2264 t-crit (", criticalvalue.display, "). See help(tost.ti).",sep=""))
        }
      }
  rlang::inform(message="")
  if (upper!=abs(eqv.level)) {
    if (eqv.type=="delta") {
      rlang::inform(message=paste0("Ho: \U03B8 \U2264 \U0394l, or \U03B8 \U2265 \U0394u:"))
      }
     else {
      rlang::inform(message=paste0("Ho: T \U2264 \U03B5l, or T \U2265 \U03B5u:"))
      }
    }
   else {
    if (eqv.type=="delta") {
      rlang::inform(message=paste0("Ho: |\U03B8| \U2265 \U0394:"))
      }
     else {
      rlang::inform(message=paste0("Ho: |T| \U2265 \U03B5"))
      }
    }
  rlang::inform(message=paste0("\n",pad.spaces(8), "t1 = ", t1.display, t.pad, "t2 = ",t2.display, "\n", sep=""))
  if (upper!=abs(eqv.level)) {
    if (eqv.type=="delta") {
      rlang::inform(message=paste0(pad.spaces(3),"Ho1: \U0394u-\U03B8 \U2264 0", pad.spaces(15),"Ho2: \U03B8-\U0394l \U2264 0",sep=""))
      rlang::inform(message=paste0(pad.spaces(3),"Ha1: \U0394u-\U03B8 > 0", pad.spaces(15),"Ha2: \U03B8-\U0394l > 0",sep=""))
      }
     else {
      rlang::inform(message=paste0(pad.spaces(3),"Ho1: \U03B5u-T \U2264 0", pad.spaces(15),"Ho2: T-\U03B5l \U2264 0",sep=""))
      rlang::inform(message=paste0(pad.spaces(3),"Ha1: \U03B5u-T > 0", pad.spaces(15),"Ha2: T-\U03B5l > 0",sep=""))
      }
    }
   else {
    if (eqv.type=="delta") {
      rlang::inform(message=paste0(pad.spaces(3),"Ho1: \U0394-\U03B8 \U2264 0", pad.spaces(16),"Ho2: \U03B8+\U0394 \U2264 0",sep=""))
      rlang::inform(message=paste0(pad.spaces(3),"Ha1: \U0394-\U03B8 > 0", pad.spaces(16),"Ha2: \U03B8+\U0394 > 0",sep=""))
      }
     else {
      rlang::inform(message=paste0(pad.spaces(3),"Ho1: \U03B5-T \U2264 0", pad.spaces(16),"Ho2: T+\U03B5 \U2264 0",sep=""))
      rlang::inform(message=paste0(pad.spaces(3),"Ha1: \U03B5-T > 0", pad.spaces(16),"Ha2: T+\U03B5 > 0",sep=""))
      }
    }
  rlang::inform(message=paste0(pad.spaces(3),"Pr(T > t1) ",p1.display,pad.spaces(9),"Pr(T > t2) ",p2.display,"\n\n",sep=""))
  if (relevance) {
    if (upper!=abs(eqv.level)) {
      if (tolower(eqv.type) == "delta") {
        rlang::inform(message=paste0(paste0("Relevance test conclusion for \U03B1 = ",round(alpha,cl.width),", \U0394l = ", sub("\\.+$","",sub("0+$", "", delta.lower.display)),", and \U0394u = ", sub("\\.+$","",sub("0+$", "", delta.upper.display)))))
        }
       else {
        rlang::inform(message=paste0(paste0("Relevance test conclusion for \U03B1 = ",round(alpha,cl.width),", \U03B5l = ", sub("\\.+$","",sub("0+$", "", epsilon.lower.display)),", and \U03B5u = ", sub("\\.+$","",sub("0+$", "", epsilon.upper.display)))))
        }
      }
     else {
      if (tolower(eqv.type) == "delta") {
        rlang::inform(message=paste0(paste0("Relevance test conclusion for \U03B1 = ",round(alpha,cl.width)," and \U0394 = ", sub("\\.+$","",sub("0+$", "", delta.display)))))
        }
       else {
        rlang::inform(message=paste0(paste0("Relevance test conclusion for \U03B1 = ",round(alpha,cl.width)," and \U03B5 = ",sub("\\.+$","",sub("0+$", "", epsilon.display)))))
        }
      }
    if (p.pos <= alpha) {
      pos.decision <- "Reject"
      }
     else {
      pos.decision <- "Fail to reject"
      }
    if (p1 <= alpha & p2 <= alpha) {
      neg.decision <- "Reject"
      }
     else {
      neg.decision <- "Fail to reject"
      }
    rel.conclusion <- "Indeterminate (underpowered test)"
    if (pos.decision == "Reject" & neg.decision == "Fail to reject") {
      rel.conclusion <- "Relevant difference"
      }
     else {
      if (pos.decision == "Fail to reject" & neg.decision == "Reject") {
        rel.conclusion = "Equivalence"
        }
       else {
        if (pos.decision == "Reject" & neg.decision == "Reject") {
          rel.conclusion <- "Trivial difference (overpowered test)"
          }
        }
      }
    rlang::inform(message=paste0(paste0("  Ho test for difference:  ",pos.decision)))
    rlang::inform(message=paste0(paste0("  Ho test for equivalence: ",neg.decision,"\n")))
    rlang::inform(message=paste0(paste0("Conclusion from combined tests: ",rel.conclusion,"\n")))
    }

  out <- list() 
  # Prepare return stuff for one-sample test
  if (relevance_header=="One-sample") {
    if (!relevance) {
      out$statistics <- c(t1,t2)
      names(out$statistics) <- c("t1","t2")
      out$p.values <- c(p1,p2)
      names(out$p.values) <- c("p1","p2")
      }
    else {
      out$statistics <- c(t1,t2,t.pos)
      names(out$statistics) <- c("t1","t2","t")
      out$p.values <- c(p1,p2,p.pos)
      names(out$p.values) <- c("p1","p2","p")
     }
    out$estimate <- x.bar
    names(out$estimate) <- paste0("mean(",trimws(x.name),")",collapse="")
    out$null.value <- mu
    names(out$null.value) <- "\U3BC\U2080"
    out$sterr <- se.x
    names(out$sterr) <- paste0("se(",trimws(x.name),")",collapse="")
    out$sample_size <- n
    names(out$sample_size) <- "n"
    out$parameter <- nu
    if(exists("df.display")) {
      names(out$parameter) <- substring(df.display,1, nchar(df.display) - 3)
      }
     else {
      names(out$parameter) <- "degrees of freedom"
      }
    if (eqv.type=="delta") {
      if (upper==abs(eqv.level)) {
        out$threshold <- abs(eqv.level)
        names(out$threshold) <- "\U394"
        }
       else {
         out$threshold <- c(upper, eqv.level)
         names(out$threshold) <- c("\U0394u", "\U0394l")
        }
      }
     else {
      if (upper==abs(eqv.level)) {
        out$threshold <- abs(eqv.level)
        names(out$threshold) <- "\U3B5"
        }
       else {
         out$threshold <- c(upper, eqv.level)
         names(out$threshold) <- c("\U03B5u", "\U03B5l")
        }
      }
    if(relevance) {
      out$conclusion <- rel.conclusion
      names(out$conclusion) <- "relevance conclusion"
      }
    }

  # Prepare return stuff for two-sample test
  if (relevance_header=="Unpaired") {
   if (!relevance) {
      out$statistics <- c(t1,t2)
      names(out$statistics) <- c("t1","t2")
      out$p.values <- c(p1,p2)
      names(out$p.values) <- c("p1","p2")
      }
     else {
      out$statistics <- c(t1,t2,t.pos)
      names(out$statistics) <- c("t1","t2","t")
      out$p.values <- c(p1,p2,p.pos)
      names(out$p.values) <- c("p1","p2","p")
      }
    out$estimate <- c(x.bar,y.bar,d.bar)
    names(out$estimate) <- c(
       paste0("mean(",trimws(x.name),")", sep="", collapse=""),
       paste0("mean(",trimws(y.name),")", sep="", collapse=""),
       paste0("mean(",trimws(x.name),") - mean(",trimws(y.name),")", sep="", collapse=""))
    out$sterr <- sd.d.bar
    names(out$sterr) <- paste0("se(",trimws(x.name),"-",trimws(y.name),")",collapse="")
    out$sd <- c(sd.x, sd.y)
    names(out$sd) <- c(paste0("sd(",trimws(x.name),")",collapse=""),paste0("sd(",trimws(y.name),")",collapse=""))
    out$sample_size <- c(nx,ny,n)
    names(out$sample_size) <- c("nx","ny","nx+ny")
    out$parameter <- nu
    if(exists("df.display")) {
      names(out$parameter) <- substring(df.display,1, nchar(df.display) - 3)
      }
     else {
      names(out$parameter) <- "degrees of freedom"
      }
    if (eqv.type=="delta") {
      if (upper==abs(eqv.level)) {
        out$threshold <- abs(eqv.level)
        names(out$threshold) <- "\U394"
        }
       else {
         out$threshold <- c(upper, eqv.level)
         names(out$threshold) <- c("\U0394u", "\U0394l")
        }
      }
     else {
      if (upper==abs(eqv.level)) {
        out$threshold <- abs(eqv.level)
        names(out$threshold) <- "\U3B5"
        }
       else {
         out$threshold <- c(upper, eqv.level)
         names(out$threshold) <- c("\U03B5u", "\U03B5l")
        }
      }
    if(relevance) {
      out$conclusion <- rel.conclusion
      names(out$conclusion) <- "relevance conclusion"
      }
    }

  invisible(out)
  }
