% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tangent3d.R
\name{tangent3d}
\alias{tangent3d}
\title{Unit tangent vectors along a 3D parametric curve}
\usage{
tangent3d(
  X,
  Y,
  Z,
  a,
  b,
  t_points,
  h = 1e-04,
  plot = FALSE,
  n_samples = 400,
  vec_scale = NULL,
  vec_factor = 1,
  curve_line = list(color = "blue", width = 2, dash = "solid"),
  T_line = list(color = "red", width = 5, dash = "solid"),
  show_curve = TRUE,
  show_points = TRUE,
  point_marker = list(color = "black", size = 3, symbol = "circle"),
  scene = list(aspectmode = "data", xaxis = list(title = "x(t)"), yaxis = list(title =
    "y(t)"), zaxis = list(title = "z(t)")),
  bg = list(paper = "white", plot = "white"),
  tol = 1e-10
)
}
\arguments{
\item{X}{Function returning the \code{x} coordinate of the curve as a
function of the parameter \code{t}.}

\item{Y}{Function returning the \code{y} coordinate of the curve as a
function of the parameter \code{t}.}

\item{Z}{Function returning the \code{z} coordinate of the curve as a
function of the parameter \code{t}.}

\item{a}{Lower endpoint of the parameter interval.}

\item{b}{Upper endpoint of the parameter interval.}

\item{t_points}{Numeric vector of parameter values at which the tangent
direction is evaluated and optionally plotted.}

\item{h}{Step size for centered finite-difference approximations.}

\item{plot}{Logical; if \code{TRUE}, shows a 3D \pkg{plotly} visualization
of the curve and tangent segments.}

\item{n_samples}{Number of points used to sample and display the curve in
the 3D plot.}

\item{vec_scale}{Base length used for the tangent segments. If
\code{NULL}, it is estimated as a small fraction of the overall size of
the sampled curve.}

\item{vec_factor}{Multiplicative factor applied to \code{vec_scale} to
control the visual size of the tangent segments.}

\item{curve_line}{List with \pkg{plotly} style options for drawing the
base curve.}

\item{T_line}{List with \pkg{plotly} style options for the tangent
segments.}

\item{show_curve}{Logical; if \code{TRUE}, the base curve is included in
the plot.}

\item{show_points}{Logical; if \code{TRUE}, the evaluation points are
marked in the plot.}

\item{point_marker}{List with \pkg{plotly} marker options for the
evaluation points.}

\item{scene}{List with 3D scene settings for \pkg{plotly}.}

\item{bg}{Background colors for the figure, usually a list with entries
such as \code{paper} and \code{plot}.}

\item{tol}{Numeric tolerance for detecting situations in which the first
derivative is too small to define a stable tangent direction.}
}
\value{
A tibble with columns \code{t}, \code{x}, \code{y}, \code{z},
\code{Tx}, \code{Ty} and \code{Tz}, where the last three columns contain
the components of the unit tangent vector at each evaluation point.
}
\description{
Computes numerical unit tangent vectors of a three-dimensional
parametric curve at selected values of the parameter. The curve is
defined by three functions that give its coordinate components. For
each evaluation point, the first derivative of the curve is
approximated numerically and then normalized to obtain a unit tangent
direction.
}
\details{
For every element of \code{t_points}, the function:
\itemize{
\item computes a centered finite-difference approximation of the
first derivative of the curve,
\item evaluates the magnitude of that derivative,
\item divides the derivative by its magnitude to obtain a unit vector
pointing in the direction of motion of the curve at that point.
}

If the magnitude of the first derivative is extremely small at a given
parameter value, the tangent direction becomes numerically unstable; in
such cases, the function returns \code{NA} for the corresponding
components and may emit a diagnostic message.

Optionally, the curve and the associated tangent directions can be shown
in an interactive 3D plot using \pkg{plotly}. Short line segments
representing the tangent direction can be anchored at each evaluation
point. The sampled curve, the reference points and the tangent segments
may be displayed or hidden independently.
}
\examples{
X <- function(t) t*cos(t)
Y <- function(t) t*sin(3*t)
Z <- function(t) t
tangent3d(X, Y, Z, a = 0, b = 2*pi, t_points = c(pi/3, pi, 5*pi/3))

}
