





use crate::collector::{Collector, LocalHandle};
use crate::guard::Guard;
use crate::primitive::thread_local;
#[cfg(not(crossbeam_loom))]
use crate::sync::once_lock::OnceLock;

fn collector() -> &'static Collector {
    #[cfg(not(crossbeam_loom))]
    {
        
        static COLLECTOR: OnceLock<Collector> = OnceLock::new();
        COLLECTOR.get_or_init(Collector::new)
    }
    
    
    #[cfg(crossbeam_loom)]
    {
        loom::lazy_static! {
            
            static ref COLLECTOR: Collector = Collector::new();
        }
        &COLLECTOR
    }
}

thread_local! {
    
    static HANDLE: LocalHandle = collector().register();
}


#[inline]
pub fn pin() -> Guard {
    with_handle(|handle| handle.pin())
}


#[inline]
pub fn is_pinned() -> bool {
    with_handle(|handle| handle.is_pinned())
}


pub fn default_collector() -> &'static Collector {
    collector()
}

#[inline]
fn with_handle<F, R>(mut f: F) -> R
where
    F: FnMut(&LocalHandle) -> R,
{
    HANDLE
        .try_with(|h| f(h))
        .unwrap_or_else(|_| f(&collector().register()))
}

#[cfg(all(test, not(crossbeam_loom)))]
mod tests {
    use crossbeam_utils::thread;

    #[test]
    fn pin_while_exiting() {
        struct Foo;

        impl Drop for Foo {
            fn drop(&mut self) {
                
                super::pin();
            }
        }

        thread_local! {
            static FOO: Foo = const { Foo };
        }

        thread::scope(|scope| {
            scope.spawn(|_| {
                
                FOO.with(|_| ());
                super::pin();
                
            });
        })
        .unwrap();
    }
}
