% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/2_the_ZABCS_distributions.R
\name{ZABCS}
\alias{ZABCS}
\alias{dZABCS}
\alias{pZABCS}
\alias{qZABCS}
\alias{rZABCS}
\title{The Zero-Adjusted Box-Cox Symmetric Distributions}
\usage{
dZABCS(x, alpha, mu, sigma, lambda, zeta, family = "NO", log = FALSE)

pZABCS(
  q,
  alpha,
  mu,
  sigma,
  lambda,
  zeta,
  family = "NO",
  lower.tail = TRUE,
  log.p = FALSE
)

qZABCS(
  p,
  alpha,
  mu,
  sigma,
  lambda,
  zeta,
  family = "NO",
  lower.tail = TRUE,
  log.p = FALSE
)

rZABCS(n, alpha, mu, sigma, lambda, zeta, family = "NO")
}
\arguments{
\item{x, q}{vector of positive quantiles.}

\item{alpha}{vector of zero-adjusted parameters, with values on (0, 1).}

\item{mu}{vector of strictly positive scale parameters.}

\item{sigma}{vector of strictly positive relative dispersion parameters.}

\item{lambda}{vector of real-valued skewness parameters. If \code{lambda = 0}, the BCS
distribution reduces to the corresponding log-symmetric distribution with parameters
\code{mu} and \code{sigma} (and a possible extra parameter \code{zeta}).}

\item{zeta}{strictly positive extra parameter. It must be specified with only one value
in cases where the BCS distribution has an extra parameter. See “Details” below.}

\item{family}{a character that specifies the symmetric generating family of the BCS distribution.
Available options are: \code{"NO"} (default), \code{"ST"}, \code{"LOI"}, \code{"LOII"},
\code{"PE"}, \code{"SN"}, \code{"HP"}, and \code{"SL"}, corresponding to the normal,
Student-\emph{t}, type I logistic, type II logistic, power exponential, sinh-normal,
hyperbolic, and slash distributions, respectively.}

\item{log, log.p}{logical; if \code{TRUE}, probabilities \code{p} are given as \code{log(p)}.
Default is \code{FALSE}.}

\item{lower.tail}{logical; if \code{TRUE} (default), probabilities are
\eqn{P(X \le x)} otherwise, \eqn{P(X > x)}.}

\item{p}{vector of probabilities.}

\item{n}{number of observations. If `n` is a vector, its length is used as the number of
required observations.}
}
\value{
\code{dZABCS} returns the density function, \code{pZABCS} gives the cumulative distribution function,
\code{qZABCS} provides the quantile function, and \code{rZABCS} generates random variables.
}
\description{
Density function, distribution function, quantile function, and random generation
    for the class of the zero-adjusted Box-Cox symmetric (ZABCS) distributions.
}
\details{
The class of the ZABCS distributions was introduced by Medeiros and
    Queiroz (2025) as an extension of the Box-Cox symmetric (BCS) distributions
    (Ferrari and Fumes, 2017). The models consists of a broad class of probability
    distributions for positive continuous data which may include zeros.

    Let \eqn{Y} be a positive continuous random variable with a ZABCS distribution
    with parameters \eqn{\alpha \in (0, 1)}, \eqn{\mu > 0}, \eqn{\sigma > 0},
    and \eqn{\lambda \in \mathbb{R}} and density generating function \eqn{r}.
    The probability density function of
    \eqn{Y} is given by

    \eqn{
      f^{(0)}(y; \alpha, \mu, \sigma, \lambda)  = \left\{
        \begin{array}{rcl}
        \alpha, &  y=0,\\
        (1 - \alpha)f(y; \alpha, \mu, \sigma, \lambda), & y > 0,\\
        \end{array}
        \right.
    }

   where

    \eqn{
    f(y; \mu, \sigma, \lambda) = \left\{\begin{array}{ll}
      \dfrac{y^{\lambda-1}}{\mu^\lambda \sigma} \dfrac{r(z^2)}{R\left(\frac{1}{\sigma |\lambda|}\right)}, & \mbox{ if } \lambda \neq 0,\\
      \dfrac{1}{y\sigma} r(z^2), & \mbox{ if } \lambda = 0,
      \end{array}\right., \quad y > 0,
    }

    with

    \eqn{
    z = \left\{
    \begin{array}{ll}
    \dfrac{1}{\sigma \lambda} \left\{\left(\frac{y}{\mu}\right)^\lambda - 1 \right\}, & \mbox{ if } \lambda \neq 0, \\
    \dfrac{1}{\sigma} \log\left(\frac{y}{\mu}\right), & \mbox{ if } \lambda = 0,
    \end{array}
    \right.
    }

    \eqn{r:[0,\infty) \longrightarrow [0, \infty)}
    satisfies \eqn{\int_0^\infty u^{-1/2}r(u)\textrm{d} u = 1}, and
    \eqn{R(x) = \int_{-\infty}^x r(u^2)\textrm{d} u, x \in \mathbb{R}}.

    The function \eqn{r} is called density generating function, and it specifies the
    generating symmetric family of \eqn{Y} within the class of the ZABCS probability
    models. This function can also depend on extra parameters, such as the zero-adjusted
    Box-Cox \emph{t} and zero-adjusted Box-Cox power exponential distributions. We call
    these extra parameters \code{zeta}. The currently available ZABCS distributions in the
    \code{BCSreg} package are listed below:
    \tabular{llc}{
    \bold{Distribution}  \tab \bold{Family abbreviation} \tab \bold{N. of extra parameters}\cr
    Zero-adjusted Box-Cox Hyperbolic  \tab \code{"HP"}      \tab  1  \cr
    Zero-adjusted Box-Cox Type I Logistic  \tab \code{"LOI"}      \tab  0  \cr
    Zero-adjusted Box-Cox Type II Logistic  \tab \code{"LOII"}      \tab  0  \cr
    Zero-adjusted Box-Cox Normal  \tab \code{"NO"}      \tab  0  \cr
    Zero-adjusted Box-Cox Power Exponential  \tab \code{"PE"}      \tab  1  \cr
    Zero-adjusted Box-Cox Sinh-Normal  \tab \code{"SN"}      \tab  1  \cr
    Zero-adjusted Box-Cox Slash  \tab \code{"SL"}      \tab  1  \cr
    Zero-adjusted Box-Cox \emph{t}  \tab \code{"ST"}      \tab  1  \cr
    }
}
\examples{
# Probability density function

## Right-skewed distributions
curve(dZABCS(x, 0.4, 3, 0.3, -1.5, family = "NO"), from = 0.001, to = 7,
      xlim = c(0, 7), ylim = c(0, 0.5), ylab = "Density")
curve(dZABCS(x, 0.4, 3, 0.3, -1.5, 4, family = "ST"), add = TRUE, col = 2, from = 0.001)
curve(dZABCS(x, 0.4, 3, 0.3, -1.5, 5, family = "PE"), add = TRUE, col = 4, from = 0.001)
points(0, 0.4, type = "h", lty = 2)
points(0, 0.4, pch = 16, lty = 2)
legend("topright", legend = c("BCNO", "BCT", "BCPE"), lty = 1, col = c(1, 2, 4))

## Truncated symmetric distributions (with support on (0, Inf))
curve(dZABCS(x, 0.4, 3, 0.3, 1, family = "NO"), from = 0.001, to = 7,
      xlim = c(0, 7), ylim = c(0, 0.5), ylab = "Density")
curve(dZABCS(x, 0.4, 3, 0.3, 1, 4, family = "ST"), add = TRUE, col = 2, from = 0.001)
curve(dZABCS(x, 0.4, 3, 0.3, 1, 5, family = "PE"), add = TRUE, col = 4, from = 0.001)
points(0, 0.4, type = "h", lty = 2)
points(0, 0.4, pch = 16, lty = 2)
legend("topright", legend = c("BCNO", "BCT", "BCPE"), lty = 1, col = c(1, 2, 4))

## Left-skewed distributions
curve(dZABCS(x, 0.4, 3, 0.3, 3, family = "NO"), from = 0.001, to = 7,
      xlim = c(0, 7), ylim = c(0, 0.5), ylab = "Density")
curve(dZABCS(x, 0.4, 3, 0.3, 3, 4, family = "ST"), add = TRUE, col = 2, from = 0.001)
curve(dZABCS(x, 0.4, 3, 0.3, 3, 5, family = "PE"), add = TRUE, col = 4, from = 0.001)
points(0, 0.4, type = "h", lty = 2)
points(0, 0.4, pch = 16, lty = 2)
legend("topright", legend = c("BCNO", "BCT", "BCPE"), lty = 1, col = c(1, 2, 4))


# Random generation

## Parameter setting
alpha <- 0.2   # zero-adjustment parameter
mu <- 5        # scale parameter
sigma <- 0.2   # relative dispersion parameter
lambda <- -0.2 # skewness parameter

## Generating family
family <- "NO"

## Visualization
x <- rZABCS(10000, alpha, mu, sigma, lambda, family = family)

hist(x, prob = TRUE, col = "white", main = "")

points(0, mean(x == 0), type = "h", lty = 2)
points(0, mean(x == 0), pch = 16, lty = 2)
curve(dZABCS(x, alpha, mu, sigma, lambda, zeta, family = family), col = "blue", add = TRUE)

plot(ecdf(x), main = "")
curve(pZABCS(x, alpha, mu, sigma, lambda, zeta, family = family), col = "blue", add = TRUE)
}
\references{
Ferrari, S. L. P., and Fumes, G. (2017). Box-Cox symmetric distributions and
     applications to nutritional data. \emph{AStA Advances in Statistical Analysis}, 101, 321-344.

 Medeiros, R. M. R., and Queiroz, F. F. (2025). Flexible modeling of nonnegative continuous
     data: Box-Cox symmetric regression and its zero-adjusted extension.
}
\seealso{
\code{\link{BCS}} to access the density function, distribution
    function, quantile function, and a random number generator for the BCS
    distributions. \code{\link{BCSreg}} for estimating the parameters of a
    ZABCS regression model.
}
\author{
Francisco F. de Queiroz <\email{felipeq@ime.usp.br}>

Rodrigo M. R. de Medeiros <\email{rodrigo.matheus@ufrn.br}>
}
