#' The EFM function is to generate Elliptical Factor Models data.
#'
#' The function supports various distribution types for generating the data,
#' including: Elliptical-Normal Distribution, Elliptical-t Distribution.
#'
#' @name EFM
#'
#' @usage EFM(n, p, m, nu, distribution_type)
#' @param n Sample size.
#' @param p Sample dimensionality.
#' @param m Number of factors.
#' @param nu A numerical parameter used exclusively in the "Elliptical-t" distribution, representing the degrees of freedom.
#' @param distribution_type The type of distribution ("Elliptical-Normal Distribution" or "Elliptical-t Distribution").
#'
#' @return A list containing:
#' \item{data}{A matrix of generated data (n x p).}
#' \item{A}{A matrix representing the factor loadings (p x m).}
#' \item{D}{A diagonal matrix representing the unique variances (p x p).}
#'
#' @examples
#' library(MASS)
#' library(pracma)
#'
#' n <- 2000
#' p <- 10
#' m <- 5
#' nu <- 5
#' distribution_type <- "Elliptical-Normal Distribution"
#' X <- EFM(n, p, m, nu, distribution_type)
#'
#' @export
EFM <- function(n, p, m, nu, distribution_type) {
  # 均值矩阵 n x p
  mu <- matrix(runif(n * p, 0, 100), nrow = n, ncol = p)
  
  # 因子均值和协方差
  mu0 <- rep(0, m)        # m x 1
  sigma0 <- diag(m)       # m x m
  
  # 因子矩阵 n x m
  F <- MASS::mvrnorm(n, mu0, sigma0)
  
  # 因子载荷矩阵 p x m
  A <- matrix(runif(p * m, -1, 1), nrow = p, ncol = m)
  
  # 随机误差矩阵 n x p
  if (distribution_type == "Elliptical-Normal Distribution") {
    epsilon <- matrix(rnorm(n * p), nrow = n, ncol = p)
  } else if (distribution_type == "Elliptical-t Distribution") {
    epsilon <- matrix(stats::rt(n * p, df = nu), nrow = n, ncol = p)
  } else {
    stop("Unsupported distribution_type")
  }
  
  # 对角误差矩阵
  D <- diag(p)
  
  # 生成数据 n x p
  data <- mu + F %*% t(A) + epsilon
  
  return(list(data = data, A = A, D = D))
}
