% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Functions.R
\name{FluxPoint_raw}
\alias{FluxPoint_raw}
\title{Core change point detection algorithm (given known parameters)}
\usage{
FluxPoint_raw(data, Sig_all_inv, w, D, needReproduce = FALSE)
}
\arguments{
\item{data}{Numeric matrix of dimension \eqn{n \times p}, the multivariate
time series.}

\item{Sig_all_inv}{Numeric matrix of dimension \eqn{(p w) \times (p w)},
the inverse of the combined covariance \eqn{\Sigma_{\mathrm{ALL}}^*}
(accounts for random walk and VAR(1) effects within a window of size \eqn{w}).}

\item{w}{Integer; window size used in the moving-window detection step.}

\item{D}{Numeric; detection threshold used in the peak-finding step.}

\item{needReproduce}{Logical; if \code{TRUE}, a fixed fold assignment is
used in cross-validation to ensure reproducibility (default \code{FALSE}).}
}
\value{
A list with:
\itemize{
  \item `shiftIndices` — Binary matrix (\eqn{n \times p}) indicating selected
        components at each index.
  \item `detectorStats` — Numeric vector of detector values over time.
  \item `Projection_list` — Cache of projection matrices by active-set pattern.
  \item `cps` — Indices of detected change points.
}
}
\description{
Implements the core step of the proposed change point
detection (CPD) algorithm to estimate the locations of change points,
given the inverse windowed covariance \eqn{\Sigma_{\mathrm{ALL}}^{*-1}}.
The method computes detector statistics over a moving window using a
projection-based quadratic form and identifies candidate change points
via peak detection.
}
\details{
For each center index \eqn{k}, a window of width \eqn{w} is formed and
contrast vectors are constructed to compare the first and second halves of
the window. Before computing the detector statistic, a component-selection
step is performed using an \eqn{\ell_1}-penalized regression (lasso, via
\pkg{glmnet}) with weights \eqn{\Sigma_{\mathrm{ALL}}^{*-1}} to identify
variables that exhibit a shift. The resulting active set determines the
projection used in the statistic. Sparse projection matrices indexed by the
active-set pattern are cached and reused for computational efficiency. The
detector statistic is defined as a weighted quadratic form measuring
deviation from the baseline (no-change) projection, and locations at which
the statistic exceeds the threshold \code{D} are declared as estimated
change points.
}
\examples{
## Minimal runnable example (fast)
set.seed(123)
p <- 1
mu0 <- rep(0, p)
deltas <- list(c(3), c(4))
Sig_eta <- diag(0.01, p)
Sig_nu  <- random_Signu(p, 0)
Phi     <- random_Phi(p, 0)
Sig_e1  <- get_Sig_e1_approx(Sig_nu, Phi)

# Simulate data with two evenly spaced change points
Y <- generate_data(mu0, deltas, Sig_eta, Sig_nu, Phi, Sig_e1,
                   errortype = "n", number_cps = 2,
                   lengthofeachpart = 100)

# Windowed covariance and its inverse
w <- 20
Sigs <- get_Sigs(w, p, Sig_eta, Sig_nu, Phi, Sig_e1)
Sig_all_inv <- Sigs$Sig_all_inv

# Run detector with a common threshold choice
n <- nrow(Y)
D <- min(4, log(exp(2) + p)) * log(n - w)
res <- FluxPoint_raw(Y, Sig_all_inv, w, D)
res$cps

\donttest{
## More realistic example (may take longer)
set.seed(123)
p <- 3
mu0 <- rep(0, p)
deltas <- list(c(3, 0, -3), c(0, -2, 4))
Sig_eta <- diag(0.01, p)
Sig_nu  <- random_Signu(p, 0)
Phi     <- random_Phi(p, p)
Sig_e1  <- get_Sig_e1_approx(Sig_nu, Phi)

Y <- generate_data(mu0, deltas, Sig_eta, Sig_nu, Phi, Sig_e1,
                   errortype = "n", number_cps = 2,
                   lengthofeachpart = 100)

w <- 20
Sigs <- get_Sigs(w, p, Sig_eta, Sig_nu, Phi, Sig_e1)
Sig_all_inv <- Sigs$Sig_all_inv

n <- nrow(Y)
D <- min(4, log(exp(2) + p)) * log(n - w)
res <- FluxPoint_raw(Y, Sig_all_inv, w, D)
res$cps
}

}
