#' Correlation matrix from Cronbach's Alpha
#'
#' @name makeCorrAlpha
#'
#' @description `makeCorrAlpha()` generates a random correlation
#'  matrix of given dimensions and predefined _Cronbach's Alpha_.
#'
#' Such a correlation matrix can be applied to the [makeItems()]
#' function to generate synthetic data with the predefined alpha.
#'
#' @param items (positive, int) matrix dimensions:
#'  number of rows & columns to generate
#' @param alpha (real) target _Cronbach's Alpha_
#'  (usually positive, must be between about -0.3 and +1)
#' @param variance (positive, real) Default = 0.5.
#'  User-provided standard deviation of values sampled from a
#'  normally-distributed log transformation.
#'  Caution: Larger values increase chance of a non-positive-definite matrix.
#' 'TRUE' is faster, but produces less natural output. Default = FALSE
#' @param precision (positive, real) Default = 0.
#'  User-defined value ranging from '0' to '3' to add some random variation
#'  around the target _Cronbach's Alpha_.
#'  '0' gives an exact alpha (to two decimal places)
#' @param sort_cors (logical) If 'TRUE', sorts the correlation coefficients in
#'  the final correlation matrix.
#'  Similar to an earlier version of this function.
#' @param diagnostics (logical) If 'TRUE', returns a list containing the
#'  correlation matrix and a diagnostics list (target/achieved alpha,
#'  average inter-item correlation, eigenvalues, PD flag, and key arguments).
#'  If 'FALSE' (default), returns the correlation matrix only.
#'
#' @importFrom stats rnorm
#'
#' @return If 'diagnostics = FALSE', a k x k correlation matrix.
#'  If 'diagnostics = TRUE', a list with components:
#'  \describe{
#'    \item{R}{k x k correlation matrix}
#'    \item{diagnostics}{list of summary statistics}
#'  }
#'
#' @note
#' Random values generated by `makeCorrAlpha()` are highly volatile.
#'  `makeCorrAlpha()` may not generate a feasible (positive-definite)
#'  correlation matrix, especially when
#'
#'  * variance is high relative to
#'      * desired Alpha, and
#'      * desired correlation dimensions
#'
#'  `makeCorrAlpha()` will inform the user if the resulting correlation
#'  matrix is positive definite, or not.
#'
#'  If the returned correlation matrix is not positive-definite,
#'  a feasible solution may still be possible. The user is encouraged to
#'  try again, possibly several times, to find one.
#'
#' @examples
#'
#' # define parameters
#' items <- 4
#' alpha <- 0.85
#' variance <- 0.5
#'
#' # apply function
#' set.seed(42)
#' cor_matrix <- makeCorrAlpha(
#'   items = items,
#'   alpha = alpha,
#'   variance = variance
#' )
#'
#' # test function output
#' print(cor_matrix)
#' alpha(cor_matrix)
#' eigenvalues(cor_matrix, 1)
#'
#' # higher alpha, more items
#' cor_matrix2 <- makeCorrAlpha(
#'   items = 8,
#'   alpha = 0.95
#' )
#'
#' # test output
#' cor_matrix2 |> round(2)
#' alpha(cor_matrix2) |> round(3)
#' eigenvalues(cor_matrix2, 1) |> round(3)
#'
#'
#' # large random variation around alpha
#' set.seed(42)
#' cor_matrix3 <- makeCorrAlpha(
#'   items = 6,
#'   alpha = 0.85,
#'   precision = 2
#' )
#'
#' # test output
#' cor_matrix3 |> round(2)
#' alpha(cor_matrix3) |> round(3)
#' eigenvalues(cor_matrix3, 1) |> round(3)
#'
#'
#' # with diagnostics
#' cor_matrix4 <- makeCorrAlpha(
#'   items = 4,
#'   alpha = 0.80,
#'   diagnostics = TRUE
#' )
#'
#' # test output
#' cor_matrix4
#'
#' @export
makeCorrAlpha <- function(items,
                          alpha,
                          variance = 0.5,
                          precision = 0,
                          sort_cors = FALSE,
                          diagnostics = FALSE) {
  ####
  ###  Helper functions

  ###
  ### Fisher's z-transformation function
  log_transform <- function(x) {
    log((1 + x) / (1 - x))
  }

  ###
  ### exp_transform function
  exp_transform <- function(y) {
    (exp(y) - 1) / (exp(y) + 1)
  }

  ###
  ### make_corMatrix function
  make_corMatrix <- function(random_cors) {
    lower_matrix <- matrix(0, nrow = k, ncol = k)
    lower_matrix[lower.tri(lower_matrix)] <- random_cors
    upper_matrix <- t(lower_matrix)
    cor_matrix <- lower_matrix + upper_matrix
    diag(cor_matrix) <- 1
    cor_matrix
  }

  ###
  ### improve_cor_matrix Function
  improve_cor_matrix <- function() {
    n_cors <- length(random_cors)
    current_vector <- random_cors

    ## Pre-compute initial eigenvalues
    best_eigen_values <- eigen(cor_matrix)$values
    best_matrix <- cor_matrix
    min_best_eigen <- min(best_eigen_values)

    ## Limit swap attempts based on problem size
    max_swaps <- max(n_cors^3, 1024)

    ## Generate random swap pairs
    swap_attempts <- 0
    no_improvement_count <- 0
    max_no_improvement <- max_swaps / 2

    while (swap_attempts < max_swaps && min_best_eigen < 0) {
      ## Randomly select two positions to swap
      swap_pair <- sample.int(n_cors, 2)
      i <- swap_pair[1]
      j <- swap_pair[2]

      swap_attempts <- swap_attempts + 1

      ## Skip if values are same
      if (current_vector[i] == current_vector[j]) {
        next
      }

      ## Swap values
      temp_val <- current_vector[i]
      current_vector[i] <- current_vector[j]
      current_vector[j] <- temp_val

      ## Generate matrix and test
      temp_matrix <- make_corMatrix(current_vector)
      eigen_values <- eigen(temp_matrix)$values
      min_eigen <- min(eigen_values)

      ## Keep if improved
      if (min_eigen > min_best_eigen) {
        best_eigen_values <- eigen_values
        best_matrix <- temp_matrix
        min_best_eigen <- min_eigen
        no_improvement_count <- 0
        cat(paste0(
          "improved at swap - ", swap_attempts,
          " (min eigenvalue: ", round(min_eigen, 6), ")\n"
        ))

        ## Check if positive definite
        if (min_best_eigen >= 0) {
          cat(paste0("positive definite at swap - ", swap_attempts, "\n"))
          break
        }
      } else {
        ## Revert swap
        current_vector[j] <- current_vector[i]
        current_vector[i] <- temp_val
        no_improvement_count <- no_improvement_count + 1

        ## terminate if stuck
        if (no_improvement_count > max_no_improvement) {
          cat(paste0(
            "stopped after ", swap_attempts,
            " swaps (no improvement for ", max_no_improvement, " attempts)\n"
          ))
          break
        }
      }
    }

    best_matrix
  }
  ### end helper functions

  k <- items

  ## Clamp precision
  precision <- max(0, min(3, precision))

  ## Calculate the mean correlation coefficient from alpha
  target_mean_r <- alpha / (k - alpha * (k - 1))

  ## Add random variation to target mean correlation
  logr_sd_coefficient <- 2^(2^(4 / (precision + 1)))
  logr_sd <- 1 / logr_sd_coefficient

  log_transformed_r <- log_transform(target_mean_r + rnorm(1, 0, logr_sd))
  mean_r <- exp_transform(log_transformed_r)

  tolerance <- 1e-5
  n_cors <- k * (k - 1) / 2

  max_iterations <- min(10000, max(500, k^2 * 100))

  best_diff <- Inf
  random_cors <- NULL

  for (iteration in 1:max_iterations) {
    ## Generate random values
    random_values <- rnorm(n_cors, mean = log_transformed_r, sd = variance)
    temp_cors <- exp_transform(random_values)

    ## Calculate mean difference
    temp_mean <- mean(temp_cors)
    temp_diff <- abs(temp_mean - mean_r)

    ## Keep best so far
    if (temp_diff < best_diff) {
      best_diff <- temp_diff
      random_cors <- temp_cors

      ## Check if within tolerance
      if (best_diff < tolerance) {
        cat(paste0(
          "correlation values consistent with desired alpha in ",
          iteration, " iterations\n"
        ))
        break
      }
    }
  }

  if (iteration == max_iterations) {
    cat(paste0(
      "reached max iterations (", max_iterations,
      ") - best mean difference: ", round(best_diff, 6), "\n"
    ))
  }

  ## Optionally sort to help positive-definiteness (faster but unnatural output)
  if (sort_cors == TRUE) {
    random_cors <- sort(random_cors, decreasing = FALSE)
  }

  ## Create correlation matrix
  cor_matrix <- make_corMatrix(random_cors)

  ## Test for positive-definite
  eigen_values <- eigen(cor_matrix)$values
  is_positive_definite <- min(eigen_values) >= 0

  ## If not positive definite, try to improve
  if (!is_positive_definite) {
    cat("Correlation matrix is not yet positive definite\nWorking on it\n\n")
    cor_matrix <- improve_cor_matrix()
  }

  # Apply row/column names to a k x k correlation matrix R
  item_names <- sprintf("item%02d", seq_len(k))
  rownames(cor_matrix) <- colnames(cor_matrix) <- item_names

  # If diagnostics not requested, just return the matrix
  if (!diagnostics) {
    return(cor_matrix)
  } else {
    # Light-weight diagnostics computed from the final matrix
    # k <- nrow(R)
    avg_r <- mean(cor_matrix[lower.tri(cor_matrix)])
    alpha_achieved <- (k * avg_r) / (1 + (k - 1) * avg_r)

    eigvals <- eigen(cor_matrix, symmetric = TRUE, only.values = TRUE)$values
    is_pd <- all(eigvals > 1e-8)

    out <- list(
      R = cor_matrix,
      diagnostics = list(
        items                = items,
        alpha_target         = alpha,
        alpha_achieved       = alpha_achieved,
        average_r            = avg_r,
        eigenvalues          = eigvals,
        is_positive_definite = is_pd,
        variance             = variance,
        precision            = precision,
        sort_cors            = sort_cors
      )
    )

    return(out)
  }

  ## Report final status
  final_min_eigen <- min(eigen(cor_matrix)$values)
  if (final_min_eigen >= 0) {
    cat("The correlation matrix is positive definite\n")
    cat(paste0("Min eigenvalue: ", round(final_min_eigen, 6), "\n\n"))
  } else {
    cat("Correlation matrix is NOT positive definite\n")
    cat(paste0("Min eigenvalue: ", round(final_min_eigen, 6), "\n"))
    cat("Try running makeCorrAlpha again (or reduce Variance parameter)\n\n")
  }
}
