% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ob_utilities.R
\name{ob_gains_table}
\alias{ob_gains_table}
\title{Compute Comprehensive Gains Table from Binning Results}
\usage{
ob_gains_table(binning_result)
}
\arguments{
\item{binning_result}{A named \code{list} or \code{data.frame} containing the
following atomic vectors (all must have the same length):
\describe{
  \item{\code{id}}{Numeric vector of bin identifiers. Determines the sort order
    for cumulative metrics (e.g., KS, Recall).}
  \item{\code{bin}}{Character vector of bin labels/intervals.}
  \item{\code{count}}{Numeric vector of total observations per bin (\eqn{O_i}).}
  \item{\code{count_pos}}{Numeric vector of positive (event) counts per bin (\eqn{E_i}).}
  \item{\code{count_neg}}{Numeric vector of negative (non-event) counts per bin (\eqn{NE_i}).}
}}
}
\value{
A \code{data.frame} with the following columns (metrics calculated per bin):
  \describe{
    \item{\strong{Identifiers}}{
      \code{id}, \code{bin}
    }
    \item{\strong{Counts & Rates}}{
      \code{count}, \code{pos}, \code{neg},
      \code{pos_rate} (\eqn{\pi_i}), \code{neg_rate} (\eqn{1-\pi_i}),
      \code{count_perc} (\eqn{O_i / O_{total}})
    }
    \item{\strong{Distributions (Shares)}}{
      \code{pos_perc} (\eqn{D_1(i)}: Share of Bad),
      \code{neg_perc} (\eqn{D_0(i)}: Share of Good)
    }
    \item{\strong{Cumulative Statistics}}{
      \code{cum_pos}, \code{cum_neg},
      \code{cum_pos_perc} (\eqn{CDF_1}), \code{cum_neg_perc} (\eqn{CDF_0}),
      \code{cum_count_perc}
    }
    \item{\strong{Credit Scoring Metrics}}{
      \code{woe}, \code{iv}, \code{total_iv}, \code{ks}, \code{lift},
      \code{odds_pos}, \code{odds_ratio}
    }
    \item{\strong{Advanced Metrics}}{
      \code{gini_contribution}, \code{log_likelihood},
      \code{kl_divergence}, \code{js_divergence}
    }
    \item{\strong{Classification Metrics}}{
      \code{precision}, \code{recall}, \code{f1_score}
    }
  }
}
\description{
This function serves as a high-performance engine (implemented in C++) to calculate
a comprehensive set of credit scoring and classification metrics based on
pre-aggregated binning results. It takes a list of bin counts and computes
metrics such as Information Value (IV), Weight of Evidence (WoE), Kolmogorov-Smirnov (KS),
Gini, Lift, and various entropy-based divergence measures.
}
\details{
\subsection{Mathematical Definitions}{

Let \eqn{E_i} and \eqn{NE_i} be the number of events and non-events in bin \eqn{i},
and \eqn{E_{total}}, \eqn{NE_{total}} be the population totals.

\strong{Weight of Evidence (WoE) & Information Value (IV):}
\deqn{WoE_i = \ln\left(\frac{E_i / E_{total}}{NE_i / NE_{total}}\right)}
\deqn{IV_i = \left(\frac{E_i}{E_{total}} - \frac{NE_i}{NE_{total}}\right) \times WoE_i}

\strong{Kolmogorov-Smirnov (KS):}
\deqn{KS_i = \left| \sum_{j=1}^i \frac{E_j}{E_{total}} - \sum_{j=1}^i \frac{NE_j}{NE_{total}} \right|}

\strong{Lift:}
\deqn{Lift_i = \frac{E_i / (E_i + NE_i)}{E_{total} / (E_{total} + NE_{total})}}

\strong{Kullback-Leibler Divergence (Bernoulli):}
Measures the divergence between the bin's event rate \eqn{p_i} and the global event rate \eqn{p_{global}}:
\deqn{KL_i = p_i \ln\left(\frac{p_i}{p_{global}}\right) + (1-p_i) \ln\left(\frac{1-p_i}{1-p_{global}}\right)}
}
}
\examples{
# Manually constructed binning result
bin_res <- list(
  id = 1:3,
  bin = c("Low", "Medium", "High"),
  count = c(100, 200, 50),
  count_pos = c(5, 30, 20),
  count_neg = c(95, 170, 30)
)

gt <- ob_gains_table(bin_res)
print(gt[, c("bin", "woe", "iv", "ks")])

}
