% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/SII_function.R
\name{phe_sii}
\alias{phe_sii}
\title{Calculate Slope Index of Inequality using phe_sii}
\usage{
phe_sii(
  data,
  quantile,
  population,
  x = NULL,
  value = NULL,
  value_type = 0,
  transform = FALSE,
  lower_cl = NULL,
  upper_cl = NULL,
  se = NULL,
  multiplier = 1,
  repetitions = 1e+05,
  confidence = 0.95,
  rii = FALSE,
  intercept = FALSE,
  reliability_stat = FALSE,
  type = "full"
)
}
\arguments{
\item{data}{data.frame containing the required input fields, pre-grouped if
an SII is required for each subgroup; unquoted string; no default}

\item{quantile}{field name within data that contains the quantile label (e.g.
decile). The number of quantiles should be between 5 and 100; unquoted
string; no default}

\item{population}{field name within data that contains the quantile
populations (ie, denominator). Non-zero populations are required for all
quantiles to calculate SII for an area; unquoted string; no default}

\item{x}{(for indicators that are proportions) field name within data that
contains the members of the population with the attribute of interest (ie,
numerator). This will be divided by population to calculate a proportion as
the indicator value (if value field is not provided); unquoted string; no
default}

\item{value}{field name within data that contains the indicator value (this
does not need to be supplied for proportions if count and population are
given); unquoted string; no default}

\item{value_type}{indicates the indicator type (1 = rate, 2 = proportion, 0 =
other). The \code{value_type} argument is used to determine whether data should
be transformed prior to calculation of the standard error and/or SII. See
the \code{Tansformations} section for full details; integer; default 0}

\item{transform}{option to transform input rates or proportions prior to
calculation of the SII. See the \code{Transformations} section for full
details; logical; default FALSE}

\item{lower_cl}{field name within data that contains 95 percent lower
confidence limit of indicator value (to calculate standard error of
indicator value). This field is needed if the se field is not supplied;
unquoted string; no default}

\item{upper_cl}{field name within data that contains 95 percent upper
confidence limit of indicator value (to calculate standard error of
indicator value). This field is needed if the se field is not supplied;
unquoted string; no default}

\item{se}{field name within data that contains the standard error of the
indicator value. If not supplied, this will be calculated from the 95
percent lower and upper confidence limits (i.e. one or the other of these
fields must be supplied); unquoted string; no default}

\item{multiplier}{factor to multiply the SII and SII confidence limits by
(e.g. set to 100 to return prevalences on a percentage scale between 0 and
100). If the multiplier is negative, the inverse of the RII is taken to
account for the change in polarity; numeric; default 1;}

\item{repetitions}{number of random samples to perform to return confidence
interval of SII (and RII). Minimum is 1000, no maximum (though the more
repetitions, the longer the run time); numeric; default 100,000}

\item{confidence}{confidence level used to calculate the lower and upper
confidence limits of SII, expressed as a number between 0.9 and 1, or 90
and 100. It can be a vector of 0.95 and 0.998, for example, to output both
95 percent and 99.8 percent CIs; numeric; default 0.95}

\item{rii}{option to return the Relative Index of Inequality (RII) with
associated confidence limits as well as the SII; logical; default FALSE}

\item{intercept}{option to return the intercept value of the regression line
(y value where x=0); logical; default FALSE}

\item{reliability_stat}{option to carry out the SII confidence interval
simulation 10 times instead of once and return the Mean Average Difference
between the first and subsequent samples (as a measure of the amount of
variation). Warning: this will significantly increase run time of the
function and should first be tested on a small number of repetitions;
logical; default FALSE}

\item{type}{"full" output includes columns in the output dataset specifying
the parameters the user has input to the function (value_type, multiplier,
CI_confidence, CI_method); character string either "full" or "standard";
default "full"}
}
\value{
The SII with lower and upper confidence limits for each subgroup of
the inputted data.frame.
}
\description{
\code{phe_sii()}returns the slope index of inequality (SII) statistic for each
subgroup of the inputted dataframe, with lower and upper confidence limits
based on the specified confidence.
}
\details{
The Relative Index of Inequality (RII) can also be returned via an optional
argument.

The SII and RII are two measures of health inequality. They show the relation
between the level of health or frequency of a health problem in different
population groups and the ranking of these groups on the social scale.

The input dataframe should be grouped before passing to the function if an
SII/RII for each subgroup is required, and quantiles ordered from least to
most advantaged.
}
\section{Calculation}{


The SII is calculated using linear regression (1). To allow for differences
in population size between quantiles (e.g. deprivation deciles), each is
given a rank score (or relative rank) based on the midpoint of its range in
the cumulative distribution of the total area population. The quantiles are
first ordered (e.g from 1 most deprived to 10 least deprived for deprivation
deciles). If quantile 1 then contains 12 percent of the total population, its
relative rank is \code{0.12/2=0.6}. If quantile 2 includes 10 percent of the
population, its relative rank is \code{0.12+(0.10/2)=0.17}. A square root
transformation is applied to the regression to account for heteroskedasticity
(the tendancy for the variances of the quantile values to be related to the
size of the values, ie larger values will tend to have larger variances). A
regression model is fitted to the transformed data:  \eqn{Y * \sqrt a = \sqrt
a + b * \sqrt a}, where Y is the value of the indicator for the quantile, a
is the proportion of the total population in the quantile and b is the
relative rank. The SII is the gradient of the resulting fitted line, and
could be positive or negative according to the indicator polarity. Since the
relative ranks, by definition, range from 0 to 1, the SII is the difference
between the fitted value at  \code{x=1} and  \code{x=0}. The RII is the ratio
of the fitted value at  \code{x=1,Y1} and the fitted value at \code{x=0,Y0}.
which can be calculated as:  \code{RII = (Y0 + SII)/Y0}
}

\section{Transformations}{


The indicator type can be specified as 1 (rate), 2 (proportion) or 0 (other),
using the \code{value_type} parameter. This setting determines the data
transformations that will be applied in the following two parts of the
method.

Use in conjunction with the \code{transform} parameter in calculation of the
SII: It is recommended that rates and proportions are transformed prior to
calculation of the SII by setting the \code{transform} parameter to TRUE for
these indicator types. This will perform a log transformation for rates, or
logit for proportions, and return outputs transformed back to the original
units of the indicator. These transformations are recommended to improve the
linearity between the indicator values and the quantile, which is an
assumption of the method. A user-provided standard error will not be accepted
when the \code{transform} parameter is set to TRUE as the confidence limits
are required for this transformation.

Use in calculation of the standard error: Rates and proportions, and their
confidence limits, are transformed prior to calculation of the standard error
for each quantile. This is because it is assumed that the confidence interval
around the indicator value is non-symmetric for these indicator types. Note
that this transformation is not controlled by the \code{transform} parameter
and is applied based on the value of the \code{value_type} parameter only. A
user-provided standard error will not be accepted when the \code{transform}
parameter is set to TRUE as the confidence limits are required for this
transformation.
}

\section{Warning}{


The SII calculation assumes a linear relationship between indicator value and
quantile. Where this is not the case the transform option should be considered.
Small populations within quantiles can make the SII unstable. This
function does not include checks for linearity or stability; it is the user's
responsibility to ensure the input data is suitable for the SII calculation.
}

\examples{
library(dplyr)

data <- data.frame(area = c(rep("Area1", 10), rep("Area2", 10)),
                   decile = c(1:10, 1:10),
                   population = c(7291, 7997, 6105, 7666, 5790, 6934, 5918, 5974, 7147, 7534, 21675,
                                  20065, 19750, 24713, 20112, 19618, 22408, 19752, 18939, 19312),
                   value = c(75.9, 78.3, 83.8, 83.6, 80.5, 81.1, 81.7, 84.2, 80.6, 86.3, 70.5,
                              71.6, 72.5, 73.5, 73.1, 76.2, 78.7, 80.6, 80.9, 80),
                   lowerCL = c(72.7,75.3,80.9,80.2,77.1,78,79,81.4,75.8,83.2,
                               70.1,71.1,72,73.1, 72.7, 75.7, 78.2,80.1,80.4,79.5),
                   upperCL = c(79.1,81.4,86.8,87.1,83.8,84.2,84.4,86.9,85.4,
                                89.4,71,72.1,73.2,73.7,75.8,78.8,79.8,81.2,81.3,80.9),
                   StandardError = c(1.64,1.58,1.51,1.78,1.7,1.56,1.37,1.4,2.43,
                                     1.57,0.23,0.26,0.3,0.16,0.79,0.78,0.4,0.28,0.23,0.35)
                   )


# Run SII function on the two areas in the data
phe_sii(group_by(data, area),
        decile,
        population,
        value_type = 0, # default normal distribution
        value = value,
        lower_cl = lowerCL,
        upper_cl = upperCL,
        confidence = 0.95,
        rii = TRUE,
        type = "standard")

# Supplying the standard error instead of the indicator 95 percent confidence limits
# gives the same result
phe_sii(group_by(data, area),
        decile,
        population,
        value_type = 0,
        value = value,
        se = StandardError,
        confidence = 0.95,
        rii = TRUE,
        type = "standard")

# multiple confidence intervals, log transforming the data if they are rates
phe_sii(group_by(data, area),
        decile,
        population,
        value_type = 1,
        transform = TRUE,
        value = value,
        lower_cl = lowerCL,
        upper_cl = upperCL,
        confidence = c(0.95, 0.998),
        repetitions = 10000,
        rii = TRUE,
        type = "standard")

}
\references{
(1) Low A & Low A. Measuring the gap: quantifying and comparing local health inequalities.
Journal of Public Health; 2004;26:388-395. \cr \cr
}
\seealso{
Other PHEindicatormethods package functions: 
\code{\link{assign_funnel_significance}()},
\code{\link{calculate_ISRate}()},
\code{\link{calculate_ISRatio}()},
\code{\link{calculate_dsr}()},
\code{\link{calculate_funnel_limits}()},
\code{\link{calculate_funnel_points}()},
\code{\link{phe_dsr}()},
\code{\link{phe_life_expectancy}()},
\code{\link{phe_mean}()},
\code{\link{phe_proportion}()},
\code{\link{phe_quantile}()},
\code{\link{phe_rate}()}
}
\concept{PHEindicatormethods package functions}
