% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/check_onland.R
\name{check_onland}
\alias{check_onland}
\title{Check whether points are located on land}
\usage{
check_onland(
  data,
  land = NULL,
  report = FALSE,
  buffer = 0,
  offline = FALSE,
  plot_leaflet = FALSE,
  only_bad = FALSE
)
}
\arguments{
\item{data}{A data frame containing at least \code{sample_longitude_dd} and \code{sample_latitude_dd}.
Both columns must be numeric, within valid ranges (longitude: -180 to 180, latitude: -90 to 90),
and use WGS84 coordinates (EPSG:4326).}

\item{land}{Optional \code{sf} object containing land polygons. Used only in offline mode.}

\item{report}{Logical; if \code{TRUE}, returns a tibble listing rows on land and warnings.
If \code{FALSE} (default), returns a subset of \code{data} containing only records on land.}

\item{buffer}{Numeric; distance in meters inland for which points are still considered valid.
Only used in online mode. Default is 0.}

\item{offline}{Logical; if \code{TRUE}, the function uses the local cached shoreline (if available). If \code{FALSE} (default),
the OBIS web service is queried.}

\item{plot_leaflet}{Logical; if \code{TRUE}, returns a leaflet map showing points colored by
whether they are on land (red) or in water (green). Default is \code{FALSE}.}

\item{only_bad}{Logical; if \code{TRUE} and \code{plot_leaflet = TRUE}, only points on land (red) are plotted.
Default is \code{FALSE}, meaning all points are plotted.}
}
\value{
If \code{report = TRUE}, a tibble with columns:
\itemize{
\item \code{field}: always \code{NA} (placeholder for future extension)
\item \code{level}: \code{"warning"} for all flagged rows
\item \code{row}: row numbers in \code{data} flagged as located on land
\item \code{message}: description of the issue
}
If \code{report = FALSE} and \code{plot_leaflet = FALSE}, returns a subset of \code{data} with only the flagged rows.
If \code{plot_leaflet = TRUE}, returns a leaflet map showing points on land (red) and in water (green),
unless \code{only_bad = TRUE}, in which case only red points are plotted.
}
\description{
Identifies records whose coordinates fall on land, optionally applying a buffer to allow
points near the coast.
}
\details{
The function supports both offline and online modes:
\itemize{
\item \strong{Offline mode (\code{offline = TRUE})}: uses a local simplified shoreline from a cached
geopackage (\code{land.gpkg}). If the file does not exist, it is downloaded automatically and cached across R sessions.
\item \strong{Online mode (\code{offline = FALSE})}: uses the OBIS web service to determine distance to the shore.
}

The function assumes all coordinates are in WGS84 (EPSG:4326). Supplying coordinates
in a different CRS will result in incorrect intersection tests.

Optionally, a leaflet map can be plotted. Points on land are displayed as red markers,
while points in water are green. If \code{only_bad = TRUE}, only the red points (on land) are plotted.
}
\examples{
# Example data frame with coordinates
example_data <- data.frame(
  sample_latitude_dd = c(59.3, 58.1, 57.5),
  sample_longitude_dd = c(18.6, 17.5, 16.7)
)

# Report points on land with a 100 m buffer
report <- check_onland(example_data, report = TRUE, buffer = 100)
print(report)

# Plot all points colored by land/water
map <- check_onland(example_data, plot_leaflet = TRUE)

# Plot only bad points on land
map_bad <- check_onland(example_data, plot_leaflet = TRUE, only_bad = TRUE)

# Remove points on land by adding a buffer of 2000 m
ok <- check_onland(example_data, report = FALSE, buffer = 2000)
print(nrow(ok))

}
