% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cggm_refit.R
\name{cggm_refit}
\alias{cggm_refit}
\title{Refit the Gaussian Graphical Model for a Given Aggregation and Sparsity Structure}
\usage{
cggm_refit(cggm_output, verbose = 0)
}
\arguments{
\item{cggm_output}{An object of class \code{"CGGM"} as returned by
\code{\link{cggm}()}.}

\item{verbose}{Determines the amount of information printed during the
optimization. Defaults to \code{0}.}
}
\value{
An object of class \code{"CGGM_refit"} with the following components:
\item{A,R}{Lists of matrices. Each pair of matrices with the same index
parametrize the estimated precision matrix after the refitting step given
the aggregation structure found with the corresponding value of the
aggregation parameter \code{lambda_cpath} (and sparsity structure found with
the value of the sparsity parameter \code{lambda_lasso}). It is not
recommended to use these directly, instead use the accessor function
\code{\link{get_Theta}()} to extract the estimated precision matrix for a
given index of the aggregation parameter.}
\item{clusters}{An integer matrix in which each row contains the cluster
assignment of each variable for the corresponding value of the aggregation
parameter \code{lambda_cpath}. Use the accessor function
\code{\link{get_clusters}()} to extract the cluster assignment for a given
index of the aggregation parameter.}
\item{lambdas}{A vector with the values for the aggregation parameter
\code{lambda_cpath} for which the CGGM loss function has been  minimized.}
\item{Theta}{List of matrices. Contains the solution to the minimization
procedure for each value of the aggregation parameter \code{lambda_cpath}.
It is not recommended to use these directly, instead use the accessor
function \code{\link{get_Theta}()} to extract the estimated precision matrix
for a given index of the aggregation parameter.}
\item{cluster_counts}{An integer vector containing the number of clusters
obtained for each value of the aggregation parameter \code{lambda_cpath}.}
\item{cluster_solution_index}{An integer vector containing the index of the
value of the aggregation parameter \code{lambda_cpath} for which a certain
number of clusters was attained. For example,
\code{cluster_solution_index[2]} yields the index of the smallest value for
\code{lambda_cpath} for which a solution with two clusters was found.
Contains -1 if there is no value for  \code{lambda_cpath} with that number
of clusters.}
\item{n}{The number of values of the aggregation parameter
\code{lambda_cpath} for which the CGGM loss function was minimized.}
}
\description{
Estimate the parameters of a clustered and sparse precision matrix or
covariance matrix based on a restricted negative log-likelihood loss
function. The restrictions are given by the provided aggregation and
sparsity structure. This function is different from \code{\link{cggm}()},
as there are no aggregation and sparsity penalties on the precision or
covariance matrix.
}
\note{
The function interface and output structure are still experimental and
may change in the next version.
}
\examples{
# Generate data
set.seed(3)
Theta <- matrix(
  c(2, 1, 0, 0,
    1, 2, 0, 0,
    0, 0, 4, 1,
    0, 0, 1, 4),
  nrow = 4
)
X <- mvtnorm::rmvnorm(n = 100, sigma = solve(Theta))

# Estimate the covariance matrix
S <- cov(X)

# Compute the weight matrix for the clusterpath (clustering) weights
W_cpath <- clusterpath_weights(S, phi = 1, k = 2)

# Compute the weight matrix for the lasso (sparsity) weights
W_lasso <- lasso_weights(S)

# Set values to be used for the aggregation parameter
lambdas <- seq(0, 0.2, by = 0.01)

# Estimate the precision matrix while automatically expanding
# the sequence of values for the aggregation parameter
fit <- cggm(S, W_cpath = W_cpath, lambda_cpath = lambdas,
            W_lasso = W_lasso, lambda_lasso = 0.2,
            expand = TRUE)

# Apply the refitting step to the results, estimating the
# precision matrix based on the clustering and sparsity
# patterns but without additional shrinkage
refit <- cggm_refit(fit)

# A solution with 2 clusters
keep <- refit$cluster_solution_index[2]
get_Theta(refit, index = keep)
get_clusters(refit, index = keep)
}
\references{
D.J.W. Touw, A. Alfons, P.J.F. Groenen and I. Wilms (2025)
\emph{Clusterpath Gaussian Graphical Modeling}. arXiv:2407.00644.
doi:10.48550/arXiv.2407.00644.
}
\seealso{
\code{\link{cggm}()}, \code{\link{cggm_cv}()}
}
\author{
Daniel J.W. Touw
}
