% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dpm.R
\name{dpm}
\alias{dpm}
\title{Dynamic panel models fit with maximum likelihood}
\usage{
dpm(
  formula,
  data,
  error.inv = FALSE,
  const.inv = FALSE,
  alpha.free = FALSE,
  y.lag = 1,
  y.free = FALSE,
  x.free = FALSE,
  fixed.effects = TRUE,
  partial.pre = FALSE,
  print.only = FALSE,
  id = NULL,
  wave = NULL,
  err.inv = NULL,
  weights = NULL,
  ...
)
}
\arguments{
\item{formula}{Model formula. See details for instructions on specifying
parameters properly.}

\item{data}{Data frame in "long" format. Prefers a "panel_data" object.}

\item{error.inv}{Constrain the error variance to be equal across waves.
Default is FALSE.}

\item{const.inv}{Constrain the dependent variable's variance to be equal
across waves (or makes its intercept equal across waves).
This removes cross-sectional dependence. Default is FALSE.}

\item{alpha.free}{Estimate each wave of the dependent variable's loading on
the alpha latent variable. Default is FALSE, meaning each wave has a loading
of 1.}

\item{y.lag}{Which lag(s) of the dependent variable to include in the
regression. Default is 1, but any number or vector of numbers can be used.}

\item{y.free}{If TRUE, allows the regression coefficient(s) for the lagged
dependent variable to vary over time. Default is FALSE. You may alternately
provide a number or vector of numbers corresponding to which lags should
vary freely.}

\item{x.free}{If TRUE, allows the regressions coefficient(s) for the
predictor(s) to vary over time. Default is FALSE. If TRUE, the predictor
regression coefficient(s) can vary over time. Alternately, you may provide
a character vector of predictors to allow to vary if you only want a subset
of predictors to vary.}

\item{fixed.effects}{Fit a fixed effects model? Default is TRUE. If FALSE,
you get a random effects specification instead.}

\item{partial.pre}{Make lagged, predetermined predictors (i.e., they are
surrounded by pre() in the model formula) correlated with the contemporaneous
error term, as discussed in Allison (2022)? Default is FALSE.}

\item{print.only}{Instead of estimating the model, print the \pkg{lavaan}
model string to the console instead.}

\item{id}{Name of the data column that identifies which individual the
observation is. Not needed if \code{data} is a "panel_data" object.}

\item{wave}{Name of the data column that identifies which wave the
observation is from. Not needed if \code{data} is a "panel_data" object.}

\item{err.inv}{Deprecated, same purpose as \code{error.inv}.}

\item{weights}{Equivalent to the argument to \code{lm}, presumably the unquoted
name of a variable in the data that represents the weight. It is passed
to \code{lavaan()}'s \code{sampling.weights} argument.}

\item{...}{Extra parameters to pass to \code{\link[lavaan]{sem}}. Examples
could be \code{missing = "fiml"} for missing data or
\code{estimator = "MLM"} for robust estimation.}
}
\value{
An object of class \code{dpm} which has its own \code{summary} method.

The \code{dpm} object is an extension of the \code{lavaan} class and has all
the capabilities of \code{lavaan} objects, with some extras.

It contains extra slots for: \itemize{

\item \code{mod_string}, the character object used to specify the model
to lavaan. This is helpful if you want to fit the model yourself or
wish to check that the specification is correct.
\item \code{wide_data}, the widened data frame necessary to fit the SEM.

}
}
\description{
Estimate dynamic panel models with fixed effects via
maximum likelihood estimation.
}
\details{
The right-hand side of the formula has two parts, separated by a bar
(\code{|}). The first part should include the time-varying predictors.
The second part, then, is for the time-invariant variables. If you put
a time-varying variable in the second part of the formula, by default
the first wave's value of that variable is treated as the constant.

You must include time-varying predictors. If you do not include a bar
in the formula, all variables are treated as time-varying.

If you would like to include an interaction between time-varying and
time-invariant predictors, you can add a third part to the formula to
specify that term.

\emph{Predetermined variables}:

To set a variable as predetermined, or weakly exogenous, surround the
variable with a \code{pre} function. For instance, if you want the variable
\code{union} to be predetermined, you could specify the formula like this:
\code{wks ~ pre(union) + lwage | ed}, where \code{wks} is the dependent
variable, \code{lwage} is a strictly exogenous time-varying predictor,
and \code{ed} is a strictly exogenous time-invariant predictor.

To lag a predictor, surround the variable with a \code{lag} function in
the same way. Note that the lag function used is specific to this package,
so it does not work the same way as the built-in lag function (i.e., it
understands that you can only lag values \emph{within} entities).

\strong{Note}: CFI and TLI model fit measures for these models should not be
used. They are anti-conservative compared to other implementations and
we have not yet figured out how to get more plausible values.
}
\examples{
# Load example data
data("WageData", package = "panelr")
# Convert data to panel_data format for ease of use
wages <- panel_data(WageData, id = id, wave = t)

# Replicates Allison, Williams, & Moral-Benito (2017) analysis
fit <- dpm(wks ~ pre(lag(union)) + lag(lwage) | ed, data = wages,
            error.inv = TRUE, information = "observed")
# Note: information = "observed" only needed to match Stata/SAS standard errors
summary(fit)


}
\references{
Allison, P. D., Williams, R., & Moral-Benito, E. (2017). Maximum likelihood
for cross-lagged panel models with fixed effects. \emph{Socius}, \emph{3}, 1–17.
http://journals.sagepub.com/doi/10.1177/2378023117710578
}
\author{
Jacob A. Long, in consultation with Richard A. Williams and
Paul D. Allison. All errors are Jacob's.
}
