% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/drf.R
\name{drf}
\alias{drf}
\title{Distributional Random Forests}
\usage{
drf(
  X,
  Y,
  num.trees = 3000,
  splitting.rule = "FourierMMD",
  num.features = 10,
  bandwidth = NULL,
  response.scaling = TRUE,
  node.scaling = FALSE,
  sample.weights = NULL,
  sample.fraction = 0.5,
  mtry = min(ceiling(sqrt(ncol(X)) + 20), ncol(X)),
  min.node.size = 15,
  honesty = TRUE,
  honesty.fraction = 0.5,
  honesty.prune.leaves = TRUE,
  alpha = 0.05,
  imbalance.penalty = 0,
  compute.oob.predictions = FALSE,
  num.threads = NULL,
  seed = stats::runif(1, 0, .Machine$integer.max),
  compute.variable.importance = FALSE,
  ci.group.size = as.integer(num.trees/30)
)
}
\arguments{
\item{X}{The covariates used in the regression. Can be either a numeric matrix or a data.frame with numeric, factor, or character columns, where the last two will be one-hot-encoded.}

\item{Y}{The (multivariate) outcome variable. Needs to be a matrix or a data frame consisting of numeric values.}

\item{num.trees}{Number of trees grown in the forest. Default is 3000.}

\item{splitting.rule}{A character value. The type of the splitting rule used, can be either "FourierMMD" (MMD splitting criterion with FastMMD approximation for speed) or "CART" (sum of standard CART criteria over the components of Y).}

\item{num.features}{A numeric value, in case of "FourierMMD", the number of random features to sample.}

\item{bandwidth}{A numeric value, the bandwidth of the Gaussian kernel used in case of "FourierMMD", the value is set to NULL by default and the square root of the median heuristic is used.}

\item{response.scaling}{A boolean value, should the responses be standardized before fitting the forest. Default is TRUE.}

\item{node.scaling}{A boolean value, should the responses be standardized in every node of every tree. Default is FALSE.}

\item{sample.weights}{(experimental) Weights given to an observation in estimation.
If NULL, each observation is given the same weight. Default is NULL.}

\item{sample.fraction}{Fraction of the data used to build each tree.
Note: If honesty = TRUE, these subsamples will
further be cut by a factor of honesty.fraction. Default is 0.5.}

\item{mtry}{Number of variables tried for each split. Default is
\eqn{\sqrt p + 20}, where p is the number of predictors.}

\item{min.node.size}{A target for the minimum number of observations in each tree leaf. Note that nodes
with size smaller than min.node.size can occur, as in the original randomForest package.
Default is 5.}

\item{honesty}{Whether to use honest splitting (i.e., sub-sample splitting). Default is TRUE.
For a detailed description of honesty, honesty.fraction, honesty.prune.leaves, and recommendations for
parameter tuning, see the \href{https://grf-labs.github.io/grf/REFERENCE.html#honesty-honesty-fraction-honesty-prune-leaves}{GRF reference}
for more information (the original source).}

\item{honesty.fraction}{The fraction of data that will be used for determining splits if honesty = TRUE. Default is 0.5 (i.e. half of the data is used for
determining splits and the other half for populating the nodes of the tree).}

\item{honesty.prune.leaves}{If TRUE, prunes the estimation sample tree such that no leaves
are empty. If FALSE, keeps the same tree as determined in the splits sample (if an empty leave is encountered, that
tree is skipped and does not contribute to the estimate). Setting this to FALSE may improve performance on
small/marginally powered data, but requires more trees (note: tuning does not adjust the number of trees).
Only applies if honesty is enabled. Default is TRUE.}

\item{alpha}{A tuning parameter that controls the maximum imbalance of a split. Default is 0.05, meaning a child node will contain at most 5\% of observations in the parent node.}

\item{imbalance.penalty}{A tuning parameter that controls how harshly imbalanced splits are penalized. Default is 0.}

\item{compute.oob.predictions}{Whether OOB predictions on training set should be precomputed.}

\item{num.threads}{Number of threads used in training. By default, the number of threads is set
to the maximum hardware concurrency.}

\item{seed}{The seed of the C++ random number generator.}

\item{compute.variable.importance}{boolean, should the variable importance be computed in the object.}

\item{ci.group.size}{The forest will grow ci.group.size trees on each subsample. 
In order to provide confidence intervals, ci.group.size must be at least 2. 
Defaults to \code{num.trees/30} which yields 30 CI groups.}
}
\value{
A trained Distributional Random Forest object.
}
\description{
Trains a Distributional Random Forest which estimates the full conditional distribution \eqn{P(Y | X)}
for possibly multivariate response Y and predictors X. The conditional distribution estimate is represented
as a weighted distribution of the training data. The weights can be conveniently used in the downstream analysis
to estimate any quantity of interest \eqn{\tau(P(Y | X))}.
}
\examples{
\donttest{
library(drf)

n = 1000
p = 20
d = 3

# Generate training data
X = matrix(rnorm(n * p), nrow=n)
Y = matrix(rnorm(n * d), nrow=n)
Y[, 1] = Y[, 1] + X[, 1]
Y[, 2] = Y[, 2] * X[, 2]
Y[, 3] = Y[, 3] * X[, 1] + X[, 2]

# Fit DRF object
drf.forest = drf(X, Y)

# Generate test data
X_test = matrix(rnorm(10 * p), nrow=10)

out = predict(drf.forest, newdata=X_test)
# Compute E[Y_1 | X] for all data in X_test directly from
# the weights representing the estimated distribution
out$weights \%*\% out$y[,1]

out = predict(drf.forest, newdata=X_test,
              functional='mean')
# Compute E[Y_1 | X] for all data in X_test using built-in functionality
out[,1]

out = predict(drf.forest, newdata=X_test,
              functional='quantile',
              quantiles=c(0.25, 0.75),
              transformation=function(y){y[1] * y[2] * y[3]})
# Compute 25\% and 75\% quantiles of Y_1*Y_2*Y_3, conditionally on X = X_test[1, ]
out[1,,]

out = predict(drf.forest, newdata=X_test,
              functional='cov',
              transformation=function(y){matrix(1:6, nrow=2) \%*\% y})
# Compute 2x2 covariance matrix for (1*Y_1 + 3*Y_2 + 5*Y_3, 2*Y_1 + 4*Y_2 + 6*Y_3),
# conditionally on X = X_test[1, ]
out[1,,]

out = predict(drf.forest, newdata=X_test,
              functional='custom',
              custom.functional=function(y, w){c(sum(y[, 1] * w), sum(y[, 2] * w))})
# Compute E[Y_1, Y_2 | X] for all data in X_test by providing custom functional that
# computes it from the weights
out
}

}
\seealso{
See \code{\link{predict.drf}} for how to make predictions, including uncertainty weights.
}
