% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bkg_grid.R
\name{bkg_grid}
\alias{bkg_grid}
\alias{bkg_grid_fast}
\alias{bkg_grid_full}
\title{INSPIRE grids}
\usage{
bkg_grid_fast(
  year = c("2019", "2018", "2017", "2015"),
  resolution = c("100km", "10km", "5km", "1km", "250m", "100m"),
  timeout = 600,
  update_cache = FALSE
)

bkg_grid_full(
  year = "latest",
  resolution = c("100km", "10km", "5km", "1km", "250m", "100m"),
  timeout = 600,
  update_cache = FALSE
)
}
\arguments{
\item{year}{Version of the grid. Can be \code{"2015"}, \code{"2017"},
\code{"2018"} or \code{"2019"}. For \code{bkg_grid_fast}, \code{"latest"}
downloads the latest version of the grid.}

\item{resolution}{Cell size of the grid. Can be \code{"100m"}, \code{"250m"},
\code{"1km"}, \code{"5km"}, \code{"10km"}, or \code{"100km"}.}

\item{timeout}{Timeout value for the data download passed to
\code{\link[httr2]{req_timeout}}. Adjust this if your internet connection is
slow or you are downloading larger datasets.}

\item{update_cache}{By default, downloaded files are cached in the
\code{tempdir()} directory of R. When downloading the same data again,
the data is not downloaded but instead taken from the cache. Sometimes
this can be not the desired behavior. If you want to overwrite the cache,
pass \code{TRUE}. Defaults to \code{FALSE}, i.e. always adopt the cache
if possible.}
}
\value{
\code{bkg_grid_fast} returns an sf dataframe with point geometries
and no features. \code{bkg_grid_full} also returns point geometries but
with the following additional features:

\itemize{
\item{\code{x_sw}}: X coordinate of the South-West corner of a grid cell
\item{\code{y_sw}}: Y coordinate of the South-West corner of a grid cell
\item{\code{f_staat}}: State area in the grid cell in square meters
\item{\code{f_land}}: Land area in the grid cell in square meters
\item{\code{f_wasser}}: Water area in the grid cell in square meters
\item{\code{p_staat}}: Share of state area in the grid cell
\item{\code{p_land}}: Share of land area in the grid cell
\item{\code{p_wasser}}: Share of water area in the grid cell
\item{\code{ags}}: Official municipality key (Amtlicher Gemeindeschlüssel).
Related to the ARS but shortened to omit position 6 to 9. Structured as
follows: \itemize{
\item{Position 1-2: Federal state}
\item{Position 3: Government region}
\item{Position 4-5: District}
\item{Position 6-8: Municipality}
}
}

Note that \code{ags} is only included for resolutions \code{"100m"} and
\code{"250m"}
}
\description{
Retrieve geometries of INSPIRE-compliant grid geometries (also called
"GeoGitter"). \code{bkg_grid_fast()} is much faster than \code{bkg_grid_full()}
by downloading heavily compressed versions grids. This happens at the cost
of data richness as \code{bkg_grid_fast()} only contains the geometries and
nothing else. Note that the \code{arrow} package needs to be installed to
use \code{bkg_grid_fast()}.

Note that the output contains point geometries. Most of the times, you
want to work with rasters instead. To convert a given object \code{out},
type the following (\code{terra} package required):

\preformatted{terra::rast(out)}

These functions interface the \code{GeoGitter} product of the BKG.
}
\details{
The following table gives a rough idea of how much less data
\code{bkg_grid_fast} needs to download for each resolution compared to
\code{bkg_grid_full}.

\tabular{lll}{
\strong{Size} \tab \strong{fast} \tab \strong{full}\cr
100km \tab 0.78 kB \tab 933 kB    \cr
10km  \tab 2.68 kB \tab 1,015 kB   \cr
5km   \tab 3.53 kB \tab 1,253 kB   \cr
1km   \tab 28.7 kB \tab 5,249 kB   \cr
500m  \tab 133 kB  \tab 15,902 kB \cr
250m  \tab 289 kB  \tab 53,900 kB \cr
100m  \tab 1,420 kB \tab 291,000 kB
}
}
\examples{
\dontshow{if (getFromNamespace("ffm_run_examples", ns = "ffm")()) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
# Return a bare-bones version of the INSPIRE grid
grid <- bkg_grid_fast(year = "2019", resolution = "100km")

# Return a fully detailed version instead
grid_full <- bkg_grid_full(resolution = "5km")

plot(grid)

# Convert grid to SpatRaster
if (requireNamespace("terra")) {
  library(terra)
  raster <- rast(vect(grid_full["p_wasser"]), type = "xyz")
  plot(raster, main = "Share of water area")
}
\dontshow{\}) # examplesIf}
}
\seealso{
\href{https://sgx.geodatenzentrum.de/web_public/gdz/dokumentation/deu/geogitter.pdf}{GeoGitter documentation}

\href{https://mis.bkg.bund.de/trefferanzeige?docuuid=02A7E63D-CAAA-4DED-B6FF-1F1E73FAF883}{GeoGitter MIS record}

Other non-administrative regions: 
\code{\link{bkg_authorities}()},
\code{\link{bkg_kfz}()},
\code{\link{bkg_ror}()}
}
\concept{non-administrative regions}
