% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/causality_pred.R
\name{causality_pred}
\alias{causality_pred}
\title{Out-of-sample Tests of Granger Causality}
\usage{
causality_pred(
  y,
  cause = NULL,
  p = NULL,
  p.free = FALSE,
  lag.restrict = 0L,
  lag.max = NULL,
  k = 2,
  B = 500L,
  test = 0.3,
  cl = 1L
)
}
\arguments{
\item{y}{matrix, data frame, or \code{ts} object with two columns
(a dependent and an explanatory time-series variable). Missing values are not allowed.}

\item{cause}{name of the cause variable. If not specified, the first variable in
\code{y} is treated as the dependent variable and the second is treated as the cause.}

\item{p}{a vector of one or two positive integers specifying the order \eqn{p} of
autoregressive dependence. The input of length one is recycled, then \code{p[1]} is used for
the dependent variable and \code{p[2]} is used for the cause variable.
The user must specify \code{p} or \code{lag.max}.
If \code{lag.max} is specified, the argument \code{p} is ignored.}

\item{p.free}{logical value indicating whether the autoregressive orders for the
dependent and cause variables should be selected independently.
The default \code{p.free = FALSE} means the same autoregressive order is
selected for both variables. Note that if \code{p.free = TRUE} and \code{lag.max} is specified,
then \code{lag.max[1] * (lag.max[2] - lag.restrict)} models are compared,
which might be slow depending on the maximal lags and sample size.}

\item{lag.restrict}{integer for the number of short-term lags in the cause variable
to remove from consideration (default is zero, meaning no lags are removed).
This setting does not affect the dependent variable lags that are always present.}

\item{lag.max}{a vector of one or two positive integers for the highest lag orders to explore.
The input of length one is recycled, then \code{lag.max[1]} used for
the dependent variable and \code{lag.max[2]} is used for the cause variable.
The order is then selected using the Akaike information criterion (AIC; default),
see the argument \code{k} to change the criterion.
\code{lag.max} of length 2 automatically sets \code{p.free = TRUE}.}

\item{k}{numeric scalar specifying the weight of the equivalent degrees of freedom part
in the AIC formula. Default \code{k = 2} corresponds to the traditional AIC.
Use \code{k = log(n)} to use the Bayesian information criterion instead
(see \code{\link[stats]{extractAIC}}).}

\item{B}{number of bootstrap replications. Default is 500.}

\item{test}{a numeric value specifying the size of the testing set. If \code{test} < 1,
the value is treated as a proportion of the sample size to be used as the testing set.
Otherwise, \code{test} is treated as the number of the most recent values to be used as the testing set.
Default is 0.3, which means that 30\% of the sample is used for calculating
out-of-sample errors. The testing set is always at the end of the time series.}

\item{cl}{parameter to specify computer cluster for bootstrapping passed to
the package \code{parallel} (default \code{cl = 1}, means no cluster is used).
Possible values are:
\itemize{
\item cluster object (list) produced by \link[parallel]{makeCluster}.
In this case, a new cluster is not started nor stopped;
\item \code{NULL}. In this case, the function will detect
available cores (see \link[parallel]{detectCores}) and, if there are
multiple cores (\eqn{>1}), a cluster will be started with
\link[parallel]{makeCluster}. If started, the cluster will be stopped
after the computations are finished;
\item positive integer defining the number of cores to start a cluster.
If \code{cl = 1} (default), no attempt to create a cluster will be made.
If \code{cl} > 1, a cluster will be started (using \link[parallel]{makeCluster})
and stopped afterward (using \link[parallel]{stopCluster}).
}}
}
\value{
A list containing the following elements:
\item{stat}{a table with the observed values of the test statistics and \eqn{p}-values.}
\item{cause}{the cause variable.}
\item{p}{the AR orders used for the dependent variable (\code{p[1]}) and for the cause variable (\code{p[2]}).}
}
\description{
Test for Granger causality using out-of-sample prediction errors from an
autoregression (AR) model, where some of the near-contemporaneous lags can be removed:
\deqn{Y_t = \sum_{i=1}^{p1}\alpha_iY_{t-i} + \sum_{i=lag.restrict+1}^{p2}\beta_iX_{t-i} + e_t,}
where \eqn{Y_t} is the dependent variable,
\eqn{X_t} is the cause variable,
\eqn{p1} and \eqn{p2} are the AR orders (if \code{p.free = FALSE}, \eqn{p1 = p2}),
\eqn{lag.restrict} is the number of restricted first lags (see the argument \code{lag.restrict}).
}
\details{
The tests include the MSE-t approach \insertCite{McCracken_2007}{funtimes} and
MSE-correlation test as in Chapter 9.3 of \insertCite{Granger_Newbold_2016;textual}{funtimes}.
The bootstrap is used to empirically derive distributions of the statistics.

In the implemented bootstrapping, residuals of the restricted model under the null hypothesis of no Granger
causality are bootstrapped to generate new data under the null hypothesis. Then, the full and restricted
models are re-estimated on the bootstrapped data to obtain new (bootstrapped) forecast errors.

In the current implementation, the bootstrapped \eqn{p}-value is calculated using Equation 4.10 in
\insertCite{Davison_Hinkley_1997;textual}{funtimes}: \code{p.value} = (1 + \eqn{n}) / (\code{B} + 1),
where \eqn{n} is the number of bootstrapped statistics smaller or equal to the observed statistic.

This function tests the Granger causation
of \eqn{X} to \eqn{Y} or from \eqn{Y} to \eqn{X}
(to test in both directions, need to run the function twice, with different argument \code{cause}).
To use the symmetric vector autoregression (VAR), use the function \code{\link{causality_predVAR}}.
}
\examples{
\dontrun{
# Example 1: Canada time series (ts object)
Canada <- vars::Canada
causality_pred(Canada[,1:2], cause = "e", lag.max = 5, p.free = TRUE)
causality_pred(Canada[,1:2], cause = "e", lag.restrict = 3, lag.max = 15, p.free = TRUE)

# Example 2 (run in parallel, initiate the cluster automatically)
# Box & Jenkins time series
# of sales and a leading indicator, see ?BJsales

D <- cbind(BJsales.lead, BJsales)
causality_pred(D, cause = "BJsales.lead", lag.max = 5, B = 1000, cl = NULL)

# Example 3 (run in parallel, initiate the cluster manually)

# Initiate a local cluster
cores <- parallel::detectCores()
cl <- parallel::makeCluster(cores)
parallel::clusterSetRNGStream(cl, 123) # to make parallel computations reproducible

causality_pred(D, cause = "BJsales.lead", lag.max = 5, B = 1000, cl = cl)
causality_pred(D, cause = "BJsales.lead", lag.restrict = 3, p = 5, B = 1000, cl = cl)
parallel::stopCluster(cl)
}

}
\references{
\insertAllCited{}
}
\seealso{
\code{\link{causality_predVAR}}
}
\author{
Vyacheslav Lyubchich
}
\keyword{causality}
\keyword{htest}
\keyword{ts}
