% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gkwreg-datasets.R
\docType{data}
\name{FoodExpenditure}
\alias{FoodExpenditure}
\title{Proportion of Household Income Spent on Food}
\format{
A data frame with 38 observations on 3 variables:
\describe{
\item{food}{numeric. Annual food expenditure in U.S. dollars.}
\item{income}{numeric. Annual household income in U.S. dollars.}
\item{persons}{numeric. Number of persons in the household.}
}
}
\source{
Taken from Griffiths et al. (1993, Table 15.4).
}
\usage{
FoodExpenditure
}
\description{
Cross-section data on annual food expenditure and annual income for a random
sample of households in a large U.S. city. The dataset models the proportion
of income spent on food as a function of total income and household size.
}
\details{
This classic econometric dataset was taken from Griffiths et al. (1993, Table
15.4) who cite Leser (1963) as the original source. The data are used to model
Engel curves, which describe how household expenditure on a particular good or
service varies with household income.

The response variable of interest is typically \code{food/income}, the proportion
of income spent on food, which follows beta distribution properties as it is
bounded between 0 and 1.
}
\examples{
\donttest{
require(gkwreg)
require(gkwdist)

data(FoodExpenditure)
FoodExpenditure$prop <- FoodExpenditure$food / FoodExpenditure$income

# Example 1: Basic Kumaraswamy regression
# Proportion spent on food decreases with income (Engel's law)
# Larger households spend more on food
fit_kw <- gkwreg(prop ~ income + persons,
  data = FoodExpenditure,
  family = "kw"
)
summary(fit_kw)

# Interpretation:
# - Alpha: Negative income effect (Engel's law)
#   Positive household size effect
# - Beta: Constant precision (homoscedastic model)

# Example 2: Heteroscedastic model
# Variability in food proportion may differ by income and household size
fit_kw_hetero <- gkwreg(
  prop ~ income + persons |
    income + persons,
  data = FoodExpenditure,
  family = "kw"
)
summary(fit_kw_hetero)

# Interpretation:
# - Beta: Precision varies with both income and household size
#   Wealthier or larger households may show different spending variability

# Test for heteroscedasticity
anova(fit_kw, fit_kw_hetero)

# Example 3: Exponentiated Kumaraswamy for extreme spending patterns
# Some households may have unusual food spending (very frugal or lavish)
fit_ekw <- gkwreg(
  prop ~ income + persons | # alpha
    persons | # beta: household size affects precision
    income, # lambda: income affects extremity
  data = FoodExpenditure,
  family = "ekw"
)
summary(fit_ekw)

# Interpretation:
# - Lambda: Income level affects tail behavior
#   Rich households may show more extreme (unusual) spending patterns

# Visualization: Engel curve
plot(prop ~ income,
  data = FoodExpenditure,
  xlab = "Annual Income ($)", ylab = "Proportion Spent on Food",
  main = "Engel Curve for Food Expenditure"
)
# Add fitted values
FoodExpenditure$fitted_kw <- fitted(fit_kw)
points(FoodExpenditure$income, FoodExpenditure$fitted_kw,
  col = "blue", pch = 19, cex = 0.8
)
legend("topright",
  legend = c("Observed", "Fitted"),
  col = c("black", "blue"), pch = c(1, 19)
)
}
}
\references{
Cribari-Neto, F., and Zeileis, A. (2010). Beta Regression in R.
\emph{Journal of Statistical Software}, \strong{34}(2), 1--24.
\doi{10.18637/jss.v034.i02}

Ferrari, S.L.P., and Cribari-Neto, F. (2004). Beta Regression for Modeling
Rates and Proportions. \emph{Journal of Applied Statistics}, \strong{31}(7),
799--815.

Griffiths, W.E., Hill, R.C., and Judge, G.G. (1993). \emph{Learning and
Practicing Econometrics}. New York: John Wiley and Sons.

Leser, C.E.V. (1963). Forms of Engel Functions. \emph{Econometrica},
\strong{31}(4), 694--703.
}
\keyword{datasets}
