/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Smodule.h" 

#include "H5private.h"  
#include "H5Eprivate.h" 
#include "H5Iprivate.h" 
#include "H5Spkg.h"     

static herr_t   H5S__none_copy(H5S_t *dst, const H5S_t *src, bool share_selection);
static herr_t   H5S__none_release(H5S_t *space);
static htri_t   H5S__none_is_valid(const H5S_t *space);
static hssize_t H5S__none_serial_size(H5S_t *space);
static herr_t   H5S__none_serialize(H5S_t *space, uint8_t **p);
static herr_t   H5S__none_deserialize(H5S_t **space, const uint8_t **p, const size_t p_size, bool skip);
static herr_t   H5S__none_bounds(const H5S_t *space, hsize_t *start, hsize_t *end);
static herr_t   H5S__none_offset(const H5S_t *space, hsize_t *off);
static int      H5S__none_unlim_dim(const H5S_t *space);
static htri_t   H5S__none_is_contiguous(const H5S_t *space);
static htri_t   H5S__none_is_single(const H5S_t *space);
static htri_t   H5S__none_is_regular(H5S_t *space);
static htri_t   H5S__none_shape_same(H5S_t *space1, H5S_t *space2);
static htri_t   H5S__none_intersect_block(H5S_t *space, const hsize_t *start, const hsize_t *end);
static herr_t   H5S__none_adjust_u(H5S_t *space, const hsize_t *offset);
static herr_t   H5S__none_adjust_s(H5S_t *space, const hssize_t *offset);
static herr_t   H5S__none_project_scalar(const H5S_t *space, hsize_t *offset);
static herr_t   H5S__none_project_simple(const H5S_t *space, H5S_t *new_space, hsize_t *offset);
static herr_t   H5S__none_iter_init(H5S_t *space, H5S_sel_iter_t *iter);

static herr_t  H5S__none_iter_coords(const H5S_sel_iter_t *iter, hsize_t *coords);
static herr_t  H5S__none_iter_block(const H5S_sel_iter_t *iter, hsize_t *start, hsize_t *end);
static hsize_t H5S__none_iter_nelmts(const H5S_sel_iter_t *iter);
static htri_t  H5S__none_iter_has_next_block(const H5S_sel_iter_t *iter);
static herr_t  H5S__none_iter_next(H5S_sel_iter_t *sel_iter, size_t nelem);
static herr_t  H5S__none_iter_next_block(H5S_sel_iter_t *sel_iter);
static herr_t  H5S__none_iter_get_seq_list(H5S_sel_iter_t *iter, size_t maxseq, size_t maxbytes, size_t *nseq,
                                           size_t *nbytes, hsize_t *off, size_t *len);
static herr_t  H5S__none_iter_release(H5S_sel_iter_t *sel_iter);

const H5S_select_class_t H5S_sel_none[1] = {{
    H5S_SEL_NONE,

    
    H5S__none_copy,
    H5S__none_release,
    H5S__none_is_valid,
    H5S__none_serial_size,
    H5S__none_serialize,
    H5S__none_deserialize,
    H5S__none_bounds,
    H5S__none_offset,
    H5S__none_unlim_dim,
    NULL,
    H5S__none_is_contiguous,
    H5S__none_is_single,
    H5S__none_is_regular,
    H5S__none_shape_same,
    H5S__none_intersect_block,
    H5S__none_adjust_u,
    H5S__none_adjust_s,
    H5S__none_project_scalar,
    H5S__none_project_simple,
    H5S__none_iter_init,
}};

static const H5S_sel_iter_class_t H5S_sel_iter_none[1] = {{
    H5S_SEL_NONE,

    
    H5S__none_iter_coords,
    H5S__none_iter_block,
    H5S__none_iter_nelmts,
    H5S__none_iter_has_next_block,
    H5S__none_iter_next,
    H5S__none_iter_next_block,
    H5S__none_iter_get_seq_list,
    H5S__none_iter_release,
}};

static herr_t
H5S__none_iter_init(H5S_t H5_ATTR_UNUSED *space, H5S_sel_iter_t *iter)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(space && H5S_SEL_NONE == H5S_GET_SELECT_TYPE(space));
    assert(iter);

    
    iter->type = H5S_sel_iter_none;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5S__none_iter_coords(const H5S_sel_iter_t H5_ATTR_UNUSED *iter, hsize_t H5_ATTR_UNUSED *coords)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(iter);
    assert(coords);

    FUNC_LEAVE_NOAPI(FAIL)
} 

static herr_t
H5S__none_iter_block(const H5S_sel_iter_t H5_ATTR_UNUSED *iter, hsize_t H5_ATTR_UNUSED *start,
                     hsize_t H5_ATTR_UNUSED *end)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(iter);
    assert(start);
    assert(end);

    FUNC_LEAVE_NOAPI(FAIL)
} 

static hsize_t
H5S__none_iter_nelmts(const H5S_sel_iter_t H5_ATTR_UNUSED *iter)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(iter);

    FUNC_LEAVE_NOAPI(0)
} 

static htri_t
H5S__none_iter_has_next_block(const H5S_sel_iter_t H5_ATTR_UNUSED *iter)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(iter);

    FUNC_LEAVE_NOAPI(FAIL)
} 

static herr_t
H5S__none_iter_next(H5S_sel_iter_t H5_ATTR_UNUSED *iter, size_t H5_ATTR_UNUSED nelem)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(iter);
    assert(nelem > 0);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5S__none_iter_next_block(H5S_sel_iter_t H5_ATTR_UNUSED *iter)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(iter);

    FUNC_LEAVE_NOAPI(FAIL)
} 

static herr_t
H5S__none_iter_get_seq_list(H5S_sel_iter_t H5_ATTR_UNUSED *iter, size_t H5_ATTR_UNUSED maxseq,
                            size_t H5_ATTR_UNUSED maxelem, size_t *nseq, size_t *nelem,
                            hsize_t H5_ATTR_UNUSED *off, size_t H5_ATTR_UNUSED *len)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(iter);
    assert(maxseq > 0);
    assert(maxelem > 0);
    assert(nseq);
    assert(nelem);
    assert(off);
    assert(len);

    
    *nseq = 0;

    
    *nelem = 0;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5S__none_iter_release(H5S_sel_iter_t H5_ATTR_UNUSED *iter)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(iter);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5S__none_release(H5S_t H5_ATTR_UNUSED *space)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(space);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5S__none_copy(H5S_t *dst, const H5S_t H5_ATTR_UNUSED *src, bool H5_ATTR_UNUSED share_selection)
{
    FUNC_ENTER_PACKAGE_NOERR

    assert(src);
    assert(dst);

    
    dst->select.num_elem = 0;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static htri_t
H5S__none_is_valid(const H5S_t H5_ATTR_UNUSED *space)
{
    FUNC_ENTER_PACKAGE_NOERR

    assert(space);

    FUNC_LEAVE_NOAPI(true)
} 

static hssize_t
H5S__none_serial_size(H5S_t H5_ATTR_UNUSED *space)
{
    FUNC_ENTER_PACKAGE_NOERR

    assert(space);

    
    FUNC_LEAVE_NOAPI(16)
} 

static herr_t
H5S__none_serialize(H5S_t *space, uint8_t **p)
{
    uint8_t *pp = (*p); 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(space);
    assert(p);
    assert(pp);

    
    UINT32ENCODE(pp, (uint32_t)H5S_GET_SELECT_TYPE(space)); 
    UINT32ENCODE(pp, (uint32_t)H5S_NONE_VERSION_1);         
    UINT32ENCODE(pp, (uint32_t)0);                          
    UINT32ENCODE(pp, (uint32_t)0);                          

    
    *p = pp;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5S__none_deserialize(H5S_t **space, const uint8_t **p, const size_t p_size, bool skip)
{
    H5S_t *tmp_space = NULL;                    
    uint32_t       version;                     
    herr_t         ret_value = SUCCEED;         
    const uint8_t *p_end     = *p + p_size - 1; 

    FUNC_ENTER_PACKAGE

    assert(p);
    assert(*p);

    
    
    if (!*space) {
        if (NULL == (tmp_space = H5S_create(H5S_SIMPLE)))
            HGOTO_ERROR(H5E_DATASPACE, H5E_CANTCREATE, FAIL, "can't create dataspace");
    } 
    else
        tmp_space = *space;

    
    if (H5_IS_KNOWN_BUFFER_OVERFLOW(skip, *p, sizeof(uint32_t), p_end))
        HGOTO_ERROR(H5E_DATASPACE, H5E_OVERFLOW, FAIL, "buffer overflow while decoding selection version");
    UINT32DECODE(*p, version);

    if (version < H5S_NONE_VERSION_1 || version > H5S_NONE_VERSION_LATEST)
        HGOTO_ERROR(H5E_DATASPACE, H5E_BADVALUE, FAIL, "bad version number for none selection");

    
    if (H5_IS_KNOWN_BUFFER_OVERFLOW(skip, *p, 8, p_end))
        HGOTO_ERROR(H5E_DATASPACE, H5E_OVERFLOW, FAIL, "buffer overflow while decoding selection header");
    *p += 8;

    
    if (H5S_select_none(tmp_space) < 0)
        HGOTO_ERROR(H5E_DATASPACE, H5E_CANTDELETE, FAIL, "can't change selection");

    
    if (!*space)
        *space = tmp_space;

done:
    
    if (!*space && tmp_space)
        if (H5S_close(tmp_space) < 0)
            HDONE_ERROR(H5E_DATASPACE, H5E_CANTFREE, FAIL, "can't close dataspace");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5S__none_bounds(const H5S_t H5_ATTR_UNUSED *space, hsize_t H5_ATTR_UNUSED *start,
                 hsize_t H5_ATTR_UNUSED *end)
{
    FUNC_ENTER_PACKAGE_NOERR

    assert(space);
    assert(start);
    assert(end);

    FUNC_LEAVE_NOAPI(FAIL)
} 

static herr_t
H5S__none_offset(const H5S_t H5_ATTR_UNUSED *space, hsize_t H5_ATTR_UNUSED *offset)
{
    FUNC_ENTER_PACKAGE_NOERR

    assert(space);
    assert(offset);

    FUNC_LEAVE_NOAPI(FAIL)
} 

static int
H5S__none_unlim_dim(const H5S_t H5_ATTR_UNUSED *space)
{
    FUNC_ENTER_PACKAGE_NOERR

    FUNC_LEAVE_NOAPI(-1)
} 

static htri_t
H5S__none_is_contiguous(const H5S_t H5_ATTR_UNUSED *space)
{
    FUNC_ENTER_PACKAGE_NOERR

    assert(space);

    FUNC_LEAVE_NOAPI(false)
} 

static htri_t
H5S__none_is_single(const H5S_t H5_ATTR_UNUSED *space)
{
    FUNC_ENTER_PACKAGE_NOERR

    assert(space);

    FUNC_LEAVE_NOAPI(false)
} 

static htri_t
H5S__none_is_regular(H5S_t H5_ATTR_UNUSED *space)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(space);

    FUNC_LEAVE_NOAPI(true)
} 

static htri_t
H5S__none_shape_same(H5S_t H5_ATTR_UNUSED *space1, H5S_t H5_ATTR_UNUSED *space2)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(space1);
    assert(space2);

    FUNC_LEAVE_NOAPI(true)
} 

htri_t
H5S__none_intersect_block(H5S_t H5_ATTR_UNUSED *space, const hsize_t H5_ATTR_UNUSED *start,
                          const hsize_t H5_ATTR_UNUSED *end)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(space);
    assert(H5S_SEL_NONE == H5S_GET_SELECT_TYPE(space));
    assert(start);
    assert(end);

    FUNC_LEAVE_NOAPI(false)
} 

static herr_t
H5S__none_adjust_u(H5S_t H5_ATTR_UNUSED *space, const hsize_t H5_ATTR_UNUSED *offset)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(space);
    assert(offset);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5S__none_adjust_s(H5S_t H5_ATTR_UNUSED *space, const hssize_t H5_ATTR_UNUSED *offset)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(space);
    assert(offset);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5S__none_project_scalar(const H5S_t H5_ATTR_UNUSED *space, hsize_t H5_ATTR_UNUSED *offset)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(space && H5S_SEL_NONE == H5S_GET_SELECT_TYPE(space));
    assert(offset);

    FUNC_LEAVE_NOAPI(FAIL)
} 

static herr_t
H5S__none_project_simple(const H5S_t H5_ATTR_UNUSED *base_space, H5S_t *new_space,
                         hsize_t H5_ATTR_UNUSED *offset)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(base_space && H5S_SEL_NONE == H5S_GET_SELECT_TYPE(base_space));
    assert(new_space);
    assert(offset);

    
    if (H5S_select_none(new_space) < 0)
        HGOTO_ERROR(H5E_DATASPACE, H5E_CANTSET, FAIL, "unable to set none selection");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5S_select_none(H5S_t *space)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(space);

    
    if (H5S_SELECT_RELEASE(space) < 0)
        HGOTO_ERROR(H5E_DATASPACE, H5E_CANTDELETE, FAIL, "can't release hyperslab");

    
    space->select.num_elem = 0;

    
    space->select.type = H5S_sel_none;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5Sselect_none(hid_t spaceid)
{
    H5S_t *space;               
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (space = (H5S_t *)H5I_object_verify(spaceid, H5I_DATASPACE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a dataspace");

    
    if (H5S_select_none(space) < 0)
        HGOTO_ERROR(H5E_DATASPACE, H5E_CANTDELETE, FAIL, "can't change selection");

done:
    FUNC_LEAVE_API(ret_value)
} 
