/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#define H5D_FRIEND 

#include "H5private.h"   
#include "H5CXprivate.h" 
#include "H5Dpkg.h"      
#include "H5Eprivate.h"  
#include "H5Fprivate.h"  
#include "H5Gprivate.h"  
#include "H5Iprivate.h"  
#include "H5MMprivate.h" 
#include "H5Pprivate.h"  
#include "H5Sprivate.h"  
#include "H5VLprivate.h" 

#include "H5VLnative_private.h" 

static herr_t H5VL__native_dataset_io_setup(size_t count, void *obj[], hid_t mem_type_id[],
                                            hid_t mem_space_id[], hid_t file_space_id[], hid_t dxpl_id,
                                            H5_flexible_const_ptr_t buf[], H5D_dset_io_info_t *dinfo);
static herr_t H5VL__native_dataset_io_cleanup(size_t count, hid_t mem_space_id[], hid_t file_space_id[],
                                              H5D_dset_io_info_t *dinfo);

static herr_t
H5VL__native_dataset_io_setup(size_t count, void *obj[], hid_t mem_type_id[], hid_t mem_space_id[],
                              hid_t file_space_id[], hid_t dxpl_id, H5_flexible_const_ptr_t buf[],
                              H5D_dset_io_info_t *dinfo)
{
    H5F_shared_t *f_sh;
    size_t        i;
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(dinfo);

    
    f_sh = H5F_SHARED(((H5D_t *)obj[0])->oloc.file);

    
    for (i = 0; i < count; i++) {
        
        memset(&dinfo[i].layout_ops, 0, sizeof(dinfo[i].layout_ops));
        memset(&dinfo[i].io_ops, 0, sizeof(dinfo[i].io_ops));
        memset(&dinfo[i].layout_io_info, 0, sizeof(dinfo[i].layout_io_info));
        memset(&dinfo[i].type_info, 0, sizeof(dinfo[i].type_info));
        dinfo[i].store   = NULL;
        dinfo[i].layout  = NULL;
        dinfo[i].nelmts  = 0;
        dinfo[i].skip_io = false;

        
        dinfo[i].dset = (H5D_t *)obj[i];
        assert(dinfo[i].dset);

        
        if (NULL == dinfo[i].dset->oloc.file)
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "dataset is not associated with a file");
        if (f_sh != H5F_SHARED(dinfo[i].dset->oloc.file))
            HGOTO_ERROR(H5E_ARGS, H5E_UNSUPPORTED, FAIL,
                        "different files detected in multi dataset I/O request");

        
        if (NULL == (dinfo[i].mem_type = (const H5T_t *)H5I_object_verify(mem_type_id[i], H5I_DATATYPE)))
            HGOTO_ERROR(H5E_DATASET, H5E_BADTYPE, FAIL, "invalid datatype");

        
        if (H5S_ALL == file_space_id[i])
            
            dinfo[i].file_space = dinfo[i].dset->shared->space;
        else if (H5S_BLOCK == file_space_id[i])
            HGOTO_ERROR(H5E_DATASET, H5E_BADTYPE, FAIL, "H5S_BLOCK is not allowed for file dataspace");
        else if (H5S_PLIST == file_space_id[i]) {
            H5P_genplist_t *plist; 
            H5S_t          *space; 

            
            if (NULL == (plist = H5P_object_verify(dxpl_id, H5P_DATASET_XFER, true)))
                HGOTO_ERROR(H5E_DATASET, H5E_BADID, FAIL, "bad dataset transfer property list");

            
            if (H5P_peek(plist, H5D_XFER_DSET_IO_SEL_NAME, &space) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "error getting dataset I/O selection");

            
            dinfo[i].file_space = dinfo[i].dset->shared->space;

            
            if (H5S_SELECT_COPY(dinfo[i].file_space, space, true) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTCOPY, FAIL, "can't copy dataset I/O selection");
        } 
        else {
            
            if (NULL == (dinfo[i].file_space = (H5S_t *)H5I_object_verify(file_space_id[i], H5I_DATASPACE)))
                HGOTO_ERROR(H5E_DATASET, H5E_BADTYPE, FAIL, "file_space_id is not a dataspace ID");
        } 

        
        if (H5S_ALL == mem_space_id[i])
            dinfo[i].mem_space = dinfo[i].file_space;
        else if (H5S_BLOCK == mem_space_id[i]) {
            hsize_t nelmts; 

            
            nelmts = H5S_GET_SELECT_NPOINTS(dinfo[i].file_space);

            
            if (nelmts > 0) {
                
                if (NULL == (dinfo[i].mem_space = H5S_create_simple(1, &nelmts, NULL)))
                    HGOTO_ERROR(H5E_DATASET, H5E_CANTCREATE, FAIL,
                                "unable to create simple memory dataspace");
            } 
            else {
                
                if (NULL == (dinfo[i].mem_space = H5S_create(H5S_NULL)))
                    HGOTO_ERROR(H5E_DATASET, H5E_CANTCREATE, FAIL, "unable to create NULL memory dataspace");
            } 
        }     
        else if (H5S_PLIST == mem_space_id[i])
            HGOTO_ERROR(H5E_DATASET, H5E_BADTYPE, FAIL, "H5S_PLIST is not allowed for memory dataspace");
        else {
            
            if (NULL == (dinfo[i].mem_space = (H5S_t *)H5I_object_verify(mem_space_id[i], H5I_DATASPACE)))
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "mem_space_id is not a dataspace ID");
        } 

        
        if (H5S_SELECT_VALID(dinfo[i].file_space) != true)
            HGOTO_ERROR(H5E_DATASPACE, H5E_BADRANGE, FAIL,
                        "selection + offset not within extent for file dataspace");
        if (H5S_SELECT_VALID(dinfo[i].mem_space) != true)
            HGOTO_ERROR(H5E_DATASPACE, H5E_BADRANGE, FAIL,
                        "selection + offset not within extent for memory dataspace");

        
        dinfo[i].buf = buf[i];
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5VL__native_dataset_io_cleanup(size_t count, hid_t mem_space_id[], hid_t file_space_id[],
                                H5D_dset_io_info_t *dinfo)
{
    size_t i;
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(dinfo);

    
    for (i = 0; i < count; i++) {
        
        if (H5S_BLOCK == mem_space_id[i] && dinfo[i].mem_space)
            if (H5S_close(dinfo[i].mem_space) < 0)
                HDONE_ERROR(H5E_DATASET, H5E_CANTRELEASE, FAIL,
                            "unable to release temporary memory dataspace for H5S_BLOCK");

        
        if (H5S_PLIST == file_space_id[i] && dinfo[i].file_space)
            if (H5S_select_all(dinfo[i].file_space, true) < 0)
                HDONE_ERROR(H5E_DATASET, H5E_CANTRELEASE, FAIL,
                            "unable to release file dataspace selection for H5S_PLIST");
    }

    FUNC_LEAVE_NOAPI(ret_value)
} 

void *
H5VL__native_dataset_create(void *obj, const H5VL_loc_params_t *loc_params, const char *name, hid_t lcpl_id,
                            hid_t type_id, hid_t space_id, hid_t dcpl_id, hid_t dapl_id,
                            hid_t H5_ATTR_UNUSED dxpl_id, void H5_ATTR_UNUSED **req)
{
    H5G_loc_t    loc;         
    H5D_t       *dset = NULL; 
    const H5S_t *space;       
    void        *ret_value;

    FUNC_ENTER_PACKAGE

    
    if (H5G_loc_real(obj, loc_params->obj_type, &loc) < 0)
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "not a file or file object");
    if (H5I_DATATYPE != H5I_get_type(type_id))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "not a datatype ID");
    if (NULL == (space = (const H5S_t *)H5I_object_verify(space_id, H5I_DATASPACE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "not a dataspace ID");

    
    if (NULL == name) {
        
        if (NULL == (dset = H5D__create(loc.oloc->file, type_id, space, dcpl_id, dapl_id)))
            HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, NULL, "unable to create dataset");
    } 
    
    else {
        
        if (NULL == (dset = H5D__create_named(&loc, name, type_id, space, lcpl_id, dcpl_id, dapl_id)))
            HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, NULL, "unable to create dataset");
    } 

    ret_value = (void *)dset;

done:
    if (NULL == name) {
        
        if (dset) {
            H5O_loc_t *oloc; 

            
            if (NULL == (oloc = H5D_oloc(dset)))
                HDONE_ERROR(H5E_DATASET, H5E_CANTGET, NULL, "unable to get object location of dataset");

            
            if (H5O_dec_rc_by_loc(oloc) < 0)
                HDONE_ERROR(H5E_DATASET, H5E_CANTDEC, NULL,
                            "unable to decrement refcount on newly created object");
        } 
    }     

    FUNC_LEAVE_NOAPI(ret_value)
} 

void *
H5VL__native_dataset_open(void *obj, const H5VL_loc_params_t *loc_params, const char *name, hid_t dapl_id,
                          hid_t H5_ATTR_UNUSED dxpl_id, void H5_ATTR_UNUSED **req)
{
    H5D_t    *dset = NULL;
    H5G_loc_t loc; 
    void     *ret_value = NULL;

    FUNC_ENTER_PACKAGE

    if (H5G_loc_real(obj, loc_params->obj_type, &loc) < 0)
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "not a file or file object");

    
    if (NULL == (dset = H5D__open_name(&loc, name, dapl_id)))
        HGOTO_ERROR(H5E_DATASET, H5E_CANTOPENOBJ, NULL, "unable to open dataset");

    ret_value = (void *)dset;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5VL__native_dataset_read(size_t count, void *obj[], hid_t mem_type_id[], hid_t mem_space_id[],
                          hid_t file_space_id[], hid_t dxpl_id, void *buf[], void H5_ATTR_UNUSED **req)
{
    H5D_dset_io_info_t  dinfo_local;
    H5D_dset_io_info_t *dinfo     = &dinfo_local;
    herr_t              ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    if (count > 1)
        if (NULL == (dinfo = (H5D_dset_io_info_t *)H5MM_malloc(count * sizeof(H5D_dset_io_info_t))))
            HGOTO_ERROR(H5E_DATASET, H5E_CANTALLOC, FAIL, "couldn't allocate dset info array buffer");

    
    if (H5VL__native_dataset_io_setup(count, obj, mem_type_id, mem_space_id, file_space_id, dxpl_id,
                                      (H5_flexible_const_ptr_t *)buf, dinfo) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to set up file and memory dataspaces");

    
    H5CX_set_dxpl(dxpl_id);

    
    if (H5D__read(count, dinfo) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_READERROR, FAIL, "can't read data");

done:
    
    if (H5VL__native_dataset_io_cleanup(count, mem_space_id, file_space_id, dinfo) < 0)
        HDONE_ERROR(H5E_DATASET, H5E_CANTRELEASE, FAIL, "unable to release dataset info");

    if (dinfo != &dinfo_local)
        H5MM_xfree(dinfo);

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5VL__native_dataset_write(size_t count, void *obj[], hid_t mem_type_id[], hid_t mem_space_id[],
                           hid_t file_space_id[], hid_t dxpl_id, const void *buf[], void H5_ATTR_UNUSED **req)
{
    H5D_dset_io_info_t  dinfo_local;
    H5D_dset_io_info_t *dinfo     = &dinfo_local;
    herr_t              ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    if (count > 1)
        if (NULL == (dinfo = (H5D_dset_io_info_t *)H5MM_malloc(count * sizeof(H5D_dset_io_info_t))))
            HGOTO_ERROR(H5E_DATASET, H5E_CANTALLOC, FAIL, "couldn't allocate dset info array buffer");

    
    if (H5VL__native_dataset_io_setup(count, obj, mem_type_id, mem_space_id, file_space_id, dxpl_id,
                                      (H5_flexible_const_ptr_t *)buf, dinfo) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to set up file and memory dataspaces");

    
    H5CX_set_dxpl(dxpl_id);

    
    if (H5D__write(count, dinfo) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_WRITEERROR, FAIL, "can't write data");

done:
    
    if (H5VL__native_dataset_io_cleanup(count, mem_space_id, file_space_id, dinfo) < 0)
        HDONE_ERROR(H5E_DATASET, H5E_CANTRELEASE, FAIL, "unable to release dataset info");

    if (dinfo != &dinfo_local)
        H5MM_xfree(dinfo);

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5VL__native_dataset_get(void *obj, H5VL_dataset_get_args_t *args, hid_t H5_ATTR_UNUSED dxpl_id,
                         void H5_ATTR_UNUSED **req)
{
    H5D_t *dset      = (H5D_t *)obj;
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    switch (args->op_type) {
        
        case H5VL_DATASET_GET_SPACE: {
            if ((args->args.get_space.space_id = H5D__get_space(dset)) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't get space ID of dataset");

            break;
        }

        
        case H5VL_DATASET_GET_SPACE_STATUS: {
            if (H5D__get_space_status(dset, args->args.get_space_status.status) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "unable to get space status");

            break;
        }

        
        case H5VL_DATASET_GET_TYPE: {
            if ((args->args.get_type.type_id = H5D__get_type(dset)) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't get datatype ID of dataset");

            break;
        }

        
        case H5VL_DATASET_GET_DCPL: {
            if ((args->args.get_dcpl.dcpl_id = H5D_get_create_plist(dset)) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't get creation property list for dataset");

            break;
        }

        
        case H5VL_DATASET_GET_DAPL: {
            if ((args->args.get_dapl.dapl_id = H5D_get_access_plist(dset)) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't get access property list for dataset");

            break;
        }

        
        case H5VL_DATASET_GET_STORAGE_SIZE: {
            if (H5D__get_storage_size(dset, args->args.get_storage_size.storage_size) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't get size of dataset's storage");
            break;
        }

        default:
            HGOTO_ERROR(H5E_VOL, H5E_CANTGET, FAIL, "can't get this type of information from dataset");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5VL__native_dataset_specific(void *obj, H5VL_dataset_specific_args_t *args, hid_t H5_ATTR_UNUSED dxpl_id,
                              void H5_ATTR_UNUSED **req)
{
    H5D_t *dset      = (H5D_t *)obj;
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    switch (args->op_type) {
        
        case H5VL_DATASET_SET_EXTENT: {
            if (H5D__set_extent(dset, args->args.set_extent.size) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTSET, FAIL, "unable to set extent of dataset");
            break;
        }

        
        case H5VL_DATASET_FLUSH: {
            if (H5D__flush(dset, args->args.flush.dset_id) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTFLUSH, FAIL, "unable to flush dataset");

            break;
        }

        
        case H5VL_DATASET_REFRESH: {
            if (H5D__refresh(dset, args->args.refresh.dset_id) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTLOAD, FAIL, "unable to refresh dataset");

            break;
        }

        default:
            HGOTO_ERROR(H5E_VOL, H5E_UNSUPPORTED, FAIL, "invalid specific operation");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5VL__native_dataset_optional(void *obj, H5VL_optional_args_t *args, hid_t dxpl_id, void H5_ATTR_UNUSED **req)
{
    H5D_t                               *dset      = (H5D_t *)obj; 
    H5VL_native_dataset_optional_args_t *opt_args  = args->args; 
    herr_t                               ret_value = SUCCEED;    

    FUNC_ENTER_PACKAGE

    
    assert(dset);

    
    H5CX_set_dxpl(dxpl_id);

    switch (args->op_type) {
        
        case H5VL_NATIVE_DATASET_FORMAT_CONVERT: {
            switch (dset->shared->layout.type) {
                case H5D_CHUNKED:
                    
                    if (dset->shared->layout.u.chunk.idx_type != H5D_CHUNK_IDX_BTREE)
                        if (H5D__format_convert(dset) < 0)
                            HGOTO_ERROR(H5E_DATASET, H5E_CANTLOAD, FAIL,
                                        "unable to downgrade chunk indexing type for dataset");
                    break;

                case H5D_CONTIGUOUS:
                case H5D_COMPACT:
                    
                    if (dset->shared->layout.version > H5O_LAYOUT_VERSION_DEFAULT)
                        if (H5D__format_convert(dset) < 0)
                            HGOTO_ERROR(H5E_DATASET, H5E_CANTLOAD, FAIL,
                                        "unable to downgrade layout version for dataset");
                    break;

                case H5D_VIRTUAL:
                    
                    break;

                case H5D_LAYOUT_ERROR:
                case H5D_NLAYOUTS:
                    HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid dataset layout type");

                default:
                    HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "unknown dataset layout type");
            } 

            break;
        }

        
        case H5VL_NATIVE_DATASET_GET_CHUNK_INDEX_TYPE: {
            
            if (H5D_CHUNKED != dset->shared->layout.type)
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a chunked dataset");

            
            *opt_args->get_chunk_idx_type.idx_type = dset->shared->layout.u.chunk.idx_type;

            break;
        }

        
        case H5VL_NATIVE_DATASET_GET_CHUNK_STORAGE_SIZE: {
            H5VL_native_dataset_get_chunk_storage_size_t *gcss_args = &opt_args->get_chunk_storage_size;

            
            if (H5D_CHUNKED != dset->shared->layout.type)
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a chunked dataset");

            
            if (H5D__get_chunk_storage_size(dset, gcss_args->offset, gcss_args->size) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't get storage size of chunk");

            break;
        }

        
        case H5VL_NATIVE_DATASET_GET_NUM_CHUNKS: {
            H5VL_native_dataset_get_num_chunks_t *gnc_args = &opt_args->get_num_chunks;
            const H5S_t                          *space    = NULL;

            assert(dset->shared);
            assert(dset->shared->space);

            
            if (gnc_args->space_id == H5S_ALL)
                space = dset->shared->space;
            else 
                if (NULL == (space = (const H5S_t *)H5I_object_verify(gnc_args->space_id, H5I_DATASPACE)))
                    HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a valid dataspace ID");

            
            if (H5D_CHUNKED != dset->shared->layout.type)
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a chunked dataset");

            
            if (H5D__get_num_chunks(dset, space, gnc_args->nchunks) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't get number of chunks");

            break;
        }

        
        case H5VL_NATIVE_DATASET_GET_CHUNK_INFO_BY_IDX: {
            H5VL_native_dataset_get_chunk_info_by_idx_t *gcibi_args = &opt_args->get_chunk_info_by_idx;
            const H5S_t                                 *space;

            assert(dset->shared);
            assert(dset->shared->space);

            
            if (gcibi_args->space_id == H5S_ALL)
                space = dset->shared->space;
            else 
                if (NULL == (space = (const H5S_t *)H5I_object_verify(gcibi_args->space_id, H5I_DATASPACE)))
                    HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a valid dataspace ID");

            
            if (H5D_CHUNKED != dset->shared->layout.type)
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a chunked dataset");

            
            if (H5D__get_chunk_info(dset, space, gcibi_args->chk_index, gcibi_args->offset,
                                    gcibi_args->filter_mask, gcibi_args->addr, gcibi_args->size) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't get chunk info by index");

            break;
        }

        
        case H5VL_NATIVE_DATASET_GET_CHUNK_INFO_BY_COORD: {
            H5VL_native_dataset_get_chunk_info_by_coord_t *gcibc_args = &opt_args->get_chunk_info_by_coord;

            assert(dset->shared);

            
            if (H5D_CHUNKED != dset->shared->layout.type)
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a chunked dataset");

            
            if (H5D__get_chunk_info_by_coord(dset, gcibc_args->offset, gcibc_args->filter_mask,
                                             gcibc_args->addr, gcibc_args->size) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL,
                            "can't get chunk info by its logical coordinates");

            break;
        }

        
        case H5VL_NATIVE_DATASET_CHUNK_READ: {
            H5VL_native_dataset_chunk_read_t *chunk_read_args = &opt_args->chunk_read;
            hsize_t offset_copy[H5O_LAYOUT_NDIMS]; 

            
            if (NULL == dset->oloc.file)
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "dataset is not associated with a file");
            if (H5D_CHUNKED != dset->shared->layout.type)
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a chunked dataset");

            
            if (H5D__chunk_get_offset_copy(dset, chunk_read_args->offset, offset_copy) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTCOPY, FAIL, "failure to copy offset array");

            
            if (H5D__chunk_direct_read(dset, offset_copy, &chunk_read_args->filters, chunk_read_args->buf,
                                       chunk_read_args->buf_size) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_READERROR, FAIL, "can't read unprocessed chunk data");

            break;
        }

        
        case H5VL_NATIVE_DATASET_CHUNK_WRITE: {
            H5VL_native_dataset_chunk_write_t *chunk_write_args = &opt_args->chunk_write;
            hsize_t offset_copy[H5O_LAYOUT_NDIMS]; 

            
            if (NULL == dset->oloc.file)
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "dataset is not associated with a file");
            if (H5D_CHUNKED != dset->shared->layout.type)
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a chunked dataset");

            
            if (H5D__chunk_get_offset_copy(dset, chunk_write_args->offset, offset_copy) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTCOPY, FAIL, "failure to copy offset array");

            
            if (H5D__chunk_direct_write(dset, chunk_write_args->filters, offset_copy, chunk_write_args->size,
                                        chunk_write_args->buf) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_WRITEERROR, FAIL, "can't write unprocessed chunk data");

            break;
        }

        
        case H5VL_NATIVE_DATASET_GET_VLEN_BUF_SIZE: {
            H5VL_native_dataset_get_vlen_buf_size_t *gvbs_args = &opt_args->get_vlen_buf_size;

            if (H5D__vlen_get_buf_size(dset, gvbs_args->type_id, gvbs_args->space_id, gvbs_args->size) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTGET, FAIL, "can't get size of vlen buf needed");
            break;
        }

        
        case H5VL_NATIVE_DATASET_GET_OFFSET: {
            
            *opt_args->get_offset.offset = H5D__get_offset(dset);

            break;
        }

        
        case H5VL_NATIVE_DATASET_CHUNK_ITER: {
            
            assert(dset->shared);

            
            if (H5D_CHUNKED != dset->shared->layout.type)
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a chunked dataset");

            
            if ((ret_value = H5D__chunk_iter(dset, opt_args->chunk_iter.op, opt_args->chunk_iter.op_data)) <
                0)
                HERROR(H5E_DATASET, H5E_BADITER, "chunk iteration failed");

            break;
        }

        default:
            HGOTO_ERROR(H5E_VOL, H5E_UNSUPPORTED, FAIL, "invalid optional operation");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5VL__native_dataset_close(void *dset, hid_t H5_ATTR_UNUSED dxpl_id, void H5_ATTR_UNUSED **req)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    if (H5D_close((H5D_t *)dset) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTDEC, FAIL, "can't close dataset");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
