/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#define H5F_FRIEND 
#define H5O_FRIEND 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5Fpkg.h"      
#include "H5Gprivate.h"  
#include "H5Iprivate.h"  
#include "H5Opkg.h"      
#include "H5VLprivate.h" 

#include "H5VLnative_private.h" 

void *
H5VL__native_object_open(void *obj, const H5VL_loc_params_t *loc_params, H5I_type_t *opened_type,
                         hid_t H5_ATTR_UNUSED dxpl_id, void H5_ATTR_UNUSED **req)
{
    H5G_loc_t loc;
    void     *ret_value = NULL;

    FUNC_ENTER_PACKAGE

    if (H5G_loc_real(obj, loc_params->obj_type, &loc) < 0)
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "not a file or file object");

    switch (loc_params->type) {
        case H5VL_OBJECT_BY_NAME: {
            
            if (NULL == (ret_value = H5O_open_name(&loc, loc_params->loc_data.loc_by_name.name, opened_type)))
                HGOTO_ERROR(H5E_OHDR, H5E_CANTOPENOBJ, NULL, "unable to open object by name");
            break;
        }

        case H5VL_OBJECT_BY_IDX: {
            
            if (NULL == (ret_value = H5O__open_by_idx(&loc, loc_params->loc_data.loc_by_idx.name,
                                                      loc_params->loc_data.loc_by_idx.idx_type,
                                                      loc_params->loc_data.loc_by_idx.order,
                                                      loc_params->loc_data.loc_by_idx.n, opened_type)))
                HGOTO_ERROR(H5E_OHDR, H5E_CANTOPENOBJ, NULL, "unable to open object by index");
            break;
        }

        case H5VL_OBJECT_BY_TOKEN: {
            H5O_token_t token = *loc_params->loc_data.loc_by_token.token;
            haddr_t     addr;

            
            if (H5VL_native_token_to_addr(loc.oloc->file, H5I_FILE, token, &addr) < 0)
                HGOTO_ERROR(H5E_OHDR, H5E_CANTUNSERIALIZE, NULL,
                            "can't deserialize object token into address");

            
            if (NULL == (ret_value = H5O__open_by_addr(&loc, addr, opened_type)))
                HGOTO_ERROR(H5E_OHDR, H5E_CANTOPENOBJ, NULL, "unable to open object by address");
            break;
        }

        case H5VL_OBJECT_BY_SELF:
        default:
            HGOTO_ERROR(H5E_VOL, H5E_UNSUPPORTED, NULL, "unknown open parameters");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5VL__native_object_copy(void *src_obj, const H5VL_loc_params_t *loc_params1, const char *src_name,
                         void *dst_obj, const H5VL_loc_params_t *loc_params2, const char *dst_name,
                         hid_t ocpypl_id, hid_t lcpl_id, hid_t H5_ATTR_UNUSED dxpl_id,
                         void H5_ATTR_UNUSED **req)
{
    H5G_loc_t src_loc; 
    H5G_loc_t dst_loc; 
    herr_t    ret_value = FAIL;

    FUNC_ENTER_PACKAGE

    
    if (H5G_loc_real(src_obj, loc_params1->obj_type, &src_loc) < 0)
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a file or file object");
    if (H5G_loc_real(dst_obj, loc_params2->obj_type, &dst_loc) < 0)
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a file or file object");

    
    if ((ret_value = H5O__copy(&src_loc, src_name, &dst_loc, dst_name, ocpypl_id, lcpl_id)) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTCOPY, FAIL, "unable to copy object");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5VL__native_object_get(void *obj, const H5VL_loc_params_t *loc_params, H5VL_object_get_args_t *args,
                        hid_t H5_ATTR_UNUSED dxpl_id, void H5_ATTR_UNUSED **req)
{
    herr_t    ret_value = SUCCEED; 
    H5G_loc_t loc;                 

    FUNC_ENTER_PACKAGE

    if (H5G_loc_real(obj, loc_params->obj_type, &loc) < 0)
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a file or file object");

    switch (args->op_type) {

        
        case H5VL_OBJECT_GET_FILE: {
            if (loc_params->type == H5VL_OBJECT_BY_SELF) {
                *args->args.get_file.file = (void *)loc.oloc->file;

                
                loc.oloc->file->id_exists = true;
            }
            else
                HGOTO_ERROR(H5E_VOL, H5E_UNSUPPORTED, FAIL, "unknown get_file parameters");

            break;
        }

        
        case H5VL_OBJECT_GET_NAME: {
            if (loc_params->type == H5VL_OBJECT_BY_SELF) {
                
                if (H5G_get_name(&loc, args->args.get_name.buf, args->args.get_name.buf_size,
                                 args->args.get_name.name_len, NULL) < 0)
                    HGOTO_ERROR(H5E_VOL, H5E_CANTGET, FAIL, "can't retrieve object name");
            } 
            else if (loc_params->type == H5VL_OBJECT_BY_TOKEN) {
                H5O_loc_t   obj_oloc; 
                H5O_token_t token = *loc_params->loc_data.loc_by_token.token;

                
                H5O_loc_reset(&obj_oloc);
                obj_oloc.file = loc.oloc->file;

                
                if (H5VL_native_token_to_addr(obj_oloc.file, H5I_FILE, token, &obj_oloc.addr) < 0)
                    HGOTO_ERROR(H5E_OHDR, H5E_CANTUNSERIALIZE, FAIL,
                                "can't deserialize object token into address");

                
                if (H5G_get_name_by_addr(loc.oloc->file, &obj_oloc, args->args.get_name.buf,
                                         args->args.get_name.buf_size, args->args.get_name.name_len) < 0)
                    HGOTO_ERROR(H5E_VOL, H5E_CANTGET, FAIL, "can't determine object name");
            } 
            else
                HGOTO_ERROR(H5E_VOL, H5E_UNSUPPORTED, FAIL, "unknown get_name parameters");

            break;
        }

        
        case H5VL_OBJECT_GET_TYPE: {
            if (loc_params->type == H5VL_OBJECT_BY_TOKEN) {
                H5O_loc_t   obj_oloc; 
                unsigned    rc;       
                H5O_token_t token = *loc_params->loc_data.loc_by_token.token;

                
                H5O_loc_reset(&obj_oloc);
                obj_oloc.file = loc.oloc->file;

                
                if (H5VL_native_token_to_addr(obj_oloc.file, H5I_FILE, token, &obj_oloc.addr) < 0)
                    HGOTO_ERROR(H5E_OHDR, H5E_CANTUNSERIALIZE, FAIL,
                                "can't deserialize object token into address");

                
                
                if (H5O_get_rc_and_type(&obj_oloc, &rc, args->args.get_type.obj_type) < 0 || 0 == rc)
                    HGOTO_ERROR(H5E_REFERENCE, H5E_LINKCOUNT, FAIL, "dereferencing deleted object");
            }
            else
                HGOTO_ERROR(H5E_VOL, H5E_UNSUPPORTED, FAIL, "unknown get_type parameters");

            break;
        }

        
        case H5VL_OBJECT_GET_INFO: {
            if (loc_params->type == H5VL_OBJECT_BY_SELF) {
                if (H5G_loc_info(&loc, ".", args->args.get_info.oinfo, args->args.get_info.fields) < 0)
                    HGOTO_ERROR(H5E_OHDR, H5E_NOTFOUND, FAIL, "object not found");
            } 
            else if (loc_params->type == H5VL_OBJECT_BY_NAME) {
                if (H5G_loc_info(&loc, loc_params->loc_data.loc_by_name.name, args->args.get_info.oinfo,
                                 args->args.get_info.fields) < 0)
                    HGOTO_ERROR(H5E_OHDR, H5E_NOTFOUND, FAIL, "object not found");
            } 
            else if (loc_params->type == H5VL_OBJECT_BY_IDX) {
                H5G_loc_t  obj_loc;  
                H5G_name_t obj_path; 
                H5O_loc_t  obj_oloc; 

                
                obj_loc.oloc = &obj_oloc;
                obj_loc.path = &obj_path;
                H5G_loc_reset(&obj_loc);

                
                if (H5G_loc_find_by_idx(&loc, loc_params->loc_data.loc_by_idx.name,
                                        loc_params->loc_data.loc_by_idx.idx_type,
                                        loc_params->loc_data.loc_by_idx.order,
                                        loc_params->loc_data.loc_by_idx.n, &obj_loc ) < 0)
                    HGOTO_ERROR(H5E_OHDR, H5E_NOTFOUND, FAIL, "group not found");

                
                if (H5O_get_info(obj_loc.oloc, args->args.get_info.oinfo, args->args.get_info.fields) < 0) {
                    H5G_loc_free(&obj_loc);
                    HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, FAIL, "can't retrieve object info");
                } 

                
                if (H5G_loc_free(&obj_loc) < 0)
                    HGOTO_ERROR(H5E_OHDR, H5E_CANTRELEASE, FAIL, "can't free location");
            } 
            else
                HGOTO_ERROR(H5E_OHDR, H5E_UNSUPPORTED, FAIL, "unknown get info parameters");

            break;
        }

        default:
            HGOTO_ERROR(H5E_VOL, H5E_CANTGET, FAIL, "can't get this type of information from object");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5VL__native_object_specific(void *obj, const H5VL_loc_params_t *loc_params,
                             H5VL_object_specific_args_t *args, hid_t H5_ATTR_UNUSED dxpl_id,
                             void H5_ATTR_UNUSED **req)
{
    H5G_loc_t loc;
    herr_t    ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    if (H5G_loc_real(obj, loc_params->obj_type, &loc) < 0)
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a file or file object");

    switch (args->op_type) {
        
        case H5VL_OBJECT_CHANGE_REF_COUNT: {
            if (H5O_link(loc.oloc, args->args.change_rc.delta) < 0)
                HGOTO_ERROR(H5E_OHDR, H5E_LINKCOUNT, FAIL, "modifying object link count failed");

            break;
        }

        
        case H5VL_OBJECT_EXISTS: {
            if (loc_params->type == H5VL_OBJECT_BY_NAME) {
                
                if (H5G_loc_exists(&loc, loc_params->loc_data.loc_by_name.name, args->args.exists.exists) < 0)
                    HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, FAIL, "unable to determine if '%s' exists",
                                loc_params->loc_data.loc_by_name.name);
            } 
            else
                HGOTO_ERROR(H5E_VOL, H5E_UNSUPPORTED, FAIL, "unknown object exists parameters");

            break;
        }

        
        case H5VL_OBJECT_LOOKUP: {
            if (loc_params->type == H5VL_OBJECT_BY_NAME) {
                H5G_loc_t  obj_loc;  
                H5G_name_t obj_path; 
                H5O_loc_t  obj_oloc; 

                
                obj_loc.oloc = &obj_oloc;
                obj_loc.path = &obj_path;
                H5G_loc_reset(&obj_loc);

                
                if (H5G_loc_find(&loc, loc_params->loc_data.loc_by_name.name, &obj_loc) < 0)
                    HGOTO_ERROR(H5E_OHDR, H5E_NOTFOUND, FAIL, "object not found");

                
                if (H5VL_native_addr_to_token(loc.oloc->file, H5I_FILE, obj_loc.oloc->addr,
                                              args->args.lookup.token_ptr) < 0)
                    HGOTO_ERROR(H5E_OHDR, H5E_CANTSERIALIZE, FAIL,
                                "can't serialize address into object token");

                
                if (H5G_loc_free(&obj_loc) < 0)
                    HGOTO_ERROR(H5E_OHDR, H5E_CANTRELEASE, FAIL, "can't free location");
            } 
            else
                HGOTO_ERROR(H5E_VOL, H5E_UNSUPPORTED, FAIL, "unknown object exists parameters");
            break;
        }

        
        case H5VL_OBJECT_VISIT: {
            H5VL_object_visit_args_t *visit_args = &args->args.visit;

            
            if (loc_params->type == H5VL_OBJECT_BY_SELF) {
                if ((ret_value = H5O__visit(&loc, ".", visit_args->idx_type, visit_args->order,
                                            visit_args->op, visit_args->op_data, visit_args->fields)) < 0)
                    HGOTO_ERROR(H5E_OHDR, H5E_BADITER, FAIL, "object visitation failed");
            } 
            else if (loc_params->type == H5VL_OBJECT_BY_NAME) {
                if ((ret_value = H5O__visit(&loc, loc_params->loc_data.loc_by_name.name, visit_args->idx_type,
                                            visit_args->order, visit_args->op, visit_args->op_data,
                                            visit_args->fields)) < 0)
                    HGOTO_ERROR(H5E_OHDR, H5E_BADITER, FAIL, "object visitation failed");
            } 
            else
                HGOTO_ERROR(H5E_VOL, H5E_UNSUPPORTED, FAIL, "unknown object visit params");

            break;
        }

        case H5VL_OBJECT_FLUSH: {
            
            if (H5O_flush(loc.oloc, args->args.flush.obj_id) < 0)
                HGOTO_ERROR(H5E_OHDR, H5E_CANTFLUSH, FAIL, "unable to flush object");

            break;
        }

        case H5VL_OBJECT_REFRESH: {
            
            if (H5O_refresh_metadata(loc.oloc, args->args.refresh.obj_id) < 0)
                HGOTO_ERROR(H5E_OHDR, H5E_CANTLOAD, FAIL, "unable to refresh object");

            break;
        }

        default:
            HGOTO_ERROR(H5E_VOL, H5E_CANTGET, FAIL, "can't recognize this operation type");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5VL__native_object_optional(void *obj, const H5VL_loc_params_t *loc_params, H5VL_optional_args_t *args,
                             hid_t H5_ATTR_UNUSED dxpl_id, void H5_ATTR_UNUSED **req)
{
    H5G_loc_t                           loc;                    
    H5VL_native_object_optional_args_t *opt_args  = args->args; 
    herr_t                              ret_value = SUCCEED;    

    FUNC_ENTER_PACKAGE

    if (H5G_loc_real(obj, loc_params->obj_type, &loc) < 0)
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a file or file object");

    switch (args->op_type) {
        
        case H5VL_NATIVE_OBJECT_GET_COMMENT: {
            H5VL_native_object_get_comment_t *gc_args = &opt_args->get_comment;

            
            if (loc_params->type == H5VL_OBJECT_BY_SELF) { 
                if (H5G_loc_get_comment(&loc, ".", gc_args->buf, gc_args->buf_size, gc_args->comment_len) < 0)
                    HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, FAIL, "can't get comment for object");
            }                                                   
            else if (loc_params->type == H5VL_OBJECT_BY_NAME) { 
                if (H5G_loc_get_comment(&loc, loc_params->loc_data.loc_by_name.name, gc_args->buf,
                                        gc_args->buf_size, gc_args->comment_len) < 0)
                    HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, FAIL, "can't get comment for object");
            } 
            else
                HGOTO_ERROR(H5E_VOL, H5E_UNSUPPORTED, FAIL, "unknown set_coment parameters");

            break;
        }

        
        case H5VL_NATIVE_OBJECT_SET_COMMENT: {
            if (loc_params->type == H5VL_OBJECT_BY_SELF) { 
                if (H5G_loc_set_comment(&loc, ".", opt_args->set_comment.comment) < 0)
                    HGOTO_ERROR(H5E_OHDR, H5E_NOTFOUND, FAIL, "object not found");
            }                                                   
            else if (loc_params->type == H5VL_OBJECT_BY_NAME) { 
                if (H5G_loc_set_comment(&loc, loc_params->loc_data.loc_by_name.name,
                                        opt_args->set_comment.comment) < 0)
                    HGOTO_ERROR(H5E_OHDR, H5E_NOTFOUND, FAIL, "object not found");
            } 
            else
                HGOTO_ERROR(H5E_VOL, H5E_UNSUPPORTED, FAIL, "unknown set_coment parameters");

            break;
        }

        
        case H5VL_NATIVE_OBJECT_DISABLE_MDC_FLUSHES: {
            if (H5O__disable_mdc_flushes(loc.oloc) < 0)
                HGOTO_ERROR(H5E_OHDR, H5E_CANTCORK, FAIL, "unable to cork the metadata cache");

            break;
        }

        
        case H5VL_NATIVE_OBJECT_ENABLE_MDC_FLUSHES: {
            if (H5O__enable_mdc_flushes(loc.oloc) < 0)
                HGOTO_ERROR(H5E_OHDR, H5E_CANTUNCORK, FAIL, "unable to uncork the metadata cache");

            break;
        }

        
        case H5VL_NATIVE_OBJECT_ARE_MDC_FLUSHES_DISABLED: {
            if (H5O__are_mdc_flushes_disabled(loc.oloc, opt_args->are_mdc_flushes_disabled.flag) < 0)
                HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, FAIL, "unable to determine metadata cache cork status");

            break;
        }

        
        case H5VL_NATIVE_OBJECT_GET_NATIVE_INFO: {
            H5VL_native_object_get_native_info_t *gni_args = &opt_args->get_native_info;

            
            if (loc_params->type == H5VL_OBJECT_BY_SELF) {
                if (H5G_loc_native_info(&loc, ".", gni_args->ninfo, gni_args->fields) < 0)
                    HGOTO_ERROR(H5E_OHDR, H5E_NOTFOUND, FAIL, "object not found");
            }                                                   
            else if (loc_params->type == H5VL_OBJECT_BY_NAME) { 
                if (H5G_loc_native_info(&loc, loc_params->loc_data.loc_by_name.name, gni_args->ninfo,
                                        gni_args->fields) < 0)
                    HGOTO_ERROR(H5E_OHDR, H5E_NOTFOUND, FAIL, "object not found");
            }                                                  
            else if (loc_params->type == H5VL_OBJECT_BY_IDX) { 
                H5G_loc_t  obj_loc;                            
                H5G_name_t obj_path;                           
                H5O_loc_t  obj_oloc;                           

                
                obj_loc.oloc = &obj_oloc;
                obj_loc.path = &obj_path;
                H5G_loc_reset(&obj_loc);

                
                if (H5G_loc_find_by_idx(&loc, loc_params->loc_data.loc_by_idx.name,
                                        loc_params->loc_data.loc_by_idx.idx_type,
                                        loc_params->loc_data.loc_by_idx.order,
                                        loc_params->loc_data.loc_by_idx.n, &obj_loc ) < 0)
                    HGOTO_ERROR(H5E_OHDR, H5E_NOTFOUND, FAIL, "group not found");

                if (H5O_get_native_info(obj_loc.oloc, gni_args->ninfo, gni_args->fields) < 0) {
                    H5G_loc_free(&obj_loc);
                    HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, FAIL, "can't retrieve object info");
                } 

                
                if (H5G_loc_free(&obj_loc) < 0)
                    HGOTO_ERROR(H5E_OHDR, H5E_CANTRELEASE, FAIL, "can't free location");
            } 
            else
                HGOTO_ERROR(H5E_OHDR, H5E_UNSUPPORTED, FAIL, "unknown get info parameters");

            break;
        }

        default:
            HGOTO_ERROR(H5E_VOL, H5E_CANTGET, FAIL, "can't perform this operation on object");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
