/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Omodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5FLprivate.h" 
#include "H5Opkg.h"      

H5FL_DEFINE(H5O_chunk_proxy_t);

herr_t
H5O__chunk_add(H5F_t *f, H5O_t *oh, unsigned idx, unsigned cont_chunkno)
{
    H5O_chunk_proxy_t *chk_proxy = NULL; 
    H5O_chunk_proxy_t *cont_chk_proxy =
        NULL; 
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE_TAG(oh->cache_info.addr)

    
    assert(f);
    assert(oh);
    assert(idx < oh->nchunks);
    assert(idx > 0);

    
    if (NULL == (chk_proxy = H5FL_CALLOC(H5O_chunk_proxy_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "memory allocation failed");

    
    if (H5O__inc_rc(oh) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTINC, FAIL, "can't increment reference count on object header");

    
    chk_proxy->f       = f;
    chk_proxy->oh      = oh;
    chk_proxy->chunkno = idx;

    
    if (cont_chunkno != 0) {
        if (NULL == (cont_chk_proxy = H5O__chunk_protect(f, oh, cont_chunkno)))
            HGOTO_ERROR(H5E_OHDR, H5E_CANTPROTECT, FAIL, "unable to load object header chunk");
        chk_proxy->fd_parent = cont_chk_proxy;
    } 

    
    if (H5AC_insert_entry(f, H5AC_OHDR_CHK, oh->chunk[idx].addr, chk_proxy, H5AC__NO_FLAGS_SET) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTINSERT, FAIL, "unable to cache object header chunk");

    chk_proxy = NULL;

done:
    
    if (ret_value < 0)
        if (chk_proxy && H5O__chunk_dest(chk_proxy) < 0)
            HDONE_ERROR(H5E_OHDR, H5E_CANTRELEASE, FAIL, "unable to destroy object header chunk");

    
    if (cont_chk_proxy)
        if (H5O__chunk_unprotect(f, cont_chk_proxy, false) < 0)
            HDONE_ERROR(H5E_OHDR, H5E_CANTUNPROTECT, FAIL, "unable to unprotect object header chunk");

    FUNC_LEAVE_NOAPI_TAG(ret_value)
} 

H5O_chunk_proxy_t *
H5O__chunk_protect(H5F_t *f, H5O_t *oh, unsigned idx)
{
    H5O_chunk_proxy_t *chk_proxy = NULL; 
    H5O_chunk_proxy_t *ret_value = NULL; 

    FUNC_ENTER_PACKAGE_TAG(oh->cache_info.addr)

    
    assert(f);
    assert(oh);
    assert(idx < oh->nchunks);

    
    if (0 == idx) {
        
        
        if (NULL == (chk_proxy = H5FL_CALLOC(H5O_chunk_proxy_t)))
            HGOTO_ERROR(H5E_OHDR, H5E_CANTALLOC, NULL, "memory allocation failed");

        
        if (H5O__inc_rc(oh) < 0)
            HGOTO_ERROR(H5E_OHDR, H5E_CANTINC, NULL, "can't increment reference count on object header");

        
        chk_proxy->f       = f;
        chk_proxy->oh      = oh;
        chk_proxy->chunkno = idx;
    } 
    else {
        H5O_chk_cache_ud_t chk_udata; 

        
        
        memset(&chk_udata, 0, sizeof(chk_udata));
        chk_udata.oh      = oh;
        chk_udata.chunkno = idx;
        chk_udata.size    = oh->chunk[idx].size;

        
        if (NULL == (chk_proxy = (H5O_chunk_proxy_t *)H5AC_protect(f, H5AC_OHDR_CHK, oh->chunk[idx].addr,
                                                                   &chk_udata, H5AC__NO_FLAGS_SET)))
            HGOTO_ERROR(H5E_OHDR, H5E_CANTPROTECT, NULL, "unable to load object header chunk");

        
        assert(chk_proxy->oh == oh);
        assert(chk_proxy->chunkno == idx);
    } 

    
    ret_value = chk_proxy;

done:
    
    if (!ret_value)
        if (0 == idx && chk_proxy && H5O__chunk_dest(chk_proxy) < 0)
            HDONE_ERROR(H5E_OHDR, H5E_CANTRELEASE, NULL, "unable to destroy object header chunk");

    FUNC_LEAVE_NOAPI_TAG(ret_value)
} 

herr_t
H5O__chunk_unprotect(H5F_t *f, H5O_chunk_proxy_t *chk_proxy, bool dirtied)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(chk_proxy);

    
    if (0 == chk_proxy->chunkno) {
        
        if (dirtied) {
            
            if (H5AC_mark_entry_dirty(chk_proxy->oh) < 0)
                HGOTO_ERROR(H5E_OHDR, H5E_CANTMARKDIRTY, FAIL, "unable to mark object header as dirty");
        } 

        
        if (H5O__dec_rc(chk_proxy->oh) < 0)
            HGOTO_ERROR(H5E_OHDR, H5E_CANTDEC, FAIL, "can't decrement reference count on object header");

        
        chk_proxy = H5FL_FREE(H5O_chunk_proxy_t, chk_proxy);
    } 
    else {
        
        if (H5AC_unprotect(f, H5AC_OHDR_CHK, chk_proxy->oh->chunk[chk_proxy->chunkno].addr, chk_proxy,
                           (dirtied ? H5AC__DIRTIED_FLAG : H5AC__NO_FLAGS_SET)) < 0)
            HGOTO_ERROR(H5E_OHDR, H5E_CANTUNPROTECT, FAIL, "unable to release object header chunk");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5O__chunk_resize(H5O_t *oh, H5O_chunk_proxy_t *chk_proxy)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(oh);
    assert(chk_proxy);

    
    if (0 == chk_proxy->chunkno) {
        
        if (H5AC_resize_entry(oh, oh->chunk[0].size) < 0)
            HGOTO_ERROR(H5E_OHDR, H5E_CANTRESIZE, FAIL, "unable to resize chunk in cache");
    } 
    else {
        
        if (H5AC_resize_entry(chk_proxy, oh->chunk[chk_proxy->chunkno].size) < 0)
            HGOTO_ERROR(H5E_OHDR, H5E_CANTRESIZE, FAIL, "unable to resize chunk in cache");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5O__chunk_update_idx(H5F_t *f, H5O_t *oh, unsigned idx)
{
    H5O_chunk_proxy_t *chk_proxy = NULL;    
    H5O_chk_cache_ud_t chk_udata;           
    herr_t             ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE_TAG(oh->cache_info.addr)

    
    assert(f);
    assert(oh);
    assert(idx < oh->nchunks);
    assert(idx > 0);

    
    
    memset(&chk_udata, 0, sizeof(chk_udata));
    chk_udata.oh      = oh;
    chk_udata.chunkno = idx;
    chk_udata.size    = oh->chunk[idx].size;

    
    if (NULL == (chk_proxy = (H5O_chunk_proxy_t *)H5AC_protect(f, H5AC_OHDR_CHK, oh->chunk[idx].addr,
                                                               &chk_udata, H5AC__NO_FLAGS_SET)))
        HGOTO_ERROR(H5E_OHDR, H5E_CANTPROTECT, FAIL, "unable to load object header chunk");

    
    chk_proxy->chunkno = idx;

    
    if (H5AC_unprotect(f, H5AC_OHDR_CHK, oh->chunk[idx].addr, chk_proxy, H5AC__DIRTIED_FLAG) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTUNPROTECT, FAIL, "unable to release object header chunk");

done:
    FUNC_LEAVE_NOAPI_TAG(ret_value)
} 

herr_t
H5O__chunk_delete(H5F_t *f, H5O_t *oh, unsigned idx)
{
    H5O_chunk_proxy_t *chk_proxy   = NULL;               
    unsigned           cache_flags = H5AC__DELETED_FLAG; 
    herr_t             ret_value   = SUCCEED;            

    FUNC_ENTER_PACKAGE_TAG(oh->cache_info.addr)

    
    assert(f);
    assert(oh);
    assert(idx < oh->nchunks);
    assert(idx > 0);

    
    if (NULL == (chk_proxy = H5O__chunk_protect(f, oh, idx)))
        HGOTO_ERROR(H5E_OHDR, H5E_CANTPROTECT, FAIL, "unable to load object header chunk");

    
    if (!oh->swmr_write)
        cache_flags |= H5AC__DIRTIED_FLAG | H5AC__FREE_FILE_SPACE_FLAG;

done:
    
    if (chk_proxy && H5AC_unprotect(f, H5AC_OHDR_CHK, oh->chunk[idx].addr, chk_proxy, cache_flags) < 0)
        HDONE_ERROR(H5E_OHDR, H5E_CANTUNPROTECT, FAIL, "unable to release object header chunk");

    FUNC_LEAVE_NOAPI_TAG(ret_value)
} 

herr_t
H5O__chunk_dest(H5O_chunk_proxy_t *chk_proxy)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(chk_proxy);

    
    if (H5O__dec_rc(chk_proxy->oh) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTDEC, FAIL, "can't decrement reference count on object header");

done:
    
    chk_proxy = H5FL_FREE(H5O_chunk_proxy_t, chk_proxy);

    FUNC_LEAVE_NOAPI(ret_value)
} 
