/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5FDmodule.h" 

#include "H5private.h"   
#include "H5CXprivate.h" 
#include "H5Eprivate.h"  
#include "H5Fprivate.h"  
#include "H5FDpkg.h"     
#include "H5FDmulti.h"   
#include "H5FLprivate.h" 

H5FL_DEFINE(H5FD_free_t);

static haddr_t
H5FD__extend(H5FD_t *file, H5FD_mem_t type, hsize_t size)
{
    haddr_t eoa;                     
    herr_t  status;                  
    haddr_t ret_value = HADDR_UNDEF; 

    FUNC_ENTER_PACKAGE

    
    assert(file);
    assert(file->cls);
    assert(type >= H5FD_MEM_DEFAULT && type < H5FD_MEM_NTYPES);
    assert(size > 0);

    
    H5_BEFORE_USER_CB(HADDR_UNDEF)
        {
            
            eoa = (file->cls->get_eoa)(file, type);
        }
    H5_AFTER_USER_CB(HADDR_UNDEF)

    
    if (H5_addr_overflow(eoa, size) || (eoa + size) > file->maxaddr)
        HGOTO_ERROR(H5E_VFL, H5E_NOSPACE, HADDR_UNDEF, "file allocation request failed");

    
    ret_value = eoa;

    
    H5_BEFORE_USER_CB(HADDR_UNDEF)
        {
            
            eoa += size;
            status = (file->cls->set_eoa)(file, type, eoa);
        }
    H5_AFTER_USER_CB(HADDR_UNDEF)
    if (status < 0)
        HGOTO_ERROR(H5E_VFL, H5E_NOSPACE, HADDR_UNDEF, "file allocation request failed");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

haddr_t
H5FD__alloc_real(H5FD_t *file, H5FD_mem_t type, hsize_t size, haddr_t *frag_addr, hsize_t *frag_size)
{
    hsize_t       orig_size = size;        
    haddr_t       eoa;                     
    hsize_t       extra;                   
    unsigned long flags = 0;               
    bool          use_alloc_size;          
    herr_t        status;                  
    haddr_t       ret_value = HADDR_UNDEF; 

    FUNC_ENTER_PACKAGE
#ifdef H5FD_ALLOC_DEBUG
    Rfprintf(Rstderr, "%s: type = %u, size = %" PRIuHSIZE "\n", __func__, (unsigned)type, size);
#endif 

    
    assert(file);
    assert(file->cls);
    assert(type >= H5FD_MEM_DEFAULT && type < H5FD_MEM_NTYPES);
    assert(size > 0);

    
    if (file->cls->query) {
        
        H5_BEFORE_USER_CB(HADDR_UNDEF)
            {
                status = (file->cls->query)(file, &flags);
            }
        H5_AFTER_USER_CB(HADDR_UNDEF)
        if (status < 0)
            HGOTO_ERROR(H5E_VFL, H5E_CANTGET, HADDR_UNDEF, "driver query request failed");
    }

    
    use_alloc_size = flags & H5FD_FEAT_USE_ALLOC_SIZE;

    
    H5_BEFORE_USER_CB(HADDR_UNDEF)
        {
            
            eoa = (file->cls->get_eoa)(file, type);
        }
    H5_AFTER_USER_CB(HADDR_UNDEF)

    
    extra = 0;
    if (!file->paged_aggr && file->alignment > 1 && orig_size >= file->threshold) {
        hsize_t mis_align; 

        
        if ((mis_align = (eoa % file->alignment)) > 0) {
            extra = file->alignment - mis_align;
            if (frag_addr)
                *frag_addr = eoa - file->base_addr; 
            if (frag_size)
                *frag_size = extra;
        } 
    }     

    
    
    
    if (file->cls->alloc) {
        
        H5_BEFORE_USER_CB(HADDR_UNDEF)
            {
                ret_value =
                    (file->cls->alloc)(file, type, H5CX_get_dxpl(), use_alloc_size ? size : size + extra);
            }
        H5_AFTER_USER_CB(HADDR_UNDEF)
        if (!H5_addr_defined(ret_value))
            HGOTO_ERROR(H5E_VFL, H5E_NOSPACE, HADDR_UNDEF, "driver allocation request failed");
    } 
    else {
        ret_value = H5FD__extend(file, type, size + extra);
        if (!H5_addr_defined(ret_value))
            HGOTO_ERROR(H5E_VFL, H5E_NOSPACE, HADDR_UNDEF, "driver eoa update request failed");
    } 

    
    if (!use_alloc_size)
        ret_value += extra;

    
    if (!file->paged_aggr && file->alignment > 1 && orig_size >= file->threshold)
        assert(!(ret_value % file->alignment));

    
    ret_value -= file->base_addr;

done:
#ifdef H5FD_ALLOC_DEBUG
    Rfprintf(Rstderr, "%s: ret_value = %" PRIuHADDR "\n", __func__, ret_value);
#endif 
    FUNC_LEAVE_NOAPI(ret_value)
} 

haddr_t
H5FD_alloc(H5FD_t *file, H5FD_mem_t type, H5F_t *f, hsize_t size, haddr_t *frag_addr, hsize_t *frag_size)
{
    haddr_t ret_value = HADDR_UNDEF; 

    FUNC_ENTER_NOAPI(HADDR_UNDEF)

    
    assert(file);
    assert(file->cls);
    assert(type >= H5FD_MEM_DEFAULT && type < H5FD_MEM_NTYPES);
    assert(size > 0);

    
    ret_value = H5FD__alloc_real(file, type, size, frag_addr, frag_size);
    if (!H5_addr_defined(ret_value))
        HGOTO_ERROR(H5E_VFL, H5E_CANTALLOC, HADDR_UNDEF, "real 'alloc' request failed");

    
    if (H5F_eoa_dirty(f) < 0)
        HGOTO_ERROR(H5E_VFL, H5E_CANTMARKDIRTY, HADDR_UNDEF, "unable to mark EOA info as dirty");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FD__free_real(H5FD_t *file, H5FD_mem_t type, haddr_t addr, hsize_t size)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(file);
    assert(file->cls);
    assert(type >= H5FD_MEM_DEFAULT && type < H5FD_MEM_NTYPES);
    assert(size > 0);

#ifdef H5FD_ALLOC_DEBUG
    Rfprintf(Rstderr, "%s: type = %u, addr = %" PRIuHADDR ", size = %" PRIuHSIZE "\n", __func__, (unsigned)type,
            addr, size);
#endif 

    
    if (!H5_addr_defined(addr))
        HGOTO_ERROR(H5E_VFL, H5E_BADVALUE, FAIL, "invalid file offset");

    
    addr += file->base_addr;

    
    if (addr > file->maxaddr || H5_addr_overflow(addr, size) || (addr + size) > file->maxaddr)
        HGOTO_ERROR(H5E_VFL, H5E_BADVALUE, FAIL, "invalid file free space region to free");

    
    if (file->cls->free) {
#ifdef H5FD_ALLOC_DEBUG
        Rfprintf(Rstderr, "%s: Letting VFD free space\n", __func__);
#endif 
        
        H5_BEFORE_USER_CB(FAIL)
            {
                
                ret_value = (file->cls->free)(file, type, H5CX_get_dxpl(), addr, size);
            }
        H5_AFTER_USER_CB(FAIL)
        if (ret_value < 0)
            HGOTO_ERROR(H5E_VFL, H5E_CANTFREE, FAIL, "driver free request failed");
    } 
    
    else if (file->cls->get_eoa) {
        haddr_t eoa;

        
        H5_BEFORE_USER_CB(FAIL)
            {
                
                eoa = (file->cls->get_eoa)(file, type);
            }
        H5_AFTER_USER_CB(FAIL)
#ifdef H5FD_ALLOC_DEBUG
        Rfprintf(Rstderr, "%s: eoa = %" PRIuHADDR "\n", __func__, eoa);
#endif 
        if (eoa == (addr + size)) {
#ifdef H5FD_ALLOC_DEBUG
            Rfprintf(Rstderr, "%s: Reducing file size to = %" PRIuHADDR "\n", __func__, addr);
#endif 
            
            H5_BEFORE_USER_CB(FAIL)
                {
                    
                    ret_value = (file->cls->set_eoa)(file, type, addr);
                }
            H5_AFTER_USER_CB(FAIL)
            if (ret_value < 0)
                HGOTO_ERROR(H5E_VFL, H5E_CANTSET, FAIL, "set end of space allocation request failed");
        } 
    }     
    else {
        
#ifdef H5FD_ALLOC_DEBUG
        Rfprintf(Rstderr, "%s: LEAKED MEMORY!!! type = %u, addr = %" PRIuHADDR ", size = %" PRIuHSIZE "\n",
                __func__, (unsigned)type, addr, size);
#endif 
    }  

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FD_free(H5FD_t *file, H5FD_mem_t type, H5F_t *f, haddr_t addr, hsize_t size)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(file);
    assert(file->cls);
    assert(type >= H5FD_MEM_DEFAULT && type < H5FD_MEM_NTYPES);
    assert(size > 0);

    
    if (H5FD__free_real(file, type, addr, size) < 0)
        HGOTO_ERROR(H5E_VFL, H5E_CANTFREE, FAIL, "real 'free' request failed");

    
    if (H5F_eoa_dirty(f) < 0)
        HGOTO_ERROR(H5E_VFL, H5E_CANTMARKDIRTY, FAIL, "unable to mark EOA info as dirty");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

htri_t
H5FD_try_extend(H5FD_t *file, H5FD_mem_t type, H5F_t *f, haddr_t blk_end, hsize_t extra_requested)
{
    haddr_t eoa;               
    htri_t  ret_value = false; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(file);
    assert(file->cls);
    assert(type >= H5FD_MEM_DEFAULT && type < H5FD_MEM_NTYPES);
    assert(extra_requested > 0);
    assert(f);

    
    H5_BEFORE_USER_CB(FAIL)
        {
            
            eoa = (file->cls->get_eoa)(file, type);
        }
    H5_AFTER_USER_CB(FAIL)
    if (!H5_addr_defined(eoa))
        HGOTO_ERROR(H5E_VFL, H5E_CANTGET, FAIL, "driver get_eoa request failed");

    
    blk_end += file->base_addr;

    
    if (H5_addr_eq(blk_end, eoa)) {
        
        if (HADDR_UNDEF == H5FD__extend(file, type, extra_requested))
            HGOTO_ERROR(H5E_VFL, H5E_CANTEXTEND, FAIL, "driver extend request failed");

        
        if (H5F_eoa_dirty(f) < 0)
            HGOTO_ERROR(H5E_VFL, H5E_CANTMARKDIRTY, FAIL, "unable to mark EOA info as dirty");

        
        HGOTO_DONE(true);
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
