/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#define H5A_FRIEND     
#include "H5Omodule.h" 
#define H5S_FRIEND     

#include "H5private.h"   
#include "H5Apkg.h"      
#include "H5Eprivate.h"  
#include "H5FLprivate.h" 
#include "H5MMprivate.h" 
#include "H5Opkg.h"      
#include "H5Spkg.h"      

static herr_t H5O__attr_encode(H5F_t *f, uint8_t *p, const void *mesg);
static void *H5O__attr_decode(H5F_t *f, H5O_t *open_oh, unsigned mesg_flags, unsigned *ioflags, size_t p_size,
                              const uint8_t *p);
static void *H5O__attr_copy(const void *_mesg, void *_dest);
static size_t H5O__attr_size(const H5F_t *f, const void *_mesg);
static herr_t H5O__attr_free(void *mesg);
static herr_t H5O__attr_pre_copy_file(H5F_t *file_src, const void *mesg_src, bool *deleted,
                                      const H5O_copy_t *cpy_info, void *udata);
static void  *H5O__attr_copy_file(H5F_t *file_src, const H5O_msg_class_t *mesg_type, void *native_src,
                                  H5F_t *file_dst, bool *recompute_size, H5O_copy_t *cpy_info, void *udata);
static herr_t H5O__attr_post_copy_file(const H5O_loc_t *src_oloc, const void *mesg_src, H5O_loc_t *dst_oloc,
                                       void *mesg_dst, H5O_copy_t *cpy_info);
static herr_t H5O__attr_get_crt_index(const void *_mesg, H5O_msg_crt_idx_t *crt_idx);
static herr_t H5O__attr_set_crt_index(void *_mesg, H5O_msg_crt_idx_t crt_idx);
static herr_t H5O__attr_debug(H5F_t *f, const void *_mesg, FILE *stream, int indent, int fwidth);

#define H5O_SHARED_TYPE                H5O_MSG_ATTR
#define H5O_SHARED_DECODE              H5O__attr_shared_decode
#define H5O_SHARED_DECODE_REAL         H5O__attr_decode
#define H5O_SHARED_ENCODE              H5O__attr_shared_encode
#define H5O_SHARED_ENCODE_REAL         H5O__attr_encode
#define H5O_SHARED_SIZE                H5O__attr_shared_size
#define H5O_SHARED_SIZE_REAL           H5O__attr_size
#define H5O_SHARED_DELETE              H5O__attr_shared_delete
#define H5O_SHARED_DELETE_REAL         H5O__attr_delete
#define H5O_SHARED_LINK                H5O__attr_shared_link
#define H5O_SHARED_LINK_REAL           H5O__attr_link
#define H5O_SHARED_COPY_FILE           H5O__attr_shared_copy_file
#define H5O_SHARED_COPY_FILE_REAL      H5O__attr_copy_file
#define H5O_SHARED_POST_COPY_FILE      H5O__attr_shared_post_copy_file
#define H5O_SHARED_POST_COPY_FILE_REAL H5O__attr_post_copy_file
#undef H5O_SHARED_POST_COPY_FILE_UPD
#define H5O_SHARED_DEBUG      H5O__attr_shared_debug
#define H5O_SHARED_DEBUG_REAL H5O__attr_debug
#include "H5Oshared.h" 

const H5O_msg_class_t H5O_MSG_ATTR[1] = {{
    H5O_ATTR_ID,                     
    "attribute",                     
    sizeof(H5A_t),                   
    H5O_SHARE_IS_SHARABLE,           
    H5O__attr_shared_decode,         
    H5O__attr_shared_encode,         
    H5O__attr_copy,                  
    H5O__attr_shared_size,           
    H5O__attr_reset,                 
    H5O__attr_free,                  
    H5O__attr_shared_delete,         
    H5O__attr_shared_link,           
    NULL,                            
    NULL,                            
    H5O__attr_pre_copy_file,         
    H5O__attr_shared_copy_file,      
    H5O__attr_shared_post_copy_file, 
    H5O__attr_get_crt_index,         
    H5O__attr_set_crt_index,         
    H5O__attr_shared_debug           
}};

#define H5O_ATTR_FLAG_TYPE_SHARED  0x01
#define H5O_ATTR_FLAG_SPACE_SHARED 0x02
#define H5O_ATTR_FLAG_ALL          0x03

H5FL_EXTERN(H5S_t);

H5FL_EXTERN(H5S_extent_t);

static void *
H5O__attr_decode(H5F_t *f, H5O_t *open_oh, unsigned H5_ATTR_UNUSED mesg_flags, unsigned *ioflags,
                 size_t p_size, const uint8_t *p)
{
    H5A_t         *attr   = NULL;
    const uint8_t *p_end  = p + p_size - 1; 
    size_t         delta  = 0;              
    H5S_extent_t  *extent = NULL;           
    size_t         name_len;                
    size_t         dt_size;                 
    hssize_t       sds_size;                
    hsize_t        ds_size;                 
    unsigned       flags     = 0;           
    H5A_t         *ret_value = NULL;        

    FUNC_ENTER_PACKAGE

    assert(f);
    assert(p);

    if (NULL == (attr = H5FL_CALLOC(H5A_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");
    if (NULL == (attr->shared = H5FL_CALLOC(H5A_shared_t)))
        HGOTO_ERROR(H5E_FILE, H5E_NOSPACE, NULL, "can't allocate shared attr structure");

    
    if (H5_IS_BUFFER_OVERFLOW(p, 1, p_end))
        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
    attr->shared->version = *p++;
    if (attr->shared->version < H5O_ATTR_VERSION_1 || attr->shared->version > H5O_ATTR_VERSION_LATEST)
        HGOTO_ERROR(H5E_ATTR, H5E_CANTLOAD, NULL, "bad version number for attribute message");

    
    if (H5_IS_BUFFER_OVERFLOW(p, 1, p_end))
        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
    if (attr->shared->version >= H5O_ATTR_VERSION_2) {
        flags = *p++;

        
        if (flags & (unsigned)~H5O_ATTR_FLAG_ALL)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTLOAD, NULL, "unknown flag for attribute message");
    }
    else
        p++; 

    
    if (H5_IS_BUFFER_OVERFLOW(p, 2, p_end))
        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
    UINT16DECODE(p, name_len); 

    
    if (name_len <= 1)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTDECODE, NULL, "decoded name length is invalid");

    if (H5_IS_BUFFER_OVERFLOW(p, 2, p_end))
        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
    UINT16DECODE(p, attr->shared->dt_size);
    if (H5_IS_BUFFER_OVERFLOW(p, 2, p_end))
        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
    UINT16DECODE(p, attr->shared->ds_size);

    
    if (attr->shared->version >= H5O_ATTR_VERSION_3) {
        if (H5_IS_BUFFER_OVERFLOW(p, 1, p_end))
            HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
        attr->shared->encoding = (H5T_cset_t)*p++;
    }

    
    if (H5_IS_BUFFER_OVERFLOW(p, name_len, p_end))
        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");

    if (NULL == (attr->shared->name = H5MM_strndup((const char *)p, name_len - 1)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");

    
    if (name_len != (strnlen(attr->shared->name, name_len) + 1))
        HGOTO_ERROR(H5E_ATTR, H5E_CANTDECODE, NULL, "attribute name has different length than stored length");

    
    if (attr->shared->version < H5O_ATTR_VERSION_2)
        delta = H5O_ALIGN_OLD(name_len);
    else
        delta = name_len;
    if (H5_IS_BUFFER_OVERFLOW(p, delta, p_end))
        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
    p += delta;

    
    if (H5_IS_BUFFER_OVERFLOW(p, attr->shared->dt_size, p_end))
        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
    if (NULL == (attr->shared->dt = (H5T_t *)(H5O_MSG_DTYPE->decode)(
                     f, open_oh, ((flags & H5O_ATTR_FLAG_TYPE_SHARED) ? H5O_MSG_FLAG_SHARED : 0), ioflags,
                     attr->shared->dt_size, p)))
        HGOTO_ERROR(H5E_ATTR, H5E_CANTDECODE, NULL, "can't decode attribute datatype");

    
    if (attr->shared->version < H5O_ATTR_VERSION_2)
        delta = H5O_ALIGN_OLD(attr->shared->dt_size);
    else
        delta = attr->shared->dt_size;
    if (H5_IS_BUFFER_OVERFLOW(p, delta, p_end))
        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
    p += delta;

    
    if (NULL == (attr->shared->ds = H5FL_CALLOC(H5S_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");

    
    if (H5_IS_BUFFER_OVERFLOW(p, attr->shared->ds_size, p_end))
        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
    if ((extent = (H5S_extent_t *)(H5O_MSG_SDSPACE->decode)(
             f, open_oh, ((flags & H5O_ATTR_FLAG_SPACE_SHARED) ? H5O_MSG_FLAG_SHARED : 0), ioflags,
             attr->shared->ds_size, p)) == NULL)
        HGOTO_ERROR(H5E_ATTR, H5E_CANTDECODE, NULL, "can't decode attribute dataspace");

    
    H5MM_memcpy(&(attr->shared->ds->extent), extent, sizeof(H5S_extent_t));

    
    extent = H5FL_FREE(H5S_extent_t, extent);

    
    if (H5S_select_all(attr->shared->ds, false) < 0)
        HGOTO_ERROR(H5E_DATASPACE, H5E_CANTSET, NULL, "unable to set all selection");

    
    if (attr->shared->version < H5O_ATTR_VERSION_2)
        delta = H5O_ALIGN_OLD(attr->shared->ds_size);
    else
        delta = attr->shared->ds_size;
    if (H5_IS_BUFFER_OVERFLOW(p, delta, p_end))
        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
    p += delta;

    
    if (0 == (dt_size = H5T_get_size(attr->shared->dt)))
        HGOTO_ERROR(H5E_ATTR, H5E_CANTGET, NULL, "unable to get datatype size");
    if ((sds_size = H5S_GET_EXTENT_NPOINTS(attr->shared->ds)) < 0)
        HGOTO_ERROR(H5E_ATTR, H5E_CANTGET, NULL, "unable to get dataspace size");
    ds_size = (hsize_t)sds_size;

    
    H5_CHECKED_ASSIGN(attr->shared->data_size, size_t, ds_size * (hsize_t)dt_size, hsize_t);
    
    if ((attr->shared->data_size / dt_size) != ds_size)
        HGOTO_ERROR(H5E_RESOURCE, H5E_OVERFLOW, NULL, "data size exceeds addressable range");

    
    if (attr->shared->data_size) {
        
        if (H5_IS_BUFFER_OVERFLOW(p, attr->shared->data_size, p_end))
            HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");

        if (NULL == (attr->shared->data = H5FL_BLK_MALLOC(attr_buf, attr->shared->data_size)))
            HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");
        H5MM_memcpy(attr->shared->data, p, attr->shared->data_size);
    }

    
    attr->shared->nrefs++;

    
    ret_value = attr;

done:
    if (NULL == ret_value) {
        if (attr) {
            if (attr->shared)
                if (H5A__shared_free(attr) < 0)
                    HDONE_ERROR(H5E_ATTR, H5E_CANTRELEASE, NULL, "can't release attribute info");
            attr = H5FL_FREE(H5A_t, attr);
        }
        if (extent)
            extent = H5FL_FREE(H5S_extent_t, extent);
    }

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__attr_encode(H5F_t *f, uint8_t *p, const void *mesg)
{
    const H5A_t *attr = (const H5A_t *)mesg;
    size_t       name_len;        
    htri_t       is_type_shared;  
    htri_t       is_space_shared; 
    unsigned     flags     = 0;   
    herr_t       ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(p);
    assert(attr);

    
    if ((is_type_shared = H5O_msg_is_shared(H5O_DTYPE_ID, attr->shared->dt)) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_BADMESG, FAIL, "can't determine if datatype is shared");

    if ((is_space_shared = H5O_msg_is_shared(H5O_SDSPACE_ID, attr->shared->ds)) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_BADMESG, FAIL, "can't determine if dataspace is shared");

    
    *p++ = attr->shared->version;

    
    if (attr->shared->version >= H5O_ATTR_VERSION_2) {
        flags = (is_type_shared ? H5O_ATTR_FLAG_TYPE_SHARED : 0);
        flags |= (is_space_shared ? H5O_ATTR_FLAG_SPACE_SHARED : 0);
        *p++ = (uint8_t)flags; 
    }                          
    else
        *p++ = 0; 

    
    name_len = strlen(attr->shared->name) + 1;
    UINT16ENCODE(p, name_len);
    UINT16ENCODE(p, attr->shared->dt_size);
    UINT16ENCODE(p, attr->shared->ds_size);

    
    if (attr->shared->version >= H5O_ATTR_VERSION_3)
        *p++ = (uint8_t)attr->shared->encoding;

    
    H5MM_memcpy(p, attr->shared->name, name_len);
    if (attr->shared->version < H5O_ATTR_VERSION_2) {
        
        memset(p + name_len, 0, H5O_ALIGN_OLD(name_len) - name_len);
        p += H5O_ALIGN_OLD(name_len);
    } 
    else
        p += name_len;

    
    if ((H5O_MSG_DTYPE->encode)(f, false, SIZE_MAX, p, attr->shared->dt) < 0)
        HGOTO_ERROR(H5E_ATTR, H5E_CANTENCODE, FAIL, "can't encode attribute datatype");

    if (attr->shared->version < H5O_ATTR_VERSION_2) {
        memset(p + attr->shared->dt_size, 0, H5O_ALIGN_OLD(attr->shared->dt_size) - attr->shared->dt_size);
        p += H5O_ALIGN_OLD(attr->shared->dt_size);
    } 
    else
        p += attr->shared->dt_size;

    
    if ((H5O_MSG_SDSPACE->encode)(f, false, SIZE_MAX, p, &(attr->shared->ds->extent)) < 0)
        HGOTO_ERROR(H5E_ATTR, H5E_CANTENCODE, FAIL, "can't encode attribute dataspace");

    if (attr->shared->version < H5O_ATTR_VERSION_2) {
        memset(p + attr->shared->ds_size, 0, H5O_ALIGN_OLD(attr->shared->ds_size) - attr->shared->ds_size);
        p += H5O_ALIGN_OLD(attr->shared->ds_size);
    } 
    else
        p += attr->shared->ds_size;

    
    if (attr->shared->data)
        H5MM_memcpy(p, attr->shared->data, attr->shared->data_size);
    else
        memset(p, 0, attr->shared->data_size);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static void *
H5O__attr_copy(const void *_src, void *_dst)
{
    void *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    assert(_src);

    
    if (NULL == (ret_value = (H5A_t *)H5A__copy((H5A_t *)_dst, (const H5A_t *)_src)))
        HGOTO_ERROR(H5E_ATTR, H5E_CANTINIT, NULL, "can't copy attribute");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static size_t
H5O__attr_size(const H5F_t H5_ATTR_UNUSED *f, const void *_mesg)
{
    const H5A_t *attr = (const H5A_t *)_mesg;
    size_t       name_len;
    size_t       ret_value = 0;

    FUNC_ENTER_PACKAGE_NOERR

    assert(attr);

    
    ret_value = 1 + 
                1 + 
                2 + 
                2 + 
                2;  

    
    name_len = strlen(attr->shared->name) + 1;

    
    if (attr->shared->version == H5O_ATTR_VERSION_1)
        ret_value += H5O_ALIGN_OLD(name_len) +              
                     H5O_ALIGN_OLD(attr->shared->dt_size) + 
                     H5O_ALIGN_OLD(attr->shared->ds_size) + 
                     attr->shared->data_size;               
    else if (attr->shared->version == H5O_ATTR_VERSION_2)
        ret_value += name_len +               
                     attr->shared->dt_size +  
                     attr->shared->ds_size +  
                     attr->shared->data_size; 
    else if (attr->shared->version == H5O_ATTR_VERSION_3)
        ret_value += 1 +                      
                     name_len +               
                     attr->shared->dt_size +  
                     attr->shared->ds_size +  
                     attr->shared->data_size; 
    else
        assert(0 && "Bad attribute version");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5O__attr_reset(void H5_ATTR_UNUSED *_mesg)
{
    FUNC_ENTER_PACKAGE_NOERR

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5O__attr_free(void *mesg)
{
    H5A_t *attr      = (H5A_t *)mesg;
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    assert(mesg);

    if (H5A__close(attr) < 0)
        HGOTO_ERROR(H5E_ATTR, H5E_CANTCLOSEOBJ, FAIL, "unable to close attribute object");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5O__attr_delete(H5F_t *f, H5O_t *oh, void *_mesg)
{
    H5A_t *attr      = (H5A_t *)_mesg;
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(attr);

    
    if ((H5O_MSG_DTYPE->del)(f, oh, attr->shared->dt) < 0)
        HGOTO_ERROR(H5E_ATTR, H5E_LINKCOUNT, FAIL, "unable to adjust datatype link count");

    
    if ((H5O_MSG_SDSPACE->del)(f, oh, attr->shared->ds) < 0)
        HGOTO_ERROR(H5E_ATTR, H5E_LINKCOUNT, FAIL, "unable to adjust dataspace link count");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5O__attr_link(H5F_t *f, H5O_t *oh, void *_mesg)
{
    H5A_t *attr      = (H5A_t *)_mesg;
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(attr);

    
    
    if ((H5O_MSG_DTYPE->link)(f, oh, attr->shared->dt) < 0)
        HGOTO_ERROR(H5E_ATTR, H5E_LINKCOUNT, FAIL, "unable to adjust datatype link count");
    if ((H5O_MSG_SDSPACE->link)(f, oh, attr->shared->ds) < 0)
        HGOTO_ERROR(H5E_ATTR, H5E_LINKCOUNT, FAIL, "unable to adjust dataspace link count");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__attr_pre_copy_file(H5F_t H5_ATTR_UNUSED *file_src, const void *native_src, bool *deleted,
                        const H5O_copy_t *cpy_info, void H5_ATTR_UNUSED *udata)
{
    const H5A_t *attr_src  = (const H5A_t *)native_src; 
    herr_t       ret_value = SUCCEED;                   

    FUNC_ENTER_PACKAGE

    
    assert(deleted);
    assert(cpy_info);
    assert(cpy_info->file_dst);

    
    if (attr_src->shared->version > H5O_attr_ver_bounds[H5F_HIGH_BOUND(cpy_info->file_dst)])
        HGOTO_ERROR(H5E_OHDR, H5E_BADRANGE, FAIL, "attribute message version out of bounds");

    
    if (cpy_info->copy_without_attr)
        *deleted = true;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static void *
H5O__attr_copy_file(H5F_t *file_src, const H5O_msg_class_t H5_ATTR_UNUSED *mesg_type, void *native_src,
                    H5F_t *file_dst, bool *recompute_size, H5O_copy_t *cpy_info, void H5_ATTR_UNUSED *udata)
{
    void *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    assert(native_src);
    assert(file_dst);
    assert(cpy_info);
    assert(!cpy_info->copy_without_attr);

    
    if (H5T_set_loc(((H5A_t *)native_src)->shared->dt, H5F_VOL_OBJ(file_src), H5T_LOC_DISK) < 0)
        HGOTO_ERROR(H5E_ATTR, H5E_CANTINIT, NULL, "invalid datatype location");

    if (NULL == (ret_value = H5A__attr_copy_file((H5A_t *)native_src, file_dst, recompute_size, cpy_info)))
        HGOTO_ERROR(H5E_ATTR, H5E_CANTCOPY, NULL, "can't copy attribute");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__attr_post_copy_file(const H5O_loc_t *src_oloc, const void *mesg_src, H5O_loc_t *dst_oloc, void *mesg_dst,
                         H5O_copy_t *cpy_info)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    if (H5A__attr_post_copy_file(src_oloc, (const H5A_t *)mesg_src, dst_oloc, (H5A_t *)mesg_dst, cpy_info) <
        0)
        HGOTO_ERROR(H5E_ATTR, H5E_CANTCOPY, FAIL, "can't copy attribute");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__attr_get_crt_index(const void *_mesg, H5O_msg_crt_idx_t *crt_idx )
{
    const H5A_t *attr = (const H5A_t *)_mesg;

    FUNC_ENTER_PACKAGE_NOERR

    assert(attr);
    assert(crt_idx);

    
    *crt_idx = attr->shared->crt_idx;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5O__attr_set_crt_index(void *_mesg, H5O_msg_crt_idx_t crt_idx)
{
    H5A_t *attr = (H5A_t *)_mesg;

    FUNC_ENTER_PACKAGE_NOERR

    assert(attr);

    
    attr->shared->crt_idx = crt_idx;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5O__attr_debug(H5F_t *f, const void *_mesg, FILE *stream, int indent, int fwidth)
{
    const H5A_t *mesg = (const H5A_t *)_mesg;
    const char  *s;                   
    char         buf[128];            
    herr_t       ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(stream);
    assert(indent >= 0);
    assert(fwidth >= 0);

    Rfprintf(stream, "%*s%-*s \"%s\"\n", indent, "", fwidth, "Name:", mesg->shared->name);
    switch (mesg->shared->encoding) {
        case H5T_CSET_ASCII:
            s = "ASCII";
            break;

        case H5T_CSET_UTF8:
            s = "UTF-8";
            break;

        case H5T_CSET_RESERVED_2:
        case H5T_CSET_RESERVED_3:
        case H5T_CSET_RESERVED_4:
        case H5T_CSET_RESERVED_5:
        case H5T_CSET_RESERVED_6:
        case H5T_CSET_RESERVED_7:
        case H5T_CSET_RESERVED_8:
        case H5T_CSET_RESERVED_9:
        case H5T_CSET_RESERVED_10:
        case H5T_CSET_RESERVED_11:
        case H5T_CSET_RESERVED_12:
        case H5T_CSET_RESERVED_13:
        case H5T_CSET_RESERVED_14:
        case H5T_CSET_RESERVED_15:
            snprintf(buf, sizeof(buf), "H5T_CSET_RESERVED_%d", (int)(mesg->shared->encoding));
            s = buf;
            break;

        case H5T_CSET_ERROR:
        default:
            snprintf(buf, sizeof(buf), "Unknown character set: %d", (int)(mesg->shared->encoding));
            s = buf;
            break;
    } 
    Rfprintf(stream, "%*s%-*s %s\n", indent, "", fwidth, "Character Set of Name:", s);
    Rfprintf(stream, "%*s%-*s %s\n", indent, "", fwidth,
            "Object opened:", mesg->obj_opened ? "TRUE" : "FALSE");
    Rfprintf(stream, "%*s%-*s %" PRIuHADDR "\n", indent, "", fwidth, "Object:", mesg->oloc.addr);

    
    if (mesg->shared->crt_idx != H5O_MAX_CRT_ORDER_IDX)
        Rfprintf(stream, "%*s%-*s %u\n", indent, "", fwidth,
                "Creation Index:", (unsigned)mesg->shared->crt_idx);

    Rfprintf(stream, "%*sDatatype...\n", indent, "");
    Rfprintf(stream, "%*s%-*s %lu\n", indent + 3, "", MAX(0, fwidth - 3),
            "Encoded Size:", (unsigned long)(mesg->shared->dt_size));
    if ((H5O_MSG_DTYPE->debug)(f, mesg->shared->dt, stream, indent + 3, MAX(0, fwidth - 3)) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_WRITEERROR, FAIL, "unable to display datatype message info");

    Rfprintf(stream, "%*sDataspace...\n", indent, "");
    Rfprintf(stream, "%*s%-*s %lu\n", indent + 3, "", MAX(0, fwidth - 3),
            "Encoded Size:", (unsigned long)(mesg->shared->ds_size));
    if (H5S_debug(f, mesg->shared->ds, stream, indent + 3, MAX(0, fwidth - 3)) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_WRITEERROR, FAIL, "unable to display dataspace message info");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
