% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/residuals-methods.R
\name{predict.pense_cvfit}
\alias{predict.pense_cvfit}
\title{Predict Method for PENSE Fits}
\usage{
\method{predict}{pense_cvfit}(object, newdata, alpha = NULL, lambda = "min", se_mult = 1, ...)
}
\arguments{
\item{object}{PENSE with cross-validated hyper-parameters to extract coefficients from.}

\item{newdata}{an optional matrix of new predictor values.
If missing, the fitted values are computed.}

\item{alpha}{Either a single number or \code{NULL} (default).
If given, only fits with the given \code{alpha} value are considered.
If \code{lambda} is a numeric value and \code{object} was fit with multiple \emph{alpha}
values and no value is provided, the first value in \code{object$alpha} is used with a warning.}

\item{lambda}{either a string specifying which penalty level to use
(\code{"min"}, \code{"se"}, \verb{"\{m\}-se}")
or a single numeric value of the penalty parameter. See details.}

\item{se_mult}{If \code{lambda = "se"}, the multiple of standard errors to tolerate.}

\item{...}{currently not used.}
}
\value{
a numeric vector of residuals for the given penalization level.
}
\description{
Predict response values using a PENSE (or LS-EN) regularization path with
hyper-parameters chosen by cross-validation.
}
\section{Hyper-parameters}{


If \code{lambda = "{m}-se"} and \code{object} contains fitted estimates for every penalization
level in the sequence, use the fit the most parsimonious model with prediction performance
statistically indistinguishable from the best model.
This is determined to be the model with prediction performance within \code{m * cv_se}
from the best model.
If \code{lambda = "se"}, the multiplier \emph{m} is taken from \code{se_mult}.

By default all \emph{alpha} hyper-parameters available in the fitted object are considered.
This can be overridden by supplying one or multiple values in parameter \code{alpha}.
For example, if \code{lambda = "1-se"} and \code{alpha} contains two values, the "1-SE" rule is applied
individually for each \code{alpha} value, and the fit with the better prediction error is considered.

In case \code{lambda} is a number and \code{object} was fit for several \emph{alpha} hyper-parameters,
\code{alpha} must also be given, or the first value in \code{object$alpha} is used with a warning.
}

\examples{
# Compute the LS-EN regularization path for Freeny's revenue data
# (see ?freeny)
data(freeny)
x <- as.matrix(freeny[ , 2:5])

regpath <- elnet(x, freeny$y, alpha = 0.75)

# Predict the response using a specific penalization level
predict(regpath, newdata = freeny[1:5, 2:5],
        lambda = regpath$lambda[[1]][[10]])

# Extract the residuals at a certain penalization level
residuals(regpath, lambda = regpath$lambda[[1]][[5]])

# Select penalization level via cross-validation
set.seed(123)
cv_results <- elnet_cv(x, freeny$y, alpha = 0.5,
                       cv_repl = 10, cv_k = 4)

# Predict the response using the "best" penalization level
predict(cv_results, newdata = freeny[1:5, 2:5])

# Extract the residuals at the "best" penalization level
residuals(cv_results)
# Extract the residuals at a more parsimonious penalization level
residuals(cv_results, lambda = "1.5-se")
}
\seealso{
Other functions for extracting components: 
\code{\link{coef.pense_cvfit}()},
\code{\link{coef.pense_fit}()},
\code{\link{predict.pense_fit}()},
\code{\link{residuals.pense_cvfit}()},
\code{\link{residuals.pense_fit}()}
}
\concept{functions for extracting components}
