% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/btsr_generic.R
\name{BTSR.functions}
\alias{BTSR.functions}
\alias{btsr.sim}
\alias{btsr.extract}
\alias{btsr.fit}
\title{Generic functions to simulate, extract components and fit BTSR models}
\usage{
btsr.sim(model, ...)

btsr.extract(model, ...)

btsr.fit(model, ...)
}
\arguments{
\item{model}{character string (case-insensitive) indicating the model to be
fitted to the data. Must be one of the options listed in the Section
\link[=arguments.model]{Supported Models}.}

\item{...}{further arguments passed to the internal functions. See, for
instance, \link{summary.btsr} for details.}
}
\value{
By default, the \emph{simulation function} return the simulated time series
\code{yt}. If \code{complete = TRUE}, it returns a list with the following components
\itemize{
\item \code{model}: string with the name of the model.

\item \code{yt}: the simulated time series \eqn{Y_t}

\item \code{mut}: the conditional mean \eqn{\mu_t}

\item \code{etat}: the linear predictor(s)\cr For all models where \eqn{\nu} is
constant over time, returns \eqn{\eta_{1t} = g_{11}(\mu_t)}\cr For models
with time varying \eqn{\nu}, returns a matrix whose columns are
\eqn{\eta_{1t} = g_{11}(\mu_t)} and \eqn{\eta_{2t} = g_{21}(\vartheta_t)}.

\item \code{nut}: the conditional precision \eqn{\nu_t} (only for models with time
varying \eqn{\nu})

\item \code{varthetat}: the transformed time series \eqn{\vartheta_t = g_2(\nu_t)}.
(only for models with time varying \eqn{\nu})\cr
For BARFIMAV models, if \code{g2 = "default"}, then \code{varthetat} is the conditional
dispersion given by \eqn{\vartheta_t = (1+\nu_t)^{-1}}.

\item \code{error}: the error term \eqn{e_{1t}}

\item \code{out.Fortran}: the output from Fortran (if requested).
}

The \emph{extraction function} returns a list with the following components
\itemize{
\item \code{model}: string with the name of the model.

\item \code{yt}: the simulated time series \eqn{Y_t}

\item \code{mut}: the conditional mean \eqn{\mu_t}

\item \code{etat}: the linear predictor(s)\cr For models with \eqn{\nu} fixed,
returns \eqn{\eta_{1t} = g_{11}(\mu_t)}\cr For models with time varying
\eqn{\nu}, returns a matrix whose columns are \eqn{\eta_{1t} = g_{11}(\mu_t)}
and \eqn{\eta_{2t} = g_{21}(\vartheta_t)}.

\item \code{nut}: the conditional precision \eqn{\nu_t} (only for models with time
varying \eqn{\nu})

\item \code{varthetat}: the transformed time series \eqn{\vartheta_t = g_2(\nu_t)}.
(only for models with time varying \eqn{\nu})\cr
For BARFIMAV models, if \code{g2 = "default"}, then \code{varthetat} is the conditional
dispersion given by \eqn{\vartheta_t = (1+\nu_t)^{-1}}.

\item \code{error}: the error term \eqn{e_{1t}} (depends on the argument
\code{error.scale})

\item \code{forecast}: the out-of-sample forecast. (if requested)\cr
If \eqn{\nu_t} is fixed: a vector with the predicted values for \eqn{\mu_t}
and \eqn{\eta_{1t}}\cr
If \eqn{\nu_t} is time varying: a matrix the predicted values for \eqn{\mu_t}
and \eqn{\eta_{1t}}, \eqn{\nu_t}, \eqn{\vartheta_t} and \eqn{\eta_{2t}}.

\item \code{xnew}: the observations of the regressors observed/predicted values
corresponding to the period of out-of-sample forecast. Only included if
\code{xreg} is not \code{NULL} and \code{nnew > 0}.

\item \code{sll}: the sum of the conditional log-likelihood (if requested)

\item \code{score}: the score vector  (if requested)

\item \code{info.Matrix.}: the score vector  (if requested)

\item \code{D}, \code{T}, \code{E}, \code{h}: additional matrices and vectors used to calculate
the score vector and the information matrix. (if requested)

\item \code{out.Fortran}: FORTRAN output  (if requested)
}

The \emph{fitting function} returns a list with the following components.

\itemize{
\item \code{model}: character; the same as the input argument.

\item \code{call}: string with a complete description of the model, including
the AR and MA order.

\item \code{n}: the sample size used for estimation.

\item \code{series}: the observed time series \eqn{Y_t}

\item \code{gyt}: a vector or a matrix with the transformed time series
\eqn{g_{11}(Y_t)} and \eqn{g_{12}(Y_t)}. Only returns a matrix if the links
\eqn{g_{11}} and \eqn{g_{12}} are not the same.

\item \code{xreg}: a vector or matrix of regressors \eqn{\boldsymbol{X}_t} (if
included in the model).

\item \code{control}: a list of control parameters.

\item \code{convergence}: An integer code. 0 indicates successful completion. The
error codes depend on the algorithm used.

\item \code{message}: A character string giving any additional information
returned by the optimizer (if any), or \code{NULL}.

\item \code{counts}: an integer giving the number of function evaluations.

\item \code{start}: the starting values used by the algorithm.

\item \code{coefficients}: The best set of parameters found.

\item \code{fitted.values}: in-sample forecast.\cr
If \eqn{\nu_t} is fixed: a vector with the in-sample value of \eqn{\mu_t}.\cr
If \eqn{\nu_t} is time varying: a matrix with the in-sample values of
\eqn{\mu_t}, \eqn{\nu_t} and \eqn{\vartheta_t}.

\item \code{etat}: the linear predictor(s)\cr For models with \eqn{\nu} fixed,
returns \eqn{\eta_{1t} = g_{11}(\mu_t)}\cr For models with time varying
\eqn{\nu}, returns a matrix whose columns are \eqn{\eta_{1t} = g_{11}(\mu_t)}
and \eqn{\eta_{2t} = g_{21}(\vartheta_t)}.

\item \code{error}: the error term \eqn{e_{1t}} (depends on the argument
\code{error.scale})

\item \code{residual}: the observed values \eqn{Y_t} minus the fitted values
\eqn{\mu_t}. The same as the \code{error} term if \code{error.scale = 0}.

\item \code{forecast}: the out-of-sample forecast. (if requested)\cr
If \eqn{\nu_t} is fixed: a vector with the predicted values for \eqn{\mu_t}
and \eqn{\eta_{1t}}\cr
If \eqn{\nu_t} is time varying: a matrix the predicted values for \eqn{\mu_t}
and \eqn{\eta_{1t}}, \eqn{\nu_t}, \eqn{\vartheta_t} and \eqn{\eta_{2t}}.

\item \code{xnew}: the observations of the regressors observed/predicted values
corresponding to the period of out-of-sample forecast. Only included if
\code{xreg} is not \code{NULL} and \code{nnew > 0}.

\item \code{sll}: the sum of the conditional log-likelihood (if requested)

\item \code{score}: the score vector (if requested)

\item \code{info.Matrix}: the information matrix (if requested)

\item \code{link}: the codes for the link functions (for summary purposes)

\item \code{configs}: a list with the configurations passed to FORTRAN to fit the
model. This information is used by the prediction function.

\item \code{out.Fortran}: FORTRAN output  (if requested).
}
}
\description{
These generic functions can be used to simulate, extract components and fit
any model of the class \code{btsr}. See \sQuote{Details} below. \cr

The package handles function arguments in two compatible formats
\itemize{
\item \strong{Legacy structure} (pre-1.0.0). Used for models with fixed or
no \eqn{\nu} parameter. Automatically converted to the new format when
processed.

\item \strong{New structure} (1.0.0+). Required for models with
time-varying \eqn{\nu} parameter.
}
All functions accept both formats seamlessly, ensuring backward compatibility.
The internal processing automatically standardizes to the new structure.
}
\details{
A detailed description of the general structure (mathematical formulation) of
BTSR models, associated to the \code{btsr} class, is presented in Section
\sQuote{The BTSR Structure} of \link{btsr-package}. Particular models are
discussed in \link{arguments.model}.\cr

All functions are compatible with the new format for the arguments,
introduced in version 1.0.0. and the previous format.
\itemize{
\item The function \code{btsr.sim} is used to generate random samples from any
BTSR models.

\item The function \code{btsr.extract} allows the user to extract all conditional
time series, the log-likelihood, and the vectors and matrices used to
calculate the score vector and the information matrix associated to a given
set of parameters. This function can be used by any user to create an
objective function that can be passed to optimization functions not available
in BTSR Package. At this point, there is no other use for which this function
was intended.

\item The function \code{btsr.fit} fits a BTSR model to a given univariate time
series. For now, available optimization algorithms are \code{"L-BFGS-B"} and
\code{"Nelder-Mead"}. Both methods accept bounds for the parameters. For
\code{"Nelder-Mead"}, bounds are set via parameter transformation.
}

For compatibility with previous versions of the package, all functions
associated to parent models (e.g. BARFIMA) are still available (see
\link{BTSR.parent.models}). Also, analogous functions are available for parent
models with time varying \eqn{\nu} (e.g. BARFIMAV). The list of arguments and
default values for these specific functions can be accessed using the
function \link{get.defaults}.\cr

Particular models (e.g. BETA, BARMA) share the same arguments as the parent
model, however, some arguments can have different default values (see the
documentation for \link[=arguments]{shared arguments} for details). Information on
the parent model can be obtained using the function \link{BTSR.model.defaults}.
}
\examples{
\donttest{
#########################################################################
#
#   Examples of usage of btsr.sim
#
#########################################################################
#------------------------------------------------------------------------
# Generating a Beta model were both mu and nu do not vary with time
# yt ~ Beta(a,b), a = mu*nu, b = (1-mu)*nu
#------------------------------------------------------------------------

# CASE 1: using the legacy format for coefs
set.seed(1234)
y1 <- btsr.sim(
  model = "BETA", n = 1000,
  coefs = list(alpha = 0.2, nu = 20)
)
hist(y1)

# CASE 2: using the new layout for coefs
set.seed(1234)
y2 <- btsr.sim(
  model = "BETA", n = 1000,
  coefs = list(part1 = list(alpha = 0.2), part2 = list(alpha = 20))
)
hist(y2)

# CASE 3: function for the parent model plus legacy format for coefs.
# - requires setting linkg = "linear", otherwhise the default "logit"
#   link is used.
set.seed(1234)
y3 <- BARFIMA.sim(
  linkg = "linear", n = 1000,
  coefs = list(alpha = 0.2, nu = 20)
)
hist(y3)

# CASE 4: function for the parent model plus new format for coefs.
# - requires setting linkg = "linear", otherwhise the default "logit"
#   link is used.
set.seed(1234)
y4 <- BARFIMA.sim(
  n = 1000, linkg = "linear",
  coefs = list(part1 = list(alpha = 0.2), part2 = list(alpha = 20))
)
hist(y4)

# comparing the results:
range(abs(y2 - y1))
range(abs(y3 - y1))
range(abs(y3 - y4))

#------------------------------------------------------------------------
# Generating a sample from a Beta regression model
#------------------------------------------------------------------------
burn <- 100
n <- 500
N <- n + burn
covar <- cbind(sin(2 * pi * (1:N) / 50), 1:N)

set.seed(1234)
y1 <- btsr.sim(
  model = "BREG", linkg = "logit",
  n = n, burn = burn, xreg = covar,
  coefs = list(alpha = -1.3, beta = c(0.6, 0.002), nu = 30),
  complete = TRUE
)

# The regressors: X1 = sin(2*pi*t/50) and X2 = t
plot.ts(
  covar,
  main = "Regressors" ~ X[1][t] == sin(2 * pi * t / 50) ~ "and" ~ X[2][t] == t
)

# Conditional time series:
plot.ts(y1$etat, main = "Linear predictor" ~ eta[t] == g[11](mu[t]))
plot.ts(y1$mut, main = "Conditional mean" ~ mu[t])
plot.ts(y1$yt, main = "Time series" ~ Y[t])

#########################################################################
#
#   Examples of usage of btsr.extract
#
#########################################################################
#------------------------------------------------------------------------
# Generating a sample from a BARMAX(1,1) model (BARMA with covariates)
#------------------------------------------------------------------------
burn <- 100
n <- 500
N <- n + burn
covar <- cbind(sin(2 * pi * (1:N) / 50), 1:N)

set.seed(1234)
m1 <- btsr.sim(
  model = "BARMA", linkg = "logit",
  n = n, burn = burn, xreg = covar,
  coefs = list(
    alpha = 0, phi = -0.65, theta = -0.25,
    beta = c(0.6, -0.002), nu = 20
  ),
  error.scale = 1, complete = TRUE
)

# Extracting components assuming that all coefficients are non-fixed
e1 <- btsr.extract(
  model = "BARMA", yt = m1$yt,
  xreg = covar[(burn + 1):N, ], linkg = "logit",
  coefs = list(
    alpha = 0, phi = -0.65, theta = -0.25,
    beta = c(0.6, -0.002), nu = 20
  ),
  llk = TRUE, sco = TRUE, info = TRUE
)

# Extracting components assuming that all coefficients are fixed
#  - no need to provide fixed.lags in this case.
e2 <- btsr.extract(
  model = "BARMA", yt = m1$yt,
  xreg = covar[(burn + 1):N, ], linkg = "logit",
  fixed.values = list(
    alpha = 0, phi = -0.65, theta = -0.25,
    beta = c(0.6, -0.002), nu = 20
  ),
  llk = TRUE, sco = TRUE, info = TRUE
)

# Extracting components assuming that some coefficients are fixed
#  - e3 and e4 give the same result
#  - e3 uses legacy format for all arguments
#  - e4 uses the new format for all arguments (not optimal here)
e3 <- btsr.extract(
  model = "BARMA", yt = m1$yt,
  xreg = covar[(burn + 1):N, ], linkg = "logit",
  coefs = list(
    phi = -0.65, theta = -0.25,
    beta = c(0.6, -0.002)
  ),
  fixed.values = list(alpha = 0, nu = 20),
  llk = TRUE, sco = TRUE, info = TRUE
)

e4 <- btsr.extract(
  model = "BARMA", yt = m1$yt,
  xreg = list(part1 = covar[(burn + 1):N, ]),
  linkg = list(g11 = "logit"),
  coefs = list(part1 = list(
    phi = -0.65, theta = -0.25,
    beta = c(0.6, -0.002)
  )),
  fixed.values = list(
    part1 = list(alpha = 0),
    part2 = list(alpha = 20)
  ),
  llk = TRUE, sco = TRUE, info = TRUE
)

#----------------------------------------------------
# comparing the simulated and the extracted values
#----------------------------------------------------
cbind(head(m1$mut), head(e1$mut), head(e2$mut), head(e3$mut), head(e4$mut))

#----------------------------------------------------
# comparing the log-likelihood values obtained (must be the all equal)
#----------------------------------------------------
c(e1$sll, e2$sll, e3$sll, e4$sll)

#----------------------------------------------------
# comparing the score vectors:
#----------------------------------------------------
# - e1 must have 6 values: dl/dro values and dl/dlambda values
# - e2 must be empty
# - e3 and e4 must have only the values corresponding
#    to the non-fixed coefficient. The value sof
#    dl/dlambda are the same as in e1, but dl/drho are
#    differente since the mixed derivatives are not present.
#----------------------------------------------------
round(e1$score, 4)
e2$score
e3$score
e4$score

#----------------------------------------------------
# comparing the information matrices.
#----------------------------------------------------
# - e1 must be a 6x6 matrix
# - e2 must be empty
# - e3 and e4 must have only the value corresponding
#    to the non-fixed coefficient
#----------------------------------------------------
round(e1$info.Matrix, 4)
e2$info.Matrix
e3$info.Matrix
e4$info.Matrix

#------------------------------------------------------------------------
# Generating a sample from a BARFIMAVX(1,d1,1)x(1,0,1) with d1 = 0.35
# (BARFIMAV with covariates)
# Here using the nre format for coefficients and xreg is required.
#------------------------------------------------------------------------
burn <- 100
n <- 500
N <- n + burn
covar1 <- cbind(sin(2 * pi * (1:N) / 50), 1:N)
covar2 <- sin(2 * pi * (1:N) / 25)

set.seed(1234)
m1 <- btsr.sim(
  model = "BARFIMAV",
  linkg = list(g11 = "logit", g2 = "default", g21 = "logit"),
  n = n, burn = burn, xreg = list(part1 = covar1, part2 = covar2),
  coefs = list(
    part1 = list(
      alpha = 0, d = 0.35, phi = -0.65,
      theta = -0.25, beta = c(0.6, -0.002)
    ),
    part2 = list(
      alpha = -3, phi = 0.25,
      theta = -0.2, beta = -0.15
    )
  ),
  error.scale = 1, complete = TRUE, vt.start = 0.02
)


# Extracting components assuming that all coefficients are non-fixed
e1 <- btsr.extract(
  model = "BARFIMAV", yt = m1$yt,
  xreg = list(part1 = covar1[(burn + 1):N, ], part2 = covar2[(burn + 1):N]),
  linkg = list(g11 = "logit", g2 = "default", g21 = "logit"),
  coefs = list(
    part1 = list(
      alpha = 0, d = 0.35, phi = -0.65,
      theta = -0.25, beta = c(0.6, -0.002)
    ),
    part2 = list(
      alpha = -3, phi = 0.25,
      theta = -0.2, beta = -0.15
    )
  ),
  vt.start = 0.02,
  llk = TRUE, sco = TRUE, info = TRUE, extra = TRUE, debug = TRUE
)

# score vector
e1$score

# information matrix
e1$info.Matrix

#########################################################################
#
#   Examples of usage of btsr.fit
#
#########################################################################
#------------------------------------------------------------------------
#  Generating a sample from a BARMAVX(1,0)x(0,1) (BARMAV with covariates)
#------------------------------------------------------------------------
burn <- 100
n <- 500
N <- n + burn
covar1 <- cbind(sin(2 * pi * (1:N) / 50), 1:N)
covar2 <- sin(2 * pi * (1:N) / 25)

set.seed(1234)
m1 <- btsr.sim(
  model = "BARMAV",
  linkg = list(g11 = "logit", g2 = "default", g21 = "logit"),
  n = n, burn = burn, xreg = list(part1 = covar1),
  coefs = list(
    part1 = list(
      alpha = 0, phi = -0.3,
      beta = c(0.6, -0.002)
    ),
    part2 = list(alpha = -2.5, theta = -0.4)
  ),
  error.scale = 1, complete = TRUE
)

# fitting the model
f1 <- btsr.fit(
  model = "BARMAV", yt = m1$yt, report = FALSE, info = TRUE,
  xreg = list(part1 = covar1[(burn + 1):N, ]),
  linkg = list(g11 = "logit", g2 = "default", g21 = "logit"),
  p = c(1, 0), q = c(0, 1)
)

# fitting the model using the name string for the parent model
#   - same result
f2 <- btsr.fit(
  model = "BARFIMAV", yt = m1$yt, report = FALSE, info = TRUE,
  xreg = list(part1 = covar1[(burn + 1):N, ]),
  linkg = list(g11 = "logit", g2 = "default", g21 = "logit"),
  p = c(1, 0), q = c(0, 1), d = FALSE
)

summary(f1, full.report = TRUE) # default
summary(f2, full.report = TRUE)

summary(f1, full.report = FALSE) # simplified output
summary(f2, full.report = FALSE)
}

}
\seealso{
\link{BARC.functions}: sim, extract and fit functions for BARC models

\link{BTSR.parent.models}: sim, extract and fit functions for parent models

\link{get.defaults}: Retrieve default arguments for BTSR package functions
}
