% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/COMPO2.R
\name{COMPO2}
\alias{COMPO2}
\title{The COMPO2 family (with mu as mean)}
\usage{
COMPO2(mu.link = "log", sigma.link = "identity")
}
\arguments{
\item{mu.link}{defines the mu.link, with "log" link as the default for the mu parameter.}

\item{sigma.link}{defines the sigma.link, with "identity" link as the default for the sigma.}
}
\value{
Returns a \code{gamlss.family} object which can be used
to fit a COMPO2 distribution
in the \code{gamlss()} function.
}
\description{
The function \code{COMPO2()} defines the
Conway-Maxwell-Poisson distribution
a two parameter distribution,
for a \code{gamlss.family} object to be used in GAMLSS
fitting using the function \code{gamlss()}.
This parameterization was
proposed by Ribeiro et al. (2020) and the main
characteristic is that \eqn{E(X)=\mu}.
}
\details{
The COMPO2 distribution with parameters \eqn{\mu} and \eqn{\sigma}
has a support 0, 1, 2, ... and mass function given by

\eqn{f(x | \mu, \sigma) = \left(\mu + \frac{\exp(\sigma)-1}{2 \exp(\sigma)} \right)^{x \exp(\sigma)} \frac{(x!)^{\exp(\sigma)}}{Z(\mu, \sigma)} }

with \eqn{\mu > 0}, \eqn{\sigma \in \Re} and

\eqn{Z(\mu, \sigma)=\sum_{j=0}^{\infty} \frac{\mu^j}{(j!)^\sigma}}.

The proposed functions here are based on the functions from
the COMPoissonReg package.
}
\examples{
# Example 1
# Generating some random values with
# known mu and sigma
y <- rCOMPO2(n=500, mu=10, sigma=-1)

# Fitting the model
library(gamlss)
mod1 <- gamlss(y~1, sigma.fo=~1, family=COMPO2,
               control=gamlss.control(n.cyc=500, trace=TRUE))

# Extracting the fitted values for mu and sigma
# using the inverse link function
exp(coef(mod1, what="mu"))
coef(mod1, what="sigma")

# Example 2
# Generating random values under some model

\dontrun{
# A function to simulate a data set with Y ~ COMPO2
gendat <- function(n) {
  x1 <- runif(n)
  x2 <- runif(n)
  mu    <- exp(2 + 1 * x1) # 12 approximately
  sigma <- 1 - 2 * x2      # 0 approximately
  y <- rCOMPO2(n=n, mu=mu, sigma=sigma)
  data.frame(y=y, x1=x1, x2=x2)
}

set.seed(123)
dat <- gendat(n=200)

# Fitting the model
mod2 <- NULL
mod2 <- gamlss(y~x1, sigma.fo=~x2, family=COMPO2, data=dat)

summary(mod2)
}
}
\references{
Ribeiro Jr, Eduardo E., et al. "Reparametrization of COM–Poisson regression
models with applications in the analysis of experimental data."
Statistical Modelling 20.5 (2020): 443-466.
}
\seealso{
\link{dCOMPO2}.
}
\author{
Freddy Hernandez, \email{fhernanb@unal.edu.co}
}
