% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/variant_SEM.R
\name{make_sem_structure}
\alias{make_sem_structure}
\title{Define Structural Equation Model (SEM) Matrices}
\usage{
make_sem_structure(
  data = NULL,
  design,
  covariate_cols = NULL,
  lambda_specs = NULL,
  b_specs = NULL,
  k_specs = NULL,
  g_specs = NULL,
  fixed_value = 0
)
}
\arguments{
\item{data}{A data frame containing a column named "subjects"
and any covariate columns specified in \code{covariate_cols}.}

\item{design}{An emc.design object, as created by the \code{design()} function.
The parameter names for the SEM are derived from \code{names(sampled_pars(design))}.}

\item{covariate_cols}{Character vector or NULL. Column names in \code{data} to be used
as covariates for K_mat and G_mat. If NULL, no covariates are processed.}

\item{lambda_specs}{A list defining factor loadings.
The list names should be factor names and each element should be a
character vector of parameter names (from \code{names(sampled_pars(design))}) that load onto that factor.
The first parameter listed for each factor will be fixed to 1 for identifiability.
Example: \code{list(Factor1 = c("v_Sleft", "a_Eneutral"), Factor2 = "t0")}
Here, \code{Lambda_mat["v_Sleft", "Factor1"]} would be 1.}

\item{b_specs}{A list defining regressions among factors.
List names are outcome factors, elements are character vectors of predictor factors.
Example: \code{list(Factor2 = "Factor1", Factor3 = c("Factor1", "Factor2"))}}

\item{k_specs}{A list defining covariate effects on subject-level parameters.
List names are parameter names (from \code{names(sampled_pars(design))}), elements are character vectors of covariate names
(must be present in \code{covariate_cols} and thus in the processed \code{covariates} data frame).
Example: \code{list(v_Sleft = "cov1", a_Eneutral = c("cov1", "cov2"))}}

\item{g_specs}{A list defining covariate effects on factors.
List names are factor names, elements are character vectors of covariate names.
Example: \code{list(Factor1 = "cov1", Factor2 = c("cov1", "cov2"))}}

\item{fixed_value}{Numeric. The value used for fixed paths in the matrices that
are not set to 1 for identifiability or \code{Inf} for estimation. Default is 0.}
}
\value{
A list containing:
\itemize{
\item \code{Lambda_mat}: The factor loading matrix.
\item \code{B_mat}: The matrix of regressions among factors.
\item \code{K_mat}: The matrix of covariate effects on subject-level parameters.
\item \code{G_mat}: The matrix of covariate effects on factors.
\item \code{par_names}: The subject-level parameter names derived from \code{sampled_pars(design)}.
\item \code{factor_names}: The provided SEM factor names.
\item \code{covariates}: A data frame with one row per unique subject and columns for each covariate,
containing the unique subject-level values. Column names are the covariate names.
}
}
\description{
This function helps create the specification matrices (Lambda, B, K, G) for an SEM.
It takes a design object, data, factor names, covariate column names, and list-based
specifications for the paths to be estimated.
The subject-level parameter names for Lambda_mat and K_mat rows are derived from \code{sampled_pars(design)}.
It validates that covariates are consistent per subject (subject column in \code{data} must be named "subjects")
and includes an aggregated subject-level covariate data frame named \code{covariates} in the output list.
For identifiability, the first parameter listed in \code{lambda_specs} for each factor is fixed to 1.
}
\examples{
# Create a design object (simplified from design.R example)
ADmat <- matrix(c(-1/2,1/2),ncol=1,dimnames=list(NULL,"diff"))
matchfun_example <- function(d) d$S==d$lR # Example match function

example_design_obj <- design(
  data = forstmann,
  model= LBA,
  matchfun=matchfun_example,
  formula=list(v~lM,sv~lM,B~E+lR,A~1,t0~1),
  contrasts=list(v=list(lM=ADmat)),
  constants=c(sv=log(1)),
)

# SEM Factor names

# Make a copy of forstmann for example modification
forstmann_mod <- forstmann
set.seed(123) # for reproducibility
subj_trait_values <- stats::setNames(rnorm(length(levels(forstmann_mod$subjects))),
                                    levels(forstmann_mod$subjects))
forstmann_mod$SubjTrait <- subj_trait_values[forstmann_mod$subjects]

my_cov_cols <- c("SubjTrait")

lambda_example_specs <- list(
  Speed = c("v", "v_lMdiff"), # "v" will be fixed to 1
  Caution = c("B", "B_Eneutral", "B_Eaccuracy", "B_lRright", "A") # "B" fixed to 1
)
b_example_specs <- list(Caution = "Speed")
k_example_specs <- list(t0 = "SubjTrait") # "SubjTrait" must be in my_cov_cols
g_example_specs <- list(Speed = "SubjTrait")

sem_settings_definition <- make_sem_structure(
  data = forstmann_mod,
  design = example_design_obj,
  covariate_cols = my_cov_cols,
  lambda_specs = lambda_example_specs,
  b_specs = b_example_specs,
  k_specs = k_example_specs,
  g_specs = g_example_specs
)

print(sem_settings_definition$Lambda_mat)
print(sem_settings_definition$B_mat)
print(sem_settings_definition$K_mat)
print(sem_settings_definition$G_mat)
print(head(sem_settings_definition$covariates))

}
