\name{gsom_Training}
\alias{gsom_Training}
\title{Train a Gower-SOM on mixed-attribute data}
\description{
Train a Self-Organizing Map (SOM) on datasets with mixed attributes
(numeric and categorical) using Gower distance to find the BMU and
heuristics to update categorical prototypes.
}
\details{
Learning rate and neighborhood radius decay exponentially per iteration:
\deqn{\alpha_t = \alpha_0 \exp(-t/T), \quad
r_t = r_0 \exp(-t/(T/\log r_0))}
where \eqn{T} is \code{num_iterations} and \eqn{r_0} is \code{radius}
(default \code{max(grid_rows, grid_cols)/2}). For categorical variables,
the prototype combines current and input values weighted by \eqn{\alpha_t}
and the neighborhood kernel; if \code{sampling = TRUE}, a weighted draw
is used; otherwise a weighted mode is applied.
}
\usage{
gsom_Training(data, grid_rows = 5, grid_cols = 5,
         learning_rate = 0.1, num_iterations = 100,
         radius = NULL, batch_size = 10,
         sampling = TRUE, set_seed = 123)
}
\arguments{
  \item{data}{\code{data.frame} with correctly typed columns (numeric, factor, etc.).}
  \item{grid_rows, grid_cols}{SOM grid dimensions (rows x cols).}
  \item{learning_rate}{Initial learning rate (decays exponentially).}
  \item{num_iterations}{Number of iterations.}
  \item{radius}{Initial neighborhood radius; default \code{max(grid_rows, grid_cols)/2}.}
  \item{batch_size}{Mini-batch size per iteration.}
  \item{sampling}{Logical; if \code{TRUE}, multinomial sampling for categorical updates,
                  else weighted mode.}
  \item{set_seed}{Integer random seed for reproducibility.}
}
\value{
An object of class \code{gowersom} with:
\itemize{
 \item \code{weights}: data.frame of trained neuron prototypes.
 \item \code{coords}: data.frame of grid coordinates per neuron.
}
}
\author{
  Patricio Sáez <patricsaez@udec.cl>;
  Patricio Salas <patricioasalas@udec.cl>
}

\references{
Sáez, P., Salas, P. Gower-SOM: a self-organizing map for mixed data with gower distance and heuristic adaptation for data analytics. Int J Data Sci Anal 22, 26 (2026). https://doi.org/10.1007/s41060-025-00941-6/."
}

\examples{
set.seed(1)
df <- data.frame(
  x1 = rnorm(50),
  x2 = rnorm(50),
  g  = factor(sample(letters[1:3], 50, TRUE))
)
fit <- gsom_Training(df, grid_rows = 3, grid_cols = 3,
                learning_rate = 0.1, num_iterations = 10,
                batch_size = 8, sampling = TRUE, set_seed = 123)
str(fit)
}
