\name{generateHSMM}
\alias{generateHSMM}
\title{Generate Data from a Hidden Semi-Markov Model}
\description{
Simulates observations and hidden states from a Hidden Semi-Markov Model (HSMM) with specified observation and dwell time distributions.
}
\usage{
generateHSMM(n, J, obsdist, dwelldist, obspar, dwellpar, Pi,
             delta = NULL, simtype = "nobs", shift = FALSE, seed = NULL)
}
\arguments{
\item{n}{Integer. The number of observations to generate (if \code{simtype = "nobs"}) or the number of state sequences (if \code{simtype = "nseq"}).}

\item{J}{Integer. The number of hidden states in the model.}

\item{obsdist}{Character string. The observation distribution. Supported distributions are: "norm", "pois", "weibull", "zip", "nbinom", "zinb", "exp", "gamma", "lnorm", "gev", "ZInormal", "ZIgamma".}

\item{dwelldist}{Character string. The dwell time distribution. Supported distributions are: "pois", "nbinom", "betabinom".}

\item{obspar}{List. Parameters for the observation distribution. Required parameters vary by distribution:
  \itemize{
    \item \code{norm}: \code{mean}, \code{sd}
    \item \code{pois}: \code{lambda}
    \item \code{weibull}: \code{shape}, \code{scale}
    \item \code{zip}: \code{pi}, \code{lambda}
    \item \code{nbinom}: \code{size}, \code{mu}
    \item \code{zinb}: \code{pi}, \code{size}, \code{mu}
    \item \code{exp}: \code{rate}
    \item \code{gamma}: \code{shape}, \code{rate}
    \item \code{lnorm}: \code{meanlog}, \code{sdlog}
    \item \code{gev}: \code{loc}, \code{scale}, \code{shape}
    \item \code{ZInormal}: \code{mean}, \code{sd}, \code{pi}
    \item \code{ZIgamma}: \code{shape}, \code{rate}, \code{pi}
  }
  Each parameter should be a vector of length \code{J} with values for each state.
}

\item{dwellpar}{List. Parameters for the dwell time distribution. Required parameters vary by distribution:
  \itemize{
    \item \code{pois}: \code{lambda}, \code{shift}
    \item \code{nbinom}: \code{shift}, \code{size}, \code{mu}
    \item \code{betabinom}: \code{size}, \code{alpha}, \code{beta}, \code{shift}
  }
  Each parameter should be a vector of length \code{J} with values for each state.
}

\item{Pi}{Matrix. The \code{J x J} transition probability matrix between states. Rows must sum to 1.}

\item{delta}{Numeric vector of length \code{J}. The initial state distribution. If \code{NULL}, the stationary distribution is computed from \code{Pi}.}

\item{simtype}{Character string. Either "nobs" (generate \code{n} observations) or "nseq" (generate \code{n} state sequences). Default is "nobs".}

\item{shift}{Logical. If \code{TRUE}, uses the shift parameter from \code{dwellpar}. If \code{FALSE} and no shift is provided in \code{dwellpar}, sets shift to 1 for all states. Default is \code{FALSE}.}

\item{seed}{Integer or \code{NULL}. Random seed for reproducibility. Default is \code{NULL}.}
}
\value{
A list containing:
\item{states}{Numeric vector of the simulated hidden state sequence.}
\item{x}{Numeric vector of the simulated observations.}
\item{N}{Integer. The number of observations generated.}
}
\details{
This function simulates data from a Hidden Semi-Markov Model where:
\itemize{
  \item Hidden states follow a Markov chain with transition matrix \code{Pi}
  \item Each state has an associated dwell time distribution that determines how long the process remains in that state
  \item Observations are generated from state-dependent distributions
}

The function supports multiple observation distributions including normal, Poisson, Weibull, zero-inflated Poisson (ZIP), negative binomial, zero-inflated negative binomial (ZINB), exponential, gamma, log-normal, generalized extreme value (GEV), zero-inflated normal and zero-inflated gamma.

Dwell time distributions include Poisson, negative binomial, and beta-binomial, all with optional shift parameters to ensure minimum dwell times.

When \code{simtype = "nobs"}, the function generates exactly \code{n} observations. When \code{simtype = "nseq"}, it generates \code{n} state sequences and the total number of observations depends on the realized dwell times.
}
\examples{
# Example with 3 states, normal observations, Poisson dwell times
J <- 3
# HSMM transition matrix
Pi <- matrix(c(0.0, 0.6, 0.4,
               0.5, 0.0, 0.5,
               0.3, 0.7, 0.0), nrow = 3, byrow = TRUE)

# Observation parameters (normal distribution)
obspar <- list(
  mean = c(-2, 0, 3),
  sd = c(1, 1.5, 2)
)

# Dwell time parameters (Poisson distribution)
dwellpar <- list(
  lambda = c(3, 5, 4)
)

# Generate 100 observations
sim_data <- generateHSMM(n = 100, J = J, obsdist = "norm", dwelldist = "pois",
                        obspar = obspar, dwellpar = dwellpar, Pi = Pi)

# View the results
head(sim_data$x)        # observations
head(sim_data$states)   # hidden states
sim_data$N              # total number of observations
}
\seealso{
\code{\link{generateHMM}} for Hidden Markov Models.
}
\author{
[Aimee Cody]
}



