# JapanAPIs - Access Japanese Data via Public APIs and Curated Datasets
# Version 0.1.1
# Copyright (c) 2025 Renzo Caceres Rossi
# Licensed under the MIT License.
# See the LICENSE file in the root directory for full license text.

#' Get Official Public Holidays in Japan for a Given Year
#'
#' @description
#' Retrieves the list of official public holidays in Japan for a specific year
#' using the Nager.Date public holidays API.
#' This function returns a tibble containing the date of the holiday, the name
#' in the local language (Japanese), and the English name.
#' It is useful for academic, planning, and data analysis purposes.
#' The information is retrieved directly from the Nager.Date API and reflects
#' the current status of holidays for the requested year.
#' The field names returned are consistent with the API structure.
#'
#' @param year An integer indicating the year (e.g., 2024 or 2025).
#'
#' @return A tibble with the following columns, or \code{NULL} if the API is unavailable:
#' \itemize{
#'   \item \code{date}: Date of the holiday (class \code{Date})
#'   \item \code{local_name}: Holiday name in the local language (Japanese)
#'   \item \code{name}: Holiday name in English
#' }
#'
#' @source Data obtained from the Nager.Date API: \url{https://date.nager.at/}
#'
#' @examples
#' \donttest{
#' # Requires internet connection
#' result <- get_japan_holidays(2024)
#' if (!is.null(result)) {
#'   print(result)
#' }
#'
#' result <- get_japan_holidays(2025)
#' if (!is.null(result)) {
#'   print(result)
#' }
#' }
#'
#' @importFrom httr GET status_code content timeout
#' @importFrom jsonlite fromJSON
#' @importFrom tibble tibble
#'
#' @export
get_japan_holidays <- function(year) {
  url <- sprintf("https://date.nager.at/api/v3/PublicHolidays/%s/JP", year)

  # Wrap the entire API call in tryCatch for robust error handling
  tryCatch({
    response <- httr::GET(url, httr::timeout(10))

    if (httr::status_code(response) != 200) {
      status <- httr::status_code(response)
      message(sprintf("API request failed with status code %d. The Nager.Date API may be temporarily unavailable or the year %d may not be supported.", status, year))
      return(NULL)
    }

    data <- jsonlite::fromJSON(httr::content(response, as = "text", encoding = "UTF-8"))

    if (length(data) == 0) {
      message(sprintf("No holiday data found for Japan in year %d.", year))
      return(NULL)
    }

    tibble::tibble(
      date = as.Date(data$date),
      local_name = data$localName,
      name = data$name
    )

  }, error = function(e) {
    message(sprintf("Error accessing Nager.Date API: %s. Please check your internet connection or try again later.", conditionMessage(e)))
    return(NULL)
  })
}
