# JapanAPIs - Access Japanese Data via Public APIs and Curated Datasets
# Version 0.1.1
# Copyright (c) 2025 Renzo Caceres Rossi
# Licensed under the MIT License.
# See the LICENSE file in the root directory for full license text.

# get_japan_energy_use

library(testthat)

test_that("get_japan_energy_use() returns a tibble with expected structure", {
  skip_on_cran()

  result <- get_japan_energy_use()

  # Structure
  expect_s3_class(result, "tbl_df")
  expect_equal(ncol(result), 4)

  # Column names
  expect_named(result, c("indicator", "country", "year", "value"))

  # Column types
  expect_type(result$indicator, "character")
  expect_type(result$country, "character")
  expect_type(result$year, "integer")
  expect_type(result$value, "double")
})

test_that("get_japan_energy_use() returns correct dimensions and years", {
  skip_on_cran()

  result <- get_japan_energy_use()

  # Rows 2010–2022 = 13 values
  expect_equal(nrow(result), 13)

  # Exact years expected
  expect_equal(sort(result$year), 2010:2022)

  # Sorted descending
  expect_true(all(diff(result$year) <= 0))
})

test_that("get_japan_energy_use() returns consistent values for Japan", {
  skip_on_cran()

  result <- get_japan_energy_use()

  # Country
  expect_true(all(result$country == "Japan"))

  # Indicator content
  expect_true(all(grepl("Energy use", result$indicator, ignore.case = TRUE)))
  expect_true(all(grepl("oil equivalent", result$indicator, ignore.case = TRUE)))

  # Values should be positive
  expect_true(all(result$value > 0, na.rm = TRUE))

  # Japan’s typical range is 2500–4000 kg → sanity test
  expect_true(all(result$value < 6000, na.rm = TRUE))
})

test_that("get_japan_energy_use() has non-missing numeric values", {
  skip_on_cran()

  result <- get_japan_energy_use()

  expect_true(is.numeric(result$value))
  expect_true(all(!is.na(result$value)))

  expect_true(is.integer(result$year))
  expect_true(all(!is.na(result$year)))
})

test_that("get_japan_energy_use() handles API errors gracefully", {
  skip_on_cran()

  result <- tryCatch(get_japan_energy_use(), error = function(e) NULL)

  # Should return NULL or tibble
  expect_true(is.null(result) || inherits(result, "tbl_df"))
})
