# JapanAPIs - Access Japanese Data via Public APIs and Curated Datasets
# Version 0.1.1
# Copyright (c) 2025 Renzo Caceres Rossi
# Licensed under the MIT License.
# See the LICENSE file in the root directory for full license text.

# get_japan_holidays

library(testthat)

test_that("get_japan_holidays() returns a tibble with expected structure", {
  skip_on_cran()

  result <- get_japan_holidays(2025)

  expect_s3_class(result, "tbl_df")
  expect_equal(ncol(result), 3)
  expect_named(result, c("date", "local_name", "name"))
  expect_s3_class(result$date, "Date")
  expect_type(result$local_name, "character")
  expect_type(result$name, "character")
})

test_that("get_japan_holidays() returns valid data for 2025", {
  skip_on_cran()

  result <- get_japan_holidays(2025)

  expect_true(nrow(result) > 0)

  # All dates in 2025
  expect_true(all(format(result$date, "%Y") == "2025"))

  # Chronological order is NOT required — both TRUE and FALSE are valid
  expect_true(all(!is.na(result$date)))

  # Integrity (no NA)
  expect_true(all(!is.na(result$local_name)))
  expect_true(all(!is.na(result$name)))
})

test_that("get_japan_holidays() returns consistent holiday information", {
  skip_on_cran()

  result <- get_japan_holidays(2025)

  expect_true(all(nchar(result$local_name) > 0))
  expect_true(all(nchar(result$name) > 0))

  expect_true(any(grepl("New Year", result$name, ignore.case = TRUE)))
  expect_true(any(grepl("Foundation", result$name, ignore.case = TRUE)))
})

test_that("get_japan_holidays() works with different years", {
  skip_on_cran()

  r2024 <- get_japan_holidays(2024)
  r2025 <- get_japan_holidays(2025)

  expect_s3_class(r2024, "tbl_df")
  expect_s3_class(r2025, "tbl_df")

  expect_true(nrow(r2024) > 0)
  expect_true(nrow(r2025) > 0)

  expect_true(all(format(r2024$date, "%Y") == "2024"))
  expect_true(all(format(r2025$date, "%Y") == "2025"))
})

test_that("get_japan_holidays() handles unsupported or extreme years gracefully", {
  skip_on_cran()

  years_to_test <- list(1900, 2100, NA)

  for (yr in years_to_test) {
    result <- tryCatch(get_japan_holidays(yr), error = function(e) NULL)

    # API returning NULL is valid
    # API returning tibble is valid
    # Messages are valid
    expect_true(is.null(result) || inherits(result, "tbl_df"))
  }
})
