#' @title Available Color Scales for QQ Regression Plots
#'
#' @description
#' Returns information about the available color scales for 3D surface plots 
#' and heatmaps in the QuantileOnQuantile package.
#'
#' @param show_preview Logical. If \code{TRUE}, prints information about 
#'   each color scale. Default is \code{TRUE}.
#'
#' @return A character vector of available color scale names, invisibly.
#'
#' @details
#' The following color scales are available:
#' \describe{
#'   \item{Jet}{MATLAB-style rainbow colorscale. Transitions from blue 
#'     through cyan, green, yellow to red. Best for general visualization 
#'     where the full range of values is important.}
#'   \item{BlueRed}{Diverging colorscale from blue to red. Also known as 
#'     "COVID style" in some contexts. Best for data with a meaningful 
#'     center point (e.g., zero for coefficients).}
#'   \item{Viridis}{Perceptually uniform colorscale. Designed to be 
#'     colorblind-friendly and print well in grayscale.}
#'   \item{Plasma}{Another perceptually uniform colorscale. Similar 
#'     benefits to Viridis but with a different color palette.}
#' }
#'
#' @examples
#' # Show available color scales
#' qq_colorscales()
#' 
#' # Get scales without printing
#' scales <- qq_colorscales(show_preview = FALSE)
#' print(scales)
#'
#' @export
qq_colorscales <- function(show_preview = TRUE) {
  
  scales <- c("Jet", "BlueRed", "Viridis", "Plasma")
  
  descriptions <- list(
    Jet = "MATLAB-style rainbow (blue -> cyan -> green -> yellow -> red)",
    BlueRed = "Diverging scale (blue = low, red = high)",
    Viridis = "Perceptually uniform, colorblind-friendly",
    Plasma = "Perceptually uniform, high contrast"
  )
  
  if (show_preview) {
    cat("\n")
    cat("Available Color Scales for QQ Regression Plots\n")
    cat("===============================================\n\n")
    
    for (scale in scales) {
      cat(sprintf("  %-10s : %s\n", scale, descriptions[[scale]]))
    }
    
    cat("\n")
    cat("Usage:\n")
    cat("  plot_qq_3d(result, colorscale = \"Jet\")\n")
    cat("  plot_qq_heatmap(result, colorscale = \"Viridis\")\n")
    cat("\n")
  }
  
  invisible(scales)
}


#' @title Summary Statistics for QQ Regression
#'
#' @description
#' Computes comprehensive summary statistics for QQ regression results.
#'
#' @param qq_result An object of class "qq_regression".
#' @param alpha Numeric. Significance level for counting significant results. 
#'   Default is 0.05.
#'
#' @return A data frame with summary statistics including:
#' \itemize{
#'   \item Mean, median, min, max of coefficients
#'   \item Mean, median, min, max of R-squared
#'   \item Number of significant results at specified alpha
#'   \item Total number of results
#' }
#'
#' @examples
#' # Generate example data
#' set.seed(42)
#' n <- 200
#' x <- rnorm(n)
#' y <- 0.5 * x + rnorm(n, sd = 0.5)
#'
#' # Run QQ regression
#' result <- qq_regression(y, x, verbose = FALSE)
#' 
#' # Get summary statistics
#' stats <- qq_statistics(result)
#' print(stats)
#'
#' @export
#' @importFrom stats median sd
qq_statistics <- function(qq_result, alpha = 0.05) {
  
  if (!inherits(qq_result, "qq_regression")) {
    stop("'qq_result' must be a qq_regression object")
  }
  
  results <- qq_result$results[complete.cases(qq_result$results), ]
  
  if (nrow(results) == 0) {
    warning("No complete results available")
    return(data.frame(Statistic = character(0), Value = numeric(0)))
  }
  
  stats <- data.frame(
    Statistic = c(
      "Mean Coefficient", 
      "Median Coefficient",
      "Min Coefficient",
      "Max Coefficient",
      "SD Coefficient",
      "Mean R-squared",
      "Median R-squared",
      "Min R-squared",
      "Max R-squared",
      paste0("Significant (p < ", alpha, ")"),
      "Total Results"
    ),
    Value = c(
      round(mean(results$coefficient, na.rm = TRUE), 4),
      round(median(results$coefficient, na.rm = TRUE), 4),
      round(min(results$coefficient, na.rm = TRUE), 4),
      round(max(results$coefficient, na.rm = TRUE), 4),
      round(sd(results$coefficient, na.rm = TRUE), 4),
      round(mean(results$r_squared, na.rm = TRUE), 4),
      round(median(results$r_squared, na.rm = TRUE), 4),
      round(min(results$r_squared, na.rm = TRUE), 4),
      round(max(results$r_squared, na.rm = TRUE), 4),
      sum(results$p_value < alpha, na.rm = TRUE),
      nrow(results)
    )
  )
  
  return(stats)
}


#' @title Convert QQ Results to Matrix
#'
#' @description
#' Converts QQ regression results to a matrix format suitable for base R 
#' plotting or export.
#'
#' @param qq_result An object of class "qq_regression" or a data frame.
#' @param type Character string. Type of values: "coefficient", "rsquared", 
#'   or "pvalue". Default is "coefficient".
#'
#' @return A matrix with y_quantiles as rows and x_quantiles as columns.
#'
#' @examples
#' # Generate example data
#' set.seed(42)
#' n <- 200
#' x <- rnorm(n)
#' y <- 0.5 * x + rnorm(n, sd = 0.5)
#'
#' # Run QQ regression
#' result <- qq_regression(y, x, verbose = FALSE)
#' 
#' # Convert to matrix
#' coef_matrix <- qq_to_matrix(result, type = "coefficient")
#' print(dim(coef_matrix))
#'
#' @export
qq_to_matrix <- function(qq_result, 
                         type = c("coefficient", "rsquared", "pvalue")) {
  
  # Extract results
  if (inherits(qq_result, "qq_regression")) {
    results <- qq_result$results
  } else if (is.data.frame(qq_result)) {
    results <- qq_result
  } else {
    stop("'qq_result' must be a qq_regression object or data frame")
  }
  
  type <- match.arg(type)
  
  # Select variable
  plot_var <- switch(type,
                     "coefficient" = results$coefficient,
                     "rsquared" = results$r_squared,
                     "pvalue" = results$p_value)
  
  # Get unique quantiles
  y_unique <- sort(unique(results$y_quantile))
  x_unique <- sort(unique(results$x_quantile))
  
  # Create matrix
  z_matrix <- matrix(NA_real_, nrow = length(y_unique), ncol = length(x_unique))
  rownames(z_matrix) <- as.character(y_unique)
  colnames(z_matrix) <- as.character(x_unique)
  
  for (i in seq_along(y_unique)) {
    for (j in seq_along(x_unique)) {
      idx <- which(results$y_quantile == y_unique[i] & 
                     results$x_quantile == x_unique[j])
      if (length(idx) > 0) {
        z_matrix[i, j] <- plot_var[idx[1]]
      }
    }
  }
  
  return(z_matrix)
}


#' @title Export QQ Results to CSV
#'
#' @description
#' Exports QQ regression results to a CSV file.
#'
#' @param qq_result An object of class "qq_regression".
#' @param file Character string. Path to the output file.
#' @param digits Integer. Number of decimal places for rounding. Default is 4.
#'
#' @return Invisible \code{NULL}. Called for its side effect of writing 
#'   a CSV file.
#'
#' @examples
#' \donttest{
#' # Generate example data
#' set.seed(42)
#' n <- 200
#' x <- rnorm(n)
#' y <- 0.5 * x + rnorm(n, sd = 0.5)
#'
#' # Run QQ regression
#' result <- qq_regression(y, x, verbose = FALSE)
#' 
#' # Export to CSV
#' qq_export(result, file.path(tempdir(), "qq_results.csv"))
#' }
#'
#' @export
#' @importFrom utils write.csv
qq_export <- function(qq_result, file, digits = 4) {
  
  if (!inherits(qq_result, "qq_regression")) {
    stop("'qq_result' must be a qq_regression object")
  }
  
  if (missing(file)) {
    stop("Please specify output file path")
  }
  
  results <- qq_result$results
  
  # Round numeric columns
  numeric_cols <- c("coefficient", "std_error", "t_value", "p_value", "r_squared")
  results[numeric_cols] <- lapply(results[numeric_cols], round, digits)
  results$y_quantile <- round(results$y_quantile, 2)
  results$x_quantile <- round(results$x_quantile, 2)
  
  # Remove incomplete cases
  results <- results[complete.cases(results), ]
  
  write.csv(results, file, row.names = FALSE)
  
  invisible(NULL)
}
