#' Coefficient matrices of the MA represention
#'
#' Returns the estimated coefficient matrices of the moving average
#' representation of a stable VAR(p).
#'
#' This is a modification of \code{vars::Phi.varest()} for the class
#' "varshrinkest", preventing redundant copying of data matrix objects.
#'
#' @param x An object of class "varshrinkest",
#' generated by \code{VARshrink()}.
#' @param nstep An integer specifying the number of moving error
#' coefficient matrices to be calculated.
#' @param ... Currently not used.
#' @returns An array with dimension \eqn{K \times K \times (nstep + 1)} holding
#' the estimated coefficients of the moving average representation. The first
#' slice of the array is the starting value, i.e., \eqn{\mathbf{\Phi}_0}.
#' @seealso \code{\link[vars]{Phi}}
#' @export
Phi.varshrinkest <- function(x, nstep = 10, ...) {
  if (!(inherits(x, "varest"))) {
    stop("\nPlease provide an object inheriting class 'varest'.\n")
  }
  nstep <- abs(as.integer(nstep))
  K <- x$K
  p <- x$p
  A <- as.array(Acoef_sh(x))
  if (nstep >= p) {
    As <- array(0, dim = c(K, K, nstep + 1))
    for (i in (p + 1):(nstep + 1)) {
      As[, , i] <- matrix(0, nrow = K, ncol = K)
    }
  } else {
    As <- array(0, dim = c(K, K, p))
  }
  for (i in 1:p) {
    As[, , i] <- A[[i]]
  }
  Phi <- array(0, dim = c(K, K, nstep + 1))
  Phi[, , 1] <- diag(K)
  Phi[, , 2] <- Phi[, , 1] %*% As[, , 1]
  if (nstep > 1) {
    for (i in 3:(nstep + 1)) {
      tmp1 <- Phi[, , 1] %*% As[, , i - 1]
      tmp2 <- matrix(0, nrow = K, ncol = K)
      idx <- (i - 2):1
      for (j in 1:(i - 2)) {
        tmp2 <- tmp2 + Phi[, , j + 1] %*% As[, , idx[j]]
      }
      Phi[, , i] <- tmp1 + tmp2
    }
  }
  return(Phi)
}
