% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/BIOMOD_RangeSize.R
\name{BIOMOD_RangeSize}
\alias{BIOMOD_RangeSize}
\title{Analyze the range size differences between projections of species distribution models}
\usage{
BIOMOD_RangeSize(
  proj.current,
  proj.future,
  models.chosen = "all",
  metric.binary = NULL
)
}
\arguments{
\item{proj.current}{a \code{BIOMOD.projection.out} object containing the initial projection(s) 
of the (ensemble) species distribution model(s)}

\item{proj.future}{a \code{BIOMOD.projection.out} object containing the final binary projection(s) 
of the (ensemble) species distribution model(s)}

\item{models.chosen}{a \code{vector} containing model names to be kept, must be either 
\code{all} or a sub-selection of model names that can be obtained with the 
\code{\link{get_projected_models}} function}

\item{metric.binary}{(\emph{optional, default} \code{NULL}) \cr 
A \code{vector} containing evaluation metric selected to transform predictions into binary 
values, must be among \code{POD}, \code{FAR}, \code{POFD}, \code{SR}, \code{ACCURACY}, 
\code{BIAS}, \code{AUCroc}, \code{AUCprg}, \code{TSS}, \code{KAPPA}, \code{OR}, \code{ORSS}, \code{CSI}, 
\code{ETS}, \code{BOYCE}, \code{MPA}}
}
\value{
A \code{BIOMOD.rangesize.out} containing principaly two objects :
\describe{
  \item{Compt.By.Species}{a \code{data.frame} containing the summary of range change for each 
  comparison
  \itemize{
    \item \code{Loss} : number of pixels predicted to be lost
    \item \code{Stable_Abs} : number of pixels not currently occupied and not predicted to be
    \item \code{Stable_Pres} : number of pixels currently occupied and predicted to remain 
    occupied
    \item \code{Gain} : number of pixels predicted to be gained
    \item \code{PercLoss} : percentage of pixels currently occupied and predicted to be lost 
    (\code{Loss / (Loss + Stable_Pres)})
    \item \code{PercGain} : percentage of pixels predicted to be gained compare to the 
    number of pixels currently occupied (\code{Gain / (Loss + Stable_Pres)})
    \item \code{SpeciesRangeChange} : percentage of pixels predicted to change (loss or gain) 
    compare to the number of pixels currently occupied (\code{PercGain - PercLoss})
    \item \code{CurrentRangeSize} : number of pixels currently occupied
    \item \code{FutureRangeSize0Disp} : number of pixels predicted to be occupied, assuming 
    no migration
    \item \code{FutureRangeSize1Disp} : number of pixels predicted to be occupied, assuming 
    migration
  }
  }
  \item{loss.gain}{an object in the same form than the input data (\code{proj.current} and 
  \code{proj.future}) and containing a value for each point/pixel of each comparison among :
  \itemize{
    \item \code{-2} : predicted to be lost
    \item \code{-1} : predicted to remain occupied
    \item \code{0} : predicted to remain unoccupied
    \item \code{1} : predicted to be gained
  }
  }
  \item{Diff.By.Pixel}{an object in the same form than the input data (\code{proj.current} and 
  \code{proj.future}) and containing a value for each point/pixel of each comparison obtain with :
  \itemize{
  \item \code{Future - 2* Current} for binary data
  \item \code{Future - Current} for nonbinary after rescaling Future and Current from 0 to 1.
  }
  }
}
}
\description{
This function allows to calculate the absolute number of locations (pixels) lost, 
stable and gained, as well as the corresponding relative proportions, between two (or more) 
binary projections of (ensemble) species distribution models (\emph{which can represent new 
time scales or environmental scenarios for example}).
}
\examples{
library(terra)

# Load species occurrences (6 species available)
data(DataSpecies)
head(DataSpecies)

# Select the name of the studied species
myRespName <- 'GuloGulo'

# Get corresponding presence/absence data
myResp <- as.numeric(DataSpecies[, myRespName])

# Get corresponding XY coordinates
myRespXY <- DataSpecies[, c('X_WGS84', 'Y_WGS84')]

# Load environmental variables extracted from BIOCLIM (bio_3, bio_4, bio_7, bio_11 & bio_12)
data(bioclim_current)
myExpl <- terra::rast(bioclim_current)

\dontshow{
myExtent <- terra::ext(0,30,45,70)
myExpl <- terra::crop(myExpl, myExtent)
}

# --------------------------------------------------------------- #
file.out <- paste0(myRespName, "/", myRespName, ".AllModels.models.out")
if (file.exists(file.out)) {
  myBiomodModelOut <- get(load(file.out))
} else {

  # Format Data with true absences
  myBiomodData <- BIOMOD_FormatingData(resp.name = myRespName,
                                       resp.var = myResp,
                                       resp.xy = myRespXY,
                                       expl.var = myExpl)

  # Model single models
  myBiomodModelOut <- BIOMOD_Modeling(bm.format = myBiomodData,
                                      modeling.id = 'AllModels',
                                      models = c('RF', 'GLM'),
                                      CV.strategy = 'random',
                                      CV.nb.rep = 2,
                                      CV.perc = 0.8,
                                      OPT.strategy = 'bigboss',
                                      metric.eval = c('TSS', 'AUCroc'),
                                      var.import = 3,
                                      seed.val = 42)
}

models.proj <- get_built_models(myBiomodModelOut, algo = "RF")
  # Project single models
  myBiomodProj <- BIOMOD_Projection(bm.mod = myBiomodModelOut,
                                    proj.name = 'CurrentRangeSize',
                                    new.env = myExpl,
                                    models.chosen = models.proj,
                                    metric.binary = 'all',
                                    build.clamping.mask = TRUE)


# --------------------------------------------------------------- #
# Load environmental variables extracted from BIOCLIM (bio_3, bio_4, bio_7, bio_11 & bio_12)
data(bioclim_future)
myExplFuture <- terra::rast(bioclim_future)
\dontshow{
myExtent <- terra::ext(0,30,45,70)
myExplFuture <- terra::crop(myExplFuture, myExtent)
}
# Project onto future conditions
myBiomodProjectionFuture <- BIOMOD_Projection(bm.mod = myBiomodModelOut,
                                              proj.name = 'FutureRangeSize',
                                              new.env = myExplFuture,
                                              models.chosen = models.proj,
                                              metric.binary = 'TSS')
                                              
# Compute differences
myBiomodRangeSize <- BIOMOD_RangeSize(proj.current = myBiomodProj,
                                      proj.future = myBiomodProjectionFuture,
                                      metric.binary = "TSS")


# Represent main results
bm_PlotRangeSize(bm.range = myBiomodRangeSize)



}
\seealso{
\code{\link{BIOMOD_Projection}}, \code{\link{BIOMOD_EnsembleForecasting}}, 
\code{\link{bm_PlotRangeSize}}

Other Main functions: 
\code{\link{BIOMOD_EnsembleForecasting}()},
\code{\link{BIOMOD_EnsembleModeling}()},
\code{\link{BIOMOD_FormatingData}()},
\code{\link{BIOMOD_LoadModels}()},
\code{\link{BIOMOD_Modeling}()},
\code{\link{BIOMOD_Projection}()}
}
\author{
Maya Guéguen, Hélène Blancheteau
}
\concept{Main functions}
\keyword{"species}
\keyword{change"}
\keyword{gain}
\keyword{loss}
\keyword{projections}
\keyword{range}
