% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/summary.bml.R
\name{summary.bml}
\alias{summary.bml}
\title{Summarize a fitted bml model}
\usage{
\method{summary}{bml}(object, r = 3, ...)
}
\arguments{
\item{object}{A fitted model object of class \code{"bml"} returned by \code{\link{bml}}.}

\item{r}{Number of decimal places for rounding numeric output. Default: 3.}

\item{...}{Additional arguments (currently unused).}
}
\value{
A data frame of class \code{"bml_summary"} containing rounded parameter
  estimates with the following columns:
  \itemize{
    \item \code{Parameter}: Labeled parameter names
    \item \code{mean}: Posterior mean
    \item \code{sd}: Posterior standard deviation
    \item \code{lb}: Lower bound of 95\% credible interval
    \item \code{ub}: Upper bound of 95\% credible interval
  }

  The object includes metadata attributes printed above the table:
  \itemize{
    \item Outcome family and link function
    \item Estimate type (posterior mean from MCMC)
    \item Credible interval specification (95\% equal-tailed)
    \item Level specification (mm and hm block details)
    \item DIC (Deviance Information Criterion) for model comparison
  }
}
\description{
S3 method for summarizing \code{bml} model objects. Returns a formatted table
of parameter estimates with posterior means, standard deviations, and credible
intervals, along with model information and convergence statistics.
}
\details{
The summary method rounds all numeric values for readability while preserving
the underlying structure and metadata from the fitted model. All columns remain
accessible via standard data frame indexing (e.g., \code{$Parameter}, \code{$mean}).

For Cox models with piecewise baseline hazards (when \code{cox_intervals} is
specified), the outcome description includes the number of intervals used.
}
\examples{
\donttest{
data(coalgov)

# Fit model
m1 <- bml(
  Surv(dur_wkb, event_wkb) ~ 1 + majority +
    mm(id = id(pid, gid), vars = vars(cohesion), fn = fn(w ~ 1/n), RE = TRUE) +
    hm(id = id(cid), type = "RE"),
  family = "Weibull",
  data = coalgov
)

# View summary
summary(m1)

# Summary with more decimal places
summary(m1, r = 4)

# Access specific columns
s <- summary(m1)
s$Parameter  # Parameter names
s$mean       # Posterior means
s$lb         # Lower credible bounds

# Custom posterior summaries (requires monitor = TRUE)
# Extract posterior draws as a tidy data frame
draws <- coda::as.mcmc.list(m1$jags.out$BUGSoutput) |> as.matrix() |> as_tibble()

# Select specific parameters and compute custom summaries
draws |>
  dplyr::select(dplyr::starts_with("b[")) |>
  tidyr::pivot_longer(everything(), names_to = "param") |>
  dplyr::group_by(param) |>
  dplyr::summarise(
    median = median(value),
    mad    = mad(value),
    q05    = quantile(value, 0.05),
    q95    = quantile(value, 0.95)
  )
}

}
\seealso{
\code{\link{bml}}, \code{\link{monetPlot}}, \code{\link{mcmcDiag}}
}
\author{
Benjamin Rosche <benrosche@nyu.edu>
}
