% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/confint.cgaim.R, R/confint.boot.cgaim.R
\name{confint.cgaim}
\alias{confint.cgaim}
\alias{confint.boot.cgaim}
\title{Confidence intervals}
\usage{
\method{confint}{cgaim}(object, parm, level = 0.95, type = c("normal",
  "bootstrap"), B = 100, ...)

\method{confint}{boot.cgaim}(object, parm, level = 0.95, ...)
}
\arguments{
\item{object}{A \code{cgaim} or \code{boot.cgaim} object.}

\item{parm}{The model components for which to get confidence intervals. 
One or several of: \code{"alpha"} for index weights,
\code{"beta"} for scaling coefficients and \code{"g"} for ridge function. By default,
returns confidence intervals for all components.}

\item{level}{The level of confidence intervals. Default to 0.95.}

\item{type}{The type of confidence intervals. Either \code{"normal"} (the default)
or \code{"bootstrap"}. See details.}

\item{B}{The number of samples to be simulated.}

\item{...}{Additional parameters to be passed to \code{\link{boot.cgaim}} for bootstrap confidence intervals.}
}
\value{
A list of confidence intervals. Contains one element per model
component in the \code{parm} argument.
}
\description{
Computes confidence intervals for the CGAIM components.
}
\details{
Two types of confidence intervals are currently implemented in the function.
When \code{type = "normal"}, confidence intervals are computed assuming 
components are normally distributed. Beta coefficients are treated as
regular linear regression coefficients and g as regular smooth functions
estimated by (shape-constrained) generalized additive models. For alpha
coefficients, we consider a linear transformation mapping them to a 
Truncated Multivariate Normal distribution (i.e. with only bound constraints). 
Simulation from the TMVN are performed (see \code{\link[TruncatedNormal]{tmvnorm}}) 
and transformed 
back into the original coefficient space (i.e. with linear constraints). The parameter \code{B} controls the number of simulations from the TMVN. 
Confidence intervals are computed as the percentiles of these simulated
coefficients, ensuring the confidence region is entirely within the feasible
region defined by the constraints.

When \code{type = "bootstrap"}, confidence intervals are estimated by 
percentile bootstrap. \code{\link{boot.cgaim}} is called internally
to create \code{B} samples of model components, and confidence intervals are then computed
as the percentiles of bootstrap samples. Alternatively, the user can directly
call \code{\link{boot.cgaim}} and feed the result into 
\code{confint.boot.cgaim}.
}
\note{
Confidence intervals for the g functions are evaluated on the 
same \code{n} index values as the functions in \code{object}.
}
\examples{
# A simple CGAIM
n <- 200
x1 <- rnorm(n)
x2 <- x1 + rnorm(n)
z <- x1 + x2
y <- z + rnorm(n)
df1 <- data.frame(y, x1, x2) 
ans <- cgaim(y ~ g(x1, x2, acons = list(monotone = 1)), data = df1)

# Normal confidence intervals
set.seed(1)
ci1 <- confint(ans, B = 1000)
ci1$alpha
ci1$beta

# Select only alphas: identical to above result
set.seed(1)
confint(ans, B = 1000, parm = "alpha")

# Select only betas: identical to above result
set.seed(1)
confint(ans, B = 1000, parm = "beta")

# Confidence intervals by bootstrap (more computationally intensive, B should be increased)
set.seed(2)
ci2 <- confint(ans, type = "boot", B = 10)

# Alternatively, bootstrap samples can be performed beforehand
set.seed(2) 
boot1 <- boot.cgaim(ans, B = 10)
ci3 <- confint(boot1)

}
\references{
Masselot, P. and others, 2022. Constrained groupwise additive index models.
    Biostatistics.
    
  Pya, N., Wood, S.N., 2015. Shape constrained additive models. 
   Stat. Comput. 25, 543–559. 
   
  Wood, S.N., 2017. Generalized Additive Models: An Introduction with R, 
    2nd ed, Texts in Statistical Science. Chapman and Hall/CRC.
    
  DiCiccio, T.J., Efron, B., 1996. Bootstrap Confidence Intervals. 
    Statistical Science 11, 189–212.
    
  Carpenter, J., Bithell, J., 2000. Bootstrap confidence intervals: 
    when, which, what? A practical guide for medical statisticians. 
    Statistics in Medicine 19, 1141–1164.
}
\seealso{
\code{\link{boot.cgaim}} for bootstrapping.
}
