% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/hcl_palettes.R
\name{hcl_palettes}
\alias{hcl_palettes}
\alias{print.hcl_palettes}
\alias{summary.hcl_palettes}
\alias{plot.hcl_palettes}
\alias{qualitative_hcl}
\alias{sequential_hcl}
\alias{diverging_hcl}
\alias{diverge_hcl}
\title{HCL Color Palettes}
\usage{
hcl_palettes(type = NULL, palette = NULL, plot = FALSE, n = 5L, ...)

\method{print}{hcl_palettes}(x, ...)

\method{summary}{hcl_palettes}(object, ...)

\method{plot}{hcl_palettes}(x, n = 5L, fixup = TRUE, off = NULL, border = NULL, ...)

qualitative_hcl(
  n,
  h = c(0, 360 * (n - 1)/n),
  c = 80,
  l = 60,
  fixup = TRUE,
  alpha = 1,
  palette = NULL,
  rev = FALSE,
  register = "",
  ...,
  h1,
  h2,
  c1,
  l1
)

sequential_hcl(
  n,
  h = 260,
  c = 80,
  l = c(30, 90),
  power = 1.5,
  gamma = NULL,
  fixup = TRUE,
  alpha = 1,
  palette = NULL,
  rev = FALSE,
  register = "",
  ...,
  h1,
  h2,
  c1,
  c2,
  l1,
  l2,
  p1,
  p2,
  cmax,
  c.
)

diverging_hcl(
  n,
  h = c(260, 0),
  c = 80,
  l = c(30, 90),
  power = 1.5,
  gamma = NULL,
  fixup = TRUE,
  alpha = 1,
  palette = NULL,
  rev = FALSE,
  register = "",
  ...,
  h1,
  h2,
  c1,
  l1,
  l2,
  p1,
  p2,
  cmax
)
}
\arguments{
\item{type}{character indicating type of HCL palette.}

\item{palette}{character. Name of HCL color palette.}

\item{plot}{logical. Should the selected HCL color palettes be visualized?}

\item{n}{the number of colors (\eqn{\ge 1}{>= 1}) to be in the palette.}

\item{\dots}{Other arguments passed to \code{\link{hex}}.}

\item{x, object}{A \code{hcl_palettes} object.}

\item{fixup}{logical. Should the color be corrected to a valid RGB value?}

\item{off}{numeric. Vector of length 2 indicating horizontal and vertical
offsets between the color rectangles displayed.}

\item{border}{character. Color of rectangle borders.}

\item{h, h1, h2}{hue value in the HCL color description, has to be in [0, 360].}

\item{c, c., c1, c2}{chroma value in the HCL color description.}

\item{l, l1, l2}{luminance value in the HCL color description.}

\item{alpha}{numeric vector of values in the range \code{[0, 1]} for alpha
transparency channel (0 means transparent and 1 means opaque).}

\item{rev}{logical. Should the color palette vector be returned in reverse order?}

\item{register}{character. If set to a non-empty character string, the corresponding
palette is registered with that name for subsequent use (within the same session).}

\item{power, p1, p2}{control parameter determining how chroma and luminance should
be increased (1 = linear, 2 = quadratic, etc.).}

\item{gamma}{Deprecated.}

\item{cmax}{Maximum chroma value in the HCL color description.}
}
\value{
\code{qualitative_hcl}, \code{sequential_hcl}, \code{diverging_hcl} return
a vector of \code{n} color strings (hex codes).

The function \code{hcl_palettes} returns a data frame of class \code{"hcl_palettes"}
where each row contains information about one of the requested palettes (name, type,
HCL trajectory coordinates). Suitable \code{print}, \code{summary}, and \code{plot}
methods are available.
}
\description{
Qualitative, sequential (single-hue and multi-hue), and diverging
color palettes based on the HCL (hue-chroma-luminance) color model.
}
\details{
The HCL (hue-chroma-luminance) color model is a perceptual color model
obtained by using polar coordinates in CIE \code{\link{LUV}} space
(i.e., \code{\link{polarLUV}}), where steps of equal size correspond to
approximately equal perceptual changes in color. By taking polar coordinates
the resulting three dimensions capture the three perceptual axes very well:
hue is the type of color, chroma the colorfulness compared
to the corresponding gray, and luminance the brightness. This makes it relatively
easy to create balanced palettes through trajectories in this HCL space.
In contrast, in the more commonly-used \code{\link{HSV}} (hue-saturation-value)
model (a simple transformation of \code{\link{RGB}}), the three axes are
confounded so that luminance changes along with the hue leading to very
unbalanced palettes (see \code{\link{rainbow_hcl}} for further illustrations).

Three types of palettes are derived based on the HCL model: \itemize{
  \item Qualitative: Designed for coding categorical information, i.e.,
    where no particular ordering of categories is available and every color
    should receive the same perceptual weight.
  \item Sequential: Designed for coding ordered/numeric information, i.e.,
    where colors go from high to low (or vice versa).
  \item Diverging: Designed for coding numeric information around a central
    neutral value, i.e., where colors diverge from neutral to two extremes.
}
The corresponding functions are \code{qualitative_hcl}, \code{sequential_hcl},
and \code{diverging_hcl}. Their construction principles are explained in more detail below.
At the core is the luminance axis (i.e., light-dark contrasts):
These are easily decoded by humans and matched to high-low differences in the underlying
data. Therefore, \code{sequential_hcl} palettes are always based on a \emph{monotonic}
luminance sequence whereas the colors in a \code{qualitative_hcl} palette all have the
\emph{same} luminance. Finally, \code{diverging_hcl} palettes use the same monotonic
luminance sequence in both \dQuote{arms} of the palette, i.e., correspond to
two balanced sequential palettes diverging from the same neutral value.
The other two axes, hue and chroma, are used to enhance the luminance information and/or
to further discriminate the color.

All three palette functions specify trajectories in HCL space and hence need either
single values or intervals of the coordinates \code{h}, \code{c}, \code{l}. Their
interfaces are always designed such that \code{h}, \code{c}, \code{l} can take vector
arguments (as needed) but alternatively or additionally \code{h1}/\code{h2},
\code{c1}/\code{cmax}/\code{c2}, and \code{l1}/\code{l2} can be specified. If so,
the latter coordinates overwrite the former.

\code{qualitative_hcl} distinguishes the underlying categories by a sequence of
hues while keeping both chroma and luminance constant to give each color in the
resulting palette the same perceptual weight. Thus, \code{h} should be a pair of
hues (or equivalently \code{h1} and \code{h2} can be used) with the starting and
ending hue of the palette. Then, an equidistant sequence between these hues is
employed, by default spanning the full color wheel (i.e, the full 360 degrees).
Chroma \code{c} (or equivalently \code{c1}) and luminance \code{l} (or equivalently
\code{l1}) are constants.

\code{sequential_hcl} codes the underlying numeric values by a monotonic sequence
of increasing (or decreasing) luminance. Thus, the \code{l} argument should provide
a vector of length 2 with starting and ending luminance (equivalently, \code{l1} and
\code{l2} can be used). Without chroma (i.e., \code{c = 0}), this simply corresponds
to a grayscale palette like \code{\link[grDevices]{gray.colors}}. For adding chroma, a simple
strategy would be to pick a single hue (via \code{h} or \code{h1}) and then decrease
chroma from some value (\code{c} or \code{c1}) to zero (i.e., gray) along with
increasing luminance. For bringing out the extremes (a dark high-chroma color vs.
a light gray) this is already very effective. For distinguishing also colors in the
middle two strategies can be employed: (a) Hue can be varied as well by specifying an
interval of hues in \code{h} (or beginning hue \code{h1} and ending hue \code{h2}).
(b) Instead of a decreasing chroma a triangular chroma trajectory can be employed
from \code{c1} over \code{cmax} to \code{c2} (or equivalently a vector \code{c} of
length 3). This yields high-chroma colors in the middle of the palette that are
more easily distinguished from the dark and light extremes. Finally, instead of
employing linear trajectories, power transformations are supported in chroma and
luminance via a vector \code{power} (or separate \code{p1} and \code{p2}). If
\code{power[2]} (or \code{p2}) for the luminance trajectory is missing, it defaults
to \code{power[1]}/\code{p1} from the chroma trajectory.

\code{diverging_hcl} codes the underlying numeric values by a triangular luminance
sequence with different hues in the left and in the right arm of the palette. Thus,
it can be seen as a combination of two sequential palettes with some restrictions:
(a) a single hue is used for each arm of the palette, (b) chroma and luminance trajectory
are balanced between the two arms, (c) the neutral central value has zero chroma.
To specify such a palette a vector of two hues \code{h} (or equivalently \code{h1}
and \code{h2}), either a single chroma value \code{c} (or \code{c1}) or a vector
of two chroma values \code{c} (or \code{c1} and \code{cmax}), a vector of two
luminances \code{l} (or \code{l1} and \code{l2}), and power parameter(s) \code{power}
(or \code{p1} and \code{p2}) are used. For more flexible diverging palettes without
the restrictrictions above (and consequently more parameters)
\code{\link{divergingx_hcl}} is available. For backward compatibility, \code{diverge_hcl}
is a copy of \code{diverging_hcl}.

To facilitate using HCL-based palettes a wide range of example palettes are
provided in the package and can be specified by a name instead of a set of
parameters/coordinates. The examples have been taken from the literature and many
approximate color palettes from other software packages such as ColorBrewer.org
(\pkg{RColorBrewer}), CARTO colors (\pkg{rcartocolor}), \pkg{scico}, or \pkg{virids}. The
function \code{hcl_palettes} can be used to query the available pre-specified palettes. It
comes with a \code{print} method (listing names and types), a \code{summary} method
(additionally listing the underlying parameters/coordinates), and a \code{plot}
method that creates a \code{\link{swatchplot}} with suitable labels.

The named HCL-based palettes have also been included in the \code{\link[grDevices]{hcl.colors}}
function in base R in order to facilitate adoption in practice and in R packages.
However, in \code{hcl.colors} it is only possible to obtain \code{n} colors from the
given palette without the possibility for further modifications.
}
\examples{
## overview of all _named_ HCL palettes
hcl_palettes()

## visualize
hcl_palettes("qualitative", plot = TRUE)
hcl_palettes("sequential (single-hue)", n = 7, plot = TRUE)
hcl_palettes("sequential (multi-hue)", n = 7, plot = TRUE)
hcl_palettes("diverging", n = 7, plot = TRUE)

## inspect a specific palette
## (upper-case, spaces, etc. are ignored for matching)
hcl_palettes(palette = "Dark 2")
hcl_palettes(palette = "dark2")

## set up actual colors
qualitative_hcl(4, h = c(0, 288), c = 50, l = 60) ## by hand
qualitative_hcl(4, palette = "dark2")             ## by name
qualitative_hcl(4, palette = "dark2", c = 80)     ## by name plus modification

## how HCL palettes are constructed:
## by varying the perceptual properties via hue/chroma/luminance
swatchplot(
  "Hue"       = sequential_hcl(5, h = c(0, 300), c = c(60, 60), l = 65),
  "Chroma"    = sequential_hcl(5, h = 0, c = c(100, 0), l = 65, rev = TRUE, power = 1),
  "Luminance" = sequential_hcl(5, h = 260, c = c(25, 25), l = c(25, 90), rev = TRUE, power = 1),
  off = 0
)

## for qualitative palettes luminance and chroma are fixed, varying only hue
hclplot(qualitative_hcl(9, c = 50, l = 70))

## single-hue sequential palette (h = 260) with linear vs. power-transformed trajectory
hclplot(sequential_hcl(7, h = 260, c = 80, l = c(35, 95), power = 1))
hclplot(sequential_hcl(7, h = 260, c = 80, l = c(35, 95), power = 1.5))

## advanced single-hue sequential palette with triangular chroma trajectory
## (piecewise linear vs. power-transformed)
hclplot(sequential_hcl(7, h = 245, c = c(40, 75, 0), l = c(30, 95), power = 1))
hclplot(sequential_hcl(7, h = 245, c = c(40, 75, 0), l = c(30, 95), power = c(0.8, 1.4)))

## multi-hue sequential palette with small hue range and triangular chroma vs.
## large hue range and linear chroma trajectory
hclplot(sequential_hcl(7, h = c(260, 220), c = c(50, 75, 0), l = c(30, 95), power = 1))
hclplot(sequential_hcl(7, h = c(260, 60), c = 60, l = c(40, 95), power = 1))

## balanced diverging palette constructed from two simple single-hue sequential
## palettes (for hues 260/blue and 0/red)
hclplot(diverging_hcl(7, h = c(260, 0), c = 80, l = c(35, 95), power = 1))

## to register a particular adapted palette for re-use in the same session
## with a new name the register=... argument can be used once, e.g.,
diverging_hcl(7, palette = "Tropic", h2 = 0, register = "mytropic")

## subsequently palete="mytropic" is available in diverging_hcl() and the diverging
## ggplot2 scales such as scale_color_continuous_diverging() etc.
demoplot(diverging_hcl(11, "mytropic"), type = "map")

## to register this palette in all R sessions you could place the following
## code in a startup script (e.g., .Rprofile):
## colorspace::diverging_hcl(7, palette = "Tropic", h2 = 0, register = "mytropic")

}
\references{
Zeileis A, Hornik K, Murrell P (2009).  Escaping RGBland:
Selecting Colors for Statistical Graphics.  \emph{Computational Statistics &
Data Analysis}, \bold{53}, 3259--3270.
\doi{10.1016/j.csda.2008.11.033}
Preprint available from
\url{https://www.zeileis.org/papers/Zeileis+Hornik+Murrell-2009.pdf}.

Stauffer R, Mayr GJ, Dabernig M, Zeileis A (2015).  Somewhere Over the
Rainbow: How to Make Effective Use of Colors in Meteorological
Visualizations.  \emph{Bulletin of the American Meteorological Society},
\bold{96}(2), 203--216.
\doi{10.1175/BAMS-D-13-00155.1}

Zeileis A, Fisher JC, Hornik K, Ihaka R, McWhite CD, Murrell P, Stauffer R, Wilke CO (2020).
\dQuote{colorspace: A Toolbox for Manipulating and Assessing Colors and Palettes.}
\emph{Journal of Statistical Software}, \bold{96}(1), 1--49. \doi{10.18637/jss.v096.i01}

Zeileis A, Murrell P (2023).
\dQuote{Coloring in R's Blind Spot.}
\emph{The R Journal}, \bold{15}(3), 240--256. \doi{10.32614/RJ-2023-071}
}
\seealso{
\code{\link{divergingx_hcl}} \code{\link[grDevices]{hcl.colors}}
}
\keyword{color}
